package ru.yandex.stockpile.server.shard.cache;

import java.time.Instant;
import java.util.concurrent.TimeUnit;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.server.shard.merge.ArchiveItemIterator;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.model.point.AggrPoints.point;
import static ru.yandex.stockpile.client.TestUtil.timeToMillis;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricDataCacheEntryTest {
    private long now;
    private MetricDataCacheEntry entry;
    private int shardId;
    private long localId;

    @Before
    public void setUp() {
        now = System.currentTimeMillis();
        entry = new MetricDataCacheEntry();
        shardId = StockpileShardId.random();
        localId = StockpileLocalId.random();
    }

    @Test
    public void readCompleteOldMetric() {
        AggrGraphDataArrayList read = AggrGraphDataArrayList.of(
                point("2017-09-07T11:00:00Z", 1),
                point("2017-09-07T11:15:00Z", -3),
                point("2017-09-07T11:30:00Z", 5),
                point("2017-09-07T11:45:00Z", 91.2),
                point("2017-09-07T12:00:00Z", 23.1));

        entry.updateAfterReadCompleted(toEntry(read));
        assertThat(getCached(now), equalTo(AggrGraphDataArrayList.empty()));
        assertThat(getCached(now - TimeUnit.HOURS.toMillis(1)), equalTo(AggrGraphDataArrayList.empty()));
        assertThat(getCached(0), equalTo(read));
        assertThat(getCached(Instant.parse("2017-09-07T11:00:00Z").toEpochMilli()), equalTo(read));
    }

    @Test
    public void cacheUpdatedByWriteBeforeCompleteRead() {
        AggrGraphDataArrayList read = AggrGraphDataArrayList.of(
                point("2017-09-07T11:00:00Z", 1),
                point("2017-09-07T11:15:00Z", 2),
                point("2017-09-07T11:30:00Z", 3),
                point(now - TimeUnit.MINUTES.toMillis(5), 4),
                point(now - TimeUnit.MINUTES.toMillis(4), 5)
        );

        AggrGraphDataArrayList fresh = AggrGraphDataArrayList.of(
                point(now - TimeUnit.MINUTES.toMillis(3), 6),
                point(now - TimeUnit.MINUTES.toMillis(2), 7),
                point(now - TimeUnit.MINUTES.toMillis(1), 8));

        // writes occurs before read complete
        entry.updateWithOrWriteCompleted(shardId, localId, MetricArchiveMutable.of(fresh));
        entry.updateAfterReadCompleted(toEntry(read));

        AggrGraphDataArrayList expectedFull = new AggrGraphDataArrayList();
        expectedFull.addAll(read);
        expectedFull.addAll(fresh);

        AggrGraphDataArrayList expectedFresh = new AggrGraphDataArrayList();
        expectedFresh.addRecord(point(now - TimeUnit.MINUTES.toMillis(5), 4));
        expectedFresh.addRecord(point(now - TimeUnit.MINUTES.toMillis(4), 5));
        expectedFresh.addAll(fresh);

        assertThat(getCached(now - TimeUnit.MINUTES.toMillis(5)), equalTo(expectedFresh));
        assertThat(getCached(now - TimeUnit.HOURS.toMillis(1)), equalTo(expectedFresh));
        assertThat(getCached(0), equalTo(expectedFull));
        assertThat(getCached(Instant.parse("2017-09-07T11:00:00Z").toEpochMilli()), equalTo(expectedFull));
    }

    @Test
    public void cacheUpdatedByWriteAfterCompleteRead() {
        AggrGraphDataArrayList read = AggrGraphDataArrayList.of(
                point("2017-09-07T11:00:00Z", 1),
                point("2017-09-07T11:15:00Z", 2),
                point("2017-09-07T11:30:00Z", 3),
                point(now - TimeUnit.MINUTES.toMillis(5), 4),
                point(now - TimeUnit.MINUTES.toMillis(4), 5)
        );

        AggrGraphDataArrayList fresh = AggrGraphDataArrayList.of(
                point(now - TimeUnit.MINUTES.toMillis(3), 6),
                point(now - TimeUnit.MINUTES.toMillis(2), 7),
                point(now - TimeUnit.MINUTES.toMillis(1), 8));

        entry.updateAfterReadCompleted(toEntry(read));
        // writes occurs after read complete
        entry.updateWithOrWriteCompleted(shardId, localId, MetricArchiveMutable.of(fresh));

        AggrGraphDataArrayList expectedFull = new AggrGraphDataArrayList();
        expectedFull.addAll(read);
        expectedFull.addAll(fresh);

        AggrGraphDataArrayList expectedFresh = new AggrGraphDataArrayList();
        expectedFresh.addRecord(point(now - TimeUnit.MINUTES.toMillis(5), 4));
        expectedFresh.addRecord(point(now - TimeUnit.MINUTES.toMillis(4), 5));
        expectedFresh.addAll(fresh);

        assertThat(getCached(now - TimeUnit.MINUTES.toMillis(5)), equalTo(expectedFresh));
        assertThat(getCached(now - TimeUnit.HOURS.toMillis(1)), equalTo(expectedFresh));
        assertThat(getCached(0), equalTo(expectedFull));
        assertThat(getCached(Instant.parse("2017-09-07T11:00:00Z").toEpochMilli()), equalTo(expectedFull));
    }

    @Test
    public void completeReadDeletedMetric() {
        AggrGraphDataArrayList read = AggrGraphDataArrayList.of(
                point("2017-09-07T11:00:00Z", 1),
                point("2017-09-07T11:15:00Z", -3),
                point("2017-09-07T11:30:00Z", 5),
                point("2017-09-07T11:45:00Z", 91.2),
                point("2017-09-07T12:00:00Z", 23.1));

        MetricArchiveMutable delete = new MetricArchiveMutable();
        delete.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
        entry.updateWithOrWriteCompleted(shardId, localId, delete);
        entry.updateAfterReadCompleted(toEntry(read));

        assertThat(getCached(now), equalTo(AggrGraphDataArrayList.empty()));
        assertThat(getCached(now - TimeUnit.HOURS.toMillis(1)), equalTo(AggrGraphDataArrayList.empty()));
        assertThat(getCached(0), equalTo(AggrGraphDataArrayList.empty()));
        assertThat(getCached(Instant.parse("2017-09-07T11:30:00Z").toEpochMilli()), equalTo(AggrGraphDataArrayList.empty()));
    }

    @Test
    public void deleteFromCache() {
        AggrGraphDataArrayList read = AggrGraphDataArrayList.of(
                point("2017-09-07T11:00:00Z", 1),
                point("2017-09-07T11:15:00Z", 2),
                point("2017-09-07T11:30:00Z", 3));
        entry.updateAfterReadCompleted(toEntry(read));

        MetricArchiveMutable delete = new MetricArchiveMutable();
        delete.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
        entry.updateWithOrWriteCompleted(shardId, localId, delete);

        assertThat(getCached(now), equalTo(AggrGraphDataArrayList.empty()));
        assertThat(getCached(now - TimeUnit.HOURS.toMillis(1)), equalTo(AggrGraphDataArrayList.empty()));
        assertThat(getCached(0), equalTo(AggrGraphDataArrayList.empty()));
        assertThat(getCached(Instant.parse("2017-09-07T11:15:00Z").toEpochMilli()), equalTo(AggrGraphDataArrayList.empty()));
    }

    @Test
    public void refreshFreshTime() {
        MetricDataCacheEntry cached = new MetricDataCacheEntry();
        cached.updateAfterReadCompleted(toEntry(AggrGraphDataArrayList.of(point(now + TimeUnit.SECONDS.toMillis(1), 9))));

        MetricDataCacheEntry fresh = new MetricDataCacheEntry();
        fresh.updateAfterReadCompleted(toEntry(AggrGraphDataArrayList.of(
                point(now - TimeUnit.SECONDS.toMillis(40), 1),
                point(now - TimeUnit.SECONDS.toMillis(30), 2),
                point(now - TimeUnit.SECONDS.toMillis(25), 3),
                point(now - TimeUnit.SECONDS.toMillis(20), 4),
                point(now - TimeUnit.SECONDS.toMillis(15), 5),
                point(now - TimeUnit.SECONDS.toMillis(10), 6),
                point(now - TimeUnit.SECONDS.toMillis(5), 7),
                point(now, 8))));

        cached.updateAfterReadCompleted(fresh);

        AggrGraphDataArrayList result =
            AggrGraphDataArrayList.of(cached.snapshot(now - TimeUnit.SECONDS.toMillis(29), Long.MAX_VALUE));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
                point(now - TimeUnit.SECONDS.toMillis(25), 3),
                point(now - TimeUnit.SECONDS.toMillis(20), 4),
                point(now - TimeUnit.SECONDS.toMillis(15), 5),
                point(now - TimeUnit.SECONDS.toMillis(10), 6),
                point(now - TimeUnit.SECONDS.toMillis(5), 7),
                point(now, 8),
                point(now + TimeUnit.SECONDS.toMillis(1), 9)
        );

        assertThat(result, equalTo(expected));
    }

    @Test
    public void saveHeaders() {
        AggrGraphDataArrayList source = AggrGraphDataArrayList.of(
                point("2017-09-07T11:00:00Z", 1),
                point("2017-09-07T11:15:00Z", 2),
                point("2017-09-07T11:30:00Z", 3));

        MetricHeader header = new MetricHeader(100, 2, 42, (short) 0, MetricType.DGAUGE);
        MetricDataCacheEntry one = new MetricDataCacheEntry(0, header, 0, source.iterator());
        assertEquals(header, one.snapshot(now - TimeUnit.SECONDS.toMillis(29), Long.MAX_VALUE).header());
        assertEquals(header, one.snapshot(Instant.parse("2017-09-07T11:15:00Z").toEpochMilli(), Long.MAX_VALUE).header());

        entry.updateAfterReadCompleted(one);
        assertEquals(header, entry.snapshot(now - TimeUnit.SECONDS.toMillis(29), Long.MAX_VALUE).header());
        assertEquals(header, entry.snapshot(Instant.parse("2017-09-07T11:15:00Z").toEpochMilli(), Long.MAX_VALUE).header());
    }

    @Test
    public void saveHeadersDuringMerge() {
        AggrGraphDataArrayList source = AggrGraphDataArrayList.of(
                point("2017-09-07T11:00:00Z", 1),
                point("2017-09-07T11:15:00Z", 2),
                point("2017-09-07T11:30:00Z", 3));

        MetricHeader header = new MetricHeader(100, 2, 42, (short) 0, MetricType.DGAUGE);
        MetricDataCacheEntry one = new MetricDataCacheEntry(0, header, 0, source.iterator());
        assertEquals(header, one.snapshot(now - TimeUnit.SECONDS.toMillis(29), Long.MAX_VALUE).header());
        assertEquals(header, one.snapshot(Instant.parse("2017-09-07T11:15:00Z").toEpochMilli(), Long.MAX_VALUE).header());

        // write occurs before complete read
        MetricArchiveMutable delta = new MetricArchiveMutable();
        delta.setType(MetricType.DGAUGE);
        delta.addRecord(randomPoint(MetricType.DGAUGE));
        entry.updateWithOrWriteCompleted(shardId, localId, delta);

        entry.updateAfterReadCompleted(one);
        assertEquals(header, entry.snapshot(now - TimeUnit.SECONDS.toMillis(29), Long.MAX_VALUE).header());
        assertEquals(header, entry.snapshot(Instant.parse("2017-09-07T11:15:00Z").toEpochMilli(), Long.MAX_VALUE).header());
    }

    @Test
    public void isNotLoaded() {
        MetricDataCacheEntry cached = new MetricDataCacheEntry();
        long ts0 = System.currentTimeMillis() + 1_000;

        assertFalse(cached.isLoaded(ts0));
        assertFalse(cached.isLoaded(ts0 - 10_000));

        cached.updateWithOrWriteCompleted(42, 42, MetricArchiveMutable.of(AggrGraphDataArrayList.of(point(ts0 + 1_000, 1))));

        assertTrue(cached.isLoaded(ts0));
        assertFalse(cached.isLoaded(ts0 - 10_000));

        cached.updateWithOrWriteCompleted(42, 42, MetricArchiveMutable.of(AggrGraphDataArrayList.of(point(ts0 + 2_000, 1))));
        assertTrue(cached.isLoaded(ts0));
        assertFalse(cached.isLoaded(ts0 - 10_000));

        cached.updateAfterReadCompleted(toEntry(AggrGraphDataArrayList.of(point(ts0 - 10_000, 42))));
        assertTrue(cached.isLoaded(ts0));
        assertTrue(cached.isLoaded(ts0 - 10_000));
        assertTrue(cached.isLoaded(ts0 - 20_000));
        assertTrue(cached.isLoaded(0));
    }

    @Test
    public void decimateOnRead() {
        var archive = new MetricArchiveMutable();
        archive.setType(MetricType.DGAUGE);
        archive.setDecimPolicyId(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber());
        archive.addAll(AggrGraphDataArrayList.of(
                point("2020-04-07T00:03:00Z", 1),
                point("2020-04-07T00:08:00Z", 2),
                point("2020-04-07T00:11:00Z", 3)));

        entry.updateWithOrWriteCompleted(shardId, localId, archive);

        assertEquals(AggrGraphDataArrayList.empty(), getCached(now));
        assertEquals(AggrGraphDataArrayList.empty(), getCached(now - TimeUnit.HOURS.toMillis(3)));

        {
            var expected = AggrGraphDataArrayList.of(
                    point("2020-04-07T00:00:00Z", 1),
                    point("2020-04-07T00:05:00Z", 2),
                    point("2020-04-07T00:10:00Z", 3));
            assertEquals(expected, getCached(0));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                    point("2020-04-07T00:05:00Z", 2),
                    point("2020-04-07T00:10:00Z", 3));
            assertEquals(expected, getCached(timeToMillis("2020-04-07T00:05:00Z")));
        }
    }

    @Test
    public void decimateCompleteRead() {
        var archive = new MetricArchiveMutable();
        archive.setType(MetricType.DGAUGE);
        archive.setDecimPolicyId(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.getNumber());
        archive.addAll(AggrGraphDataArrayList.of(
                point("2020-04-07T00:03:00Z", 1),
                point("2020-04-07T00:08:00Z", 2),
                point("2020-04-07T00:11:00Z", 3)));

        var read = MetricDataCacheEntry.of(0, archive.header(), ArchiveItemIterator.of(archive));
        entry.updateAfterReadCompleted(read);
        {
            var expected = AggrGraphDataArrayList.of(
                    point("2020-04-07T00:00:00Z", 1),
                    point("2020-04-07T00:05:00Z", 2),
                    point("2020-04-07T00:10:00Z", 3));
            assertEquals(expected, getCached(0));
        }
        {
            var expected = AggrGraphDataArrayList.of(
                    point("2020-04-07T00:05:00Z", 2),
                    point("2020-04-07T00:10:00Z", 3));
            assertEquals(expected, getCached(timeToMillis("2020-04-07T00:05:00Z")));
        }
    }

    private AggrGraphDataArrayList getCached(long fromTime) {
        return AggrGraphDataArrayList.of(entry.snapshot(fromTime, Long.MAX_VALUE));
    }

    private MetricDataCacheEntry toEntry(AggrGraphDataIterable source) {
        return new MetricDataCacheEntry(0, MetricHeader.defaultValue, 0, source.iterator());
    }
}
