package ru.yandex.stockpile.server.shard.cache;

import java.util.concurrent.ThreadLocalRandom;

import org.junit.Before;
import org.junit.Test;
import org.openjdk.jol.info.GraphLayout;

import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContent;
import ru.yandex.stockpile.tool.BitBufAssume;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.hamcrest.Matchers.lessThan;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMask;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricDataCacheTest {

    private MetricDataCache cache;

    private static AggrGraphDataArrayList randomPointsList(int count) {
        int mask = randomMask(MetricType.DGAUGE);
        AggrGraphDataArrayList list = new AggrGraphDataArrayList(mask, count);
        for (int index = 0; index < count; index++) {
            list.addRecord(randomPoint(mask));
        }
        list.sortAndMerge();
        return list;
    }

    private static MetricDataCacheEntry randomCacheEntry() {
        return new MetricDataCacheEntry(0, MetricHeader.defaultValue, 0, randomPointsList(50).iterator());
    }

    @Before
    public void setUp() {
        cache = new MetricDataCache(StockpileShardId.random());
        cache.setMaxSizeBytes(1 << 20); // 1 MiB
    }

    @Test
    public void initialSize() {
        assertEquals(0L, cache.getBytesUsed());
    }

    @Test
    public void getArchiveCreateEntryToWarm() {
        final long localId = StockpileLocalId.random();

        long initSize = cache.getBytesUsed();
        assertNull(cache.getSnapshot(localId));
        assertThat(cache.getRecordsUsed(), equalTo(1L));
        assertThat(cache.getBytesUsed(), greaterThan(initSize));

        cache.removeEntry(localId);
        assertEquals(initSize, cache.getBytesUsed());
    }

    @Test
    public void updateSizeByCompleteRead() {
        final long localId = StockpileLocalId.random();
        long initSize = cache.getBytesUsed();
        assertNull(cache.getSnapshot(localId));

        long beforeReadSize = cache.getBytesUsed();
        cache.readCompleted(localId, randomCacheEntry());
        assertThat(cache.getBytesUsed(), greaterThan(beforeReadSize));

        cache.removeEntry(localId);
        assertThat(cache.getBytesUsed(), equalTo(initSize));
    }

    @Test
    public void removeReadArchiveUpdateSize() {
        final long localId = StockpileLocalId.random();
        long init = cache.getBytesUsed();
        assertNull(cache.getSnapshot(localId));
        cache.readCompleted(localId, randomCacheEntry());
        long appended = cache.getBytesUsed();
        cache.removeEntry(localId);
        long after = cache.getBytesUsed();

        assertThat(after, lessThan(appended));
        assertThat(after, equalTo(init));
        assertThat(cache.getRecordsUsed(), equalTo(0L));
    }

    @Test
    public void appendToAlreadyReadAffectSize() {
        final long localId = StockpileLocalId.random();
        final long initSize = cache.getBytesUsed();

        assertNull(cache.getSnapshot(localId));
        cache.readCompleted(localId, randomCacheEntry());
        assertThat(cache.getBytesUsed(), greaterThan(initSize));
        appendByOnePoint(localId);
        cache.removeEntry(localId);

        assertThat(cache.getBytesUsed(), equalTo(initSize));
    }

    @Test
    public void appendToReadInProgressAffectSize() {
        final long localId = StockpileLocalId.random();
        final long initSize = cache.getBytesUsed();

        assertNull(cache.getSnapshot(localId));
        // read not completed yet but already receive writes
        appendByOnePoint(localId);

        cache.removeEntry(localId);
        assertThat(cache.getBytesUsed(), equalTo(initSize));
    }

    @Test
    public void asOnlyCacheLimitReachedArchiveRemoved() {
        final long localId = StockpileLocalId.random();
        final long initSize = cache.getBytesUsed();

        assertNull(cache.getSnapshot(localId));
        cache.readCompleted(localId, randomCacheEntry());

        while (cache.getRecordsUsed() > 0) {
            StockpileLogEntryContent logEntry = new StockpileLogEntryContent();
            logEntry.addMetric(localId, randomPointsList(1000));
            cache.updateWithOnWriteCompleted(logEntry);
        }

        assertThat(cache.getBytesUsed(), equalTo(initSize));
        assertNull(cache.getSnapshot(localId));
    }

    @Test
    public void objectSizeEmpty() {
        BitBufAssume.assumeUsedClass(HeapBitBuf.class);
        assertEquals(expectedLayout(cache), cache.memorySizeIncludingSelf());
    }

    @Test
    public void objectSize() {
        BitBufAssume.assumeUsedClass(HeapBitBuf.class);
        final long localId = StockpileLocalId.random();
        assertNull(cache.getSnapshot(localId));
        cache.readCompleted(localId, randomCacheEntry());

        for (int index = 0; index < 50; index++) {
            StockpileLogEntryContent logEntry = new StockpileLogEntryContent();
            logEntry.addMetric(index % 20, randomPointsList(1000));
            cache.updateWithOnWriteCompleted(logEntry);
        }
        assertEquals(expectedLayout(cache), cache.memorySizeIncludingSelf());
    }

    private long expectedLayout(MetricDataCache source) {
        GraphLayout gl = GraphLayout.parseInstance(MetricType.DGAUGE, StockpileFormat.CURRENT, source);
        long enums = GraphLayout.parseInstance(MetricType.DGAUGE, StockpileFormat.CURRENT).totalSize();
        return gl.totalSize() - enums;
    }

    private void appendByOnePoint(long localId) {
        int mask = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask();
        long now = System.currentTimeMillis();
        long prevSize = cache.getBytesUsed();
        for (int index = 0; index < 100; index++) {
            now += ThreadLocalRandom.current().nextLong(15_000, 380_000);
            AggrPoint point = randomPoint(mask);
            point.tsMillis = now;

            StockpileLogEntryContent logEntry = new StockpileLogEntryContent();
            logEntry.addAggrPoint(localId, point);
            cache.updateWithOnWriteCompleted(logEntry);
        }
        long used = cache.getBytesUsed();
        assertThat(used, greaterThanOrEqualTo(prevSize));
    }
}
