package ru.yandex.stockpile.server.shard.iter;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.ChunkWithNo;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.shard.load.Async;

import static org.hamcrest.CoreMatchers.containsString;
import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class KvSnapshotChunkIteratorTest {
    private KikimrKvClientInMem kvClient = new KikimrKvClientInMem();
    private long tabletId;
    private long tabletGen;
    private ExecutorService executor;
    private SnapshotAddress address = new SnapshotAddress(SnapshotLevel.ETERNITY, 42);
    private ThreadLocalRandom random;

    private static String fileName(SnapshotAddress address, int chunkNo) {
        return StockpileKvNames.chunkFileName(
            address.level(),
            address.txn(),
            chunkNo,
            FileNamePrefix.Current.instance);
    }

    @Before
    public void before() {
        tabletId = kvClient.createKvTablet();
        tabletGen = kvClient.incrementGeneration(tabletId, 0).join();
        executor = ForkJoinPool.commonPool();
        random = ThreadLocalRandom.current();
    }

    @Test
    public void noChunks() {
        var it = new TestIterator(0);
        assertNull(it.next().join());
        assertNull(it.next().join());
    }

    @Test
    public void oneChunk() {
        var bytes = randomBytes();
        writeSync(address, 0, bytes);

        var it = new TestIterator(1);
        ChunkWithNo chunk = it.next().join();
        assertEquals(0, chunk.getNo());
        assertArrayEquals(bytes, chunk.getContent());

        assertNull(it.next().join());
    }

    @Test
    public void fewChunk() {
        var one = randomBytes();
        writeSync(address, 0, one);
        var two = randomBytes();
        writeSync(address, 1, two);
        var tree = randomBytes();
        writeSync(address, 2, tree);

        var it = new TestIterator(3);
        ChunkWithNo chunkOne = it.next().join();
        assertEquals(0, chunkOne.getNo());
        assertArrayEquals(one, chunkOne.getContent());

        ChunkWithNo chunkTwo = it.next().join();
        assertEquals(1, chunkTwo.getNo());
        assertArrayEquals(two, chunkTwo.getContent());

        ChunkWithNo chunkTree = it.next().join();
        assertEquals(2, chunkTree.getNo());
        assertArrayEquals(tree, chunkTree.getContent());

        assertNull(it.next().join());
    }

    @Test
    public void manyChunks() {
        List<ChunkWithNo> expected = IntStream.range(0, random.nextInt(100, 1000))
            .parallel()
            .mapToObj(chunkNo -> {
                var bytes = randomBytes();
                var chunk = new ChunkWithNo(chunkNo, bytes);
                return write(address, chunkNo, bytes)
                    .thenApply(ignore -> chunk);
            })
            .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOf))
            .join();

        var index = new AtomicInteger();
        var it = new TestIterator(expected.size());
        Async.forEach(it, chunk -> {
            var expect = expected.get(index.getAndIncrement());
            assertEquals(expect.getNo(), chunk.getNo());
            assertArrayEquals(expect.getContent(), chunk.getContent());
        }).join();

        assertNull(it.next().join());
        assertEquals(expected.size(), index.get());
    }

    @Test
    public void slowLoad() {
        List<ChunkWithNo> expected = IntStream.range(0, 50)
            .parallel()
            .mapToObj(chunkNo -> {
                var bytes = randomBytes();
                var chunk = new ChunkWithNo(chunkNo, bytes);
                return write(address, chunkNo, bytes)
                    .thenApply(ignore -> chunk);
            })
            .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOf))
            .join();

        var index = new AtomicInteger();
        var it = new TestIterator(expected.size());

        Async.forEach(it, chunk -> {
            kvClient.pauseReads();
            var expect = expected.get(index.getAndIncrement());
            assertEquals(expect.getNo(), chunk.getNo());
            assertArrayEquals(expect.getContent(), chunk.getContent());

            CompletableFuture.delayedExecutor(random.nextInt(1, 10), TimeUnit.MILLISECONDS, executor)
                .execute(() -> kvClient.resumeReads());
        }).join();

        assertNull(it.next().join());
        assertEquals(expected.size(), index.get());
    }

    @Test
    public void failedRead() {
        List<ChunkWithNo> expected = IntStream.range(0, 50)
            .parallel()
            .mapToObj(chunkNo -> {
                var bytes = randomBytes();
                var chunk = new ChunkWithNo(chunkNo, bytes);
                return write(address, chunkNo, bytes)
                    .thenApply(ignore -> chunk);
            })
            .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOf))
            .join();

        var index = new AtomicInteger();
        var it = new TestIterator(expected.size());

        var future = Async.forEach(it, chunk -> {
            var expect = expected.get(index.getAndIncrement());
            assertEquals(expect.getNo(), chunk.getNo());

            kvClient.throwOnRead();
        });

        var error = future
            .thenApply(ignore -> (Throwable) null)
            .exceptionally(e -> e)
            .join();

        assertNotNull(error);
        assertThat(error.getMessage(), containsString("readMode == THROW"));
    }

    private byte[] randomBytes() {
        int size = random.nextInt(1, 1 << 10);
        byte[] result = new byte[size];
        random.nextBytes(result);
        return result;
    }

    private CompletableFuture<?> write(SnapshotAddress address, int chunkNo, byte[] bytes) {
        return kvClient.write(
            tabletId,
            tabletGen,
            fileName(address, chunkNo),
            bytes,
            MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN,
            MsgbusKv.TKeyValueRequest.EPriority.REALTIME,
            0);
    }

    private void writeSync(SnapshotAddress address, int chunkNo, byte[] bytes) {
        write(address, chunkNo, bytes).join();
    }

    private class TestIterator extends KvSnapshotChunkIterator {
        TestIterator(int chunkCount) {
            super(address, chunkCount);
        }

        @Override
        protected CompletableFuture<ChunkWithNo> readNext(SnapshotAddress snapshotAddress, int chunkNo) {
            Supplier<CompletableFuture<ChunkWithNo>> supplier =
                () -> kvClient.readDataLarge(tabletId, tabletGen, fileName(snapshotAddress, chunkNo), 0, MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND)
                    .thenApply(bytes -> new ChunkWithNo(chunkNo, bytes));

            return CompletableFutures.supplyAsync(supplier, executor)
                .thenCompose(future -> future);
        }
    }
}
