package ru.yandex.stockpile.server.shard.iter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nullable;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.shard.load.Async;
import ru.yandex.stockpile.server.shard.load.AsyncIterator;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricsIteratorTest {

    private ExecutorService executor;
    private ThreadLocalRandom random;

    @Before
    public void setUp() {
        executor = ForkJoinPool.commonPool();
        random = ThreadLocalRandom.current();
    }

    @Test
    public void empty() {
        var it = iterator(List.of(List.of(), List.of(), List.of()));
        assertNull(it.next().join());
    }

    @Test
    public void one() {
        for (int index = 1; index < 10; index++) {
            var source = List.of(generateData(index));
            var it = iterator(source);
            assertRead(expect(source), it);
        }
    }

    @Test
    public void oneAndEmpty() {
        {
            var source = List.<List<MetricIdAndData>>of(generateData(10), List.of());
            var it = iterator(source);
            assertRead(expect(source), it);
        }
        {
            var source = List.<List<MetricIdAndData>>of(List.of(), generateData(10));
            var it = iterator(source);
            assertRead(expect(source), it);
        }
        {
            var source = List.<List<MetricIdAndData>>of(List.of(), generateData(10), List.of());
            var it = iterator(source);
            assertRead(expect(source), it);
        }
    }

    @Test
    public void differentSize() {
        {
            var source = List.of(generateData(5), generateData(2), generateData(1));
            var it = iterator(source);
            assertRead(expect(source), it);
        }
        {
            var source = List.of(generateData(1), generateData(2), generateData(5));
            var it = iterator(source);
            assertRead(expect(source), it);
        }
        {
            var source = List.of(generateData(2), generateData(5), generateData(1));
            var it = iterator(source);
            assertRead(expect(source), it);
        }
    }

    @Test
    public void sameIdsGrouped() {
        var source = new ArrayList<List<MetricIdAndData>>();
        var orig = generateData(10);

        for (int index = 0; index < random.nextLong(2, 10); index++) {
            var partlySame = orig.stream()
                .map(data -> overrideLocalId(data.localId(), generateData()))
                .filter(data -> random.nextBoolean())
                .collect(Collectors.toList());
            partlySame.addAll(generateData(3));
            partlySame.sort((o1, o2) -> StockpileLocalId.compare(o1.localId(), o2.localId()));
            source.add(partlySame);
        }

        var it = iterator(source);
        assertRead(expect(source), it);
    }

    @Test
    public void localIdsOrdered() {
        var source = IntStream.range(0, random.nextInt(2, 10))
            .mapToObj(idx -> generateData(random.nextInt(0, 1000)))
            .collect(Collectors.toList());

        var it = iterator(source);
        assertRead(expect(source), it);
    }

    private List<List<MetricIdAndData>> expect(List<List<MetricIdAndData>> expected) {
        return expected.stream()
            .flatMap(Collection::stream)
            .collect(Collectors.groupingBy(MetricIdAndData::localId))
            .entrySet()
            .stream()
            .sorted((o1, o2) -> StockpileLocalId.compare(o1.getKey(), o2.getKey()))
            .map(Map.Entry::getValue)
            .collect(Collectors.toList());
    }

    private void assertRead(List<List<MetricIdAndData>> expected, MetricsIterator it) {
        System.out.println("expected order: " + expected.stream()
            .filter(data -> !data.isEmpty())
            .map(data -> StockpileLocalId.toString(data.get(0).localId()))
            .collect(Collectors.toList()));

        AtomicInteger index = new AtomicInteger();
        Async.forEach(it, result -> {
            var expect = expected.get(index.getAndIncrement());
            assertEquals(expect.size(), result.size());
            for (int i = 0; i < expect.size(); i++) {
                var e = expect.get(i);
                var a = result.get(i);
                assertEquals(e.localId(), a.localId());
                assertEquals(e.lastTsMillis(), a.lastTsMillis());
                assertEquals(e.archive(), a.archive());
                close(e.archive(), a.archive());
            }
        }).join();

        assertNull(it.next().join());
        assertEquals(expected.size(), index.get());
    }

    private MetricsIterator iterator(List<List<MetricIdAndData>> source) {
        return new MetricsIterator(source.stream()
            .map(TestIterator::new)
            .toArray(TestIterator[]::new));
    }

    private MetricIdAndData generateData() {
        long localId = StockpileLocalId.random(random);

        var point = RecyclableAggrPoint.newInstance();
        int mask = TsColumn.mask | ValueColumn.mask;
        MetricArchiveMutable archive = new MetricArchiveMutable();
        archive.setType(MetricType.DGAUGE);
        archive.addRecord(randomPoint(point, mask, random));
        point.recycle();

        return new MetricIdAndData(localId, archive.getLastTsMillis(), 0, archive.toImmutableNoCopy());
    }

    private MetricIdAndData overrideLocalId(long localId, MetricIdAndData data) {
        return new MetricIdAndData(
            localId,
            data.lastTsMillis(),
            data.decimatedAt(),
            data.archive());

    }

    private List<MetricIdAndData> generateData(int size) {
        return IntStream.range(0, size)
            .mapToObj(value -> generateData())
            .sorted((o1, o2) -> StockpileLocalId.compare(o1.localId(), o2.localId()))
            .collect(Collectors.toList());
    }

    private class TestIterator implements AsyncIterator<MetricIdAndData> {
        private AtomicInteger idx = new AtomicInteger();
        private AtomicBoolean inFlight = new AtomicBoolean();
        private List<MetricIdAndData> source;

        public TestIterator(List<MetricIdAndData> source) {
            this.source = source;
        }

        @Override
        public CompletableFuture<MetricIdAndData> next() {
            if (!inFlight.compareAndSet(false, true)) {
                return failedFuture(new IllegalStateException("previous fetch not completed yet"));
            }

            var i = idx.getAndIncrement();
            if (random.nextBoolean()) {
                inFlight.compareAndSet(true, false);
                return completedFuture(getByIndex(i));
            }

            return CompletableFutures.supplyAsync(() -> getByIndex(i), executor)
                .whenComplete((ignore, ignore2) -> inFlight.set(false));
        }

        @Nullable
        private MetricIdAndData getByIndex(int index) {
            if (index >= source.size()) {
                return null;
            }

            return source.get(index);
        }
    }
}
