package ru.yandex.stockpile.server.shard.iter;

import java.util.stream.IntStream;

import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClientSync;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.ChunkWithNo;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.names.file.ChunkFile;
import ru.yandex.stockpile.server.shard.ExceptionHandler;
import ru.yandex.stockpile.server.shard.ShardThread;
import ru.yandex.stockpile.server.shard.StockpileShard;
import ru.yandex.stockpile.server.shard.StockpileShardGlobals;
import ru.yandex.stockpile.server.shard.test.StockpileShardTestContext;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

/**
 * @author Stepan Koltsov
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = {
    StockpileShardTestContext.class,
    KikimrKvClientSync.class
})
public class SnapshotChunkIteratorPrefetchTest {

    @Autowired
    private StockpileShardGlobals stockpileShardGlobals;
    @Autowired
    private KikimrKvClientSync kikimrKvClientSync;
    @Autowired
    private KikimrKvClientInMem kikimrKvClientInMem;

    private static class TestShardThread extends ShardThread {

        public TestShardThread(StockpileShard shard) {
            super(shard);
        }

        @Override
        protected void stoppedReleaseResources() {

        }
    }

    @Test
    public void test() throws Exception {
        int shardId = 10;
        long kvTabletId = kikimrKvClientInMem.createKvTablet();

        SnapshotAddress snapshotAddress = new SnapshotAddress(SnapshotLevel.ETERNITY, 1123);

        for (int i = 0; i < 3; ++i) {
            String key = new ChunkFile(snapshotAddress.level(), snapshotAddress.txn(), i).reconstructCurrent();
            kikimrKvClientSync.writeDefault(kvTabletId, 0, key, new byte[] { 1, 3, 5 });
        }

        StockpileShard shard = new StockpileShard(stockpileShardGlobals, shardId, kvTabletId, "aabb");

        shard.start();

        shard.waitForInitializedOrAnyError();

        SnapshotChunkIteratorPrefetch iter = new SnapshotChunkIteratorPrefetch(
            shard.storage,
            ReadClass.OTHER, new TestShardThread(shard),
            snapshotAddress,
            3);

        for (int i = 0; i < 3; ++i) {
            assertNotNull(iter.next().join());
        }

        assertNull(iter.next().join());
    }

    @Test
    public void fastProcessing() {
        final int chunks = 1_000;
        int shardId = 10;
        long kvTabletId = kikimrKvClientInMem.createKvTablet();

        SnapshotAddress snapshotAddress = new SnapshotAddress(SnapshotLevel.ETERNITY, 1123);
        IntStream.range(0, chunks)
            .mapToObj(index -> {
                String key = new ChunkFile(snapshotAddress.level(), snapshotAddress.txn(), index).reconstructCurrent();
                byte[] bytes = {(byte) index};
                return kikimrKvClientInMem.write(kvTabletId, 0, key, bytes,
                    MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN, KikimrKvClient.Write.defaultPriority, 0);
            })
            .collect(collectingAndThen(toList(), CompletableFutures::allOfVoid))
            .join();

        StockpileShard shard = new StockpileShard(stockpileShardGlobals, shardId, kvTabletId, "aabb");
        shard.start();
        shard.waitForInitializedOrAnyError();

        SnapshotChunkIteratorPrefetch iter = new SnapshotChunkIteratorPrefetch(
            shard.storage,
            ReadClass.MERGE_READ_CHUNK, new TestShardThread(shard),
            snapshotAddress,
            chunks);

        for (int index = 0; index < chunks; index++) {
            ChunkWithNo chunk = iter.next().join();
            assertEquals(index, chunk.getNo());
            assertArrayEquals(new byte[]{(byte) index}, chunk.getContent());
        }
        assertNull(iter.next().join());
    }

    @Test
    public void changeGen() throws Exception {
        int shardId = 10;
        long kvTabletId = kikimrKvClientInMem.createKvTablet();

        SnapshotAddress snapshotAddress = new SnapshotAddress(SnapshotLevel.ETERNITY, 1123);

        for (int i = 0; i < 3; ++i) {
            String key = new ChunkFile(snapshotAddress.level(), snapshotAddress.txn(), i).reconstructCurrent();
            kikimrKvClientSync.writeDefault(kvTabletId, 0, key, new byte[] { 1, 3, 5 });
        }

        StockpileShard shard = new StockpileShard(stockpileShardGlobals, shardId, kvTabletId, "aabb");
        shard.start();

        shard.waitForInitializedOrAnyError();

        SnapshotChunkIteratorPrefetch iter = new SnapshotChunkIteratorPrefetch(
            shard.storage,
            ReadClass.OTHER, new TestShardThread(shard),
            snapshotAddress,
            3);

        assertNotNull(iter.next().join());

        kikimrKvClientInMem.incrementGeneration(kvTabletId, 0).join();

        // due to concurrent nature of prefetch iterator,
        // it can fail on first or on second iteration
        for (int i = 0; i < 2; ++i) {
            boolean genMismatch = iter.next()
                .thenApply(ignore -> false)
                .exceptionally(ExceptionHandler::isGenerationChanged)
                .join();

            if (genMismatch) {
                return;
            }
        }

        Assert.fail();
    }

}
