package ru.yandex.stockpile.server.shard.iter;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.WillClose;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.HeapStockpileSerializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.util.CloseableUtils;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.ChunkWithNo;
import ru.yandex.stockpile.server.data.dao.StockpileKvLimits;
import ru.yandex.stockpile.server.data.index.ChunkIndexArray;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.index.SnapshotIndexProperties;
import ru.yandex.stockpile.server.shard.load.Async;
import ru.yandex.stockpile.server.shard.load.AsyncIterator;

import static java.util.concurrent.CompletableFuture.failedFuture;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Vladimir Gordiychuk
 */
public class SnapshotIteratorTest {

    private ExecutorService executor;
    private ThreadLocalRandom random;

    private HeapStockpileSerializer chunkSerializer;
    private ChunkIndexArray index;
    private List<ChunkWithNo> chunks;
    private List<MetricIdAndData> expected;

    @Before
    public void before() {
        executor = ForkJoinPool.commonPool();
        random = ThreadLocalRandom.current();
        chunkSerializer = new HeapStockpileSerializer(StockpileKvLimits.RECOMMENDED_MAX_FILE_SIZE);
        index = new ChunkIndexArray();
        chunks = new ArrayList<>();
        expected = new ArrayList<>();
        executor = ForkJoinPool.commonPool();
    }

    @After
    public void tearDown() {
        CloseableUtils.close(expected);
    }

    @Test
    public void empty() {
        var it = iterator();
        assertNull(it.next().join());
    }

    @Test
    public void oneMetricOneChunk() {
        write(1, randomArray());
        nextChunk();

        assertRead();
    }

    @Test
    public void fewMetricOneChunk() {
        write(1, randomArray());
        write(2, randomArray());
        write(3, randomArray());
        nextChunk();

        assertRead();
    }

    @Test
    public void oneMetricFewChunk() {
        write(1, randomArray());
        nextChunk();
        write(2, randomArray());
        nextChunk();

        assertRead();
    }

    @Test
    public void fewMetricFewChunk() {
        write(1, randomArray());
        write(2, randomArray());
        write(3, randomArray());
        nextChunk();
        write(4, randomArray());
        write(5, randomArray());
        nextChunk();

        assertRead();
    }

    @Test
    public void many() {
        int chunksCount = random.nextInt(1, 100);
        long localId = 0;
        for (int chunkNo = 0; chunkNo < chunksCount; chunkNo++) {
            int arrayCount = random.nextInt(1, 20);
            for (int arrayIndex = 0; arrayIndex < arrayCount; arrayIndex++) {
                write(localId++, randomArray());
            }
            nextChunk();
        }

        assertRead();
    }

    private MetricArchiveMutable randomArray() {
        AggrPoint point = new AggrPoint();

        MetricArchiveMutable archive = new MetricArchiveMutable();
        archive.setType(MetricType.DGAUGE);
        int mask = TsColumn.mask | ValueColumn.mask;

        int size = random.nextInt(1, 5);
        for (int index = 0; index < size; index++) {
            archive.addRecord(randomPoint(point, mask, random));
        }
        archive.sortAndMerge();
        return archive;
    }

    private void write(long localId, @WillClose MetricArchiveMutable archive) {
        try (archive) {
            var immutable = archive.toImmutableNoCopy();
            write(localId, archive.getLastTsMillis(), immutable);
        }
    }

    private void write(long localId, long lastTsMillis, @WillClose MetricArchiveImmutable archive) {
        int before = chunkSerializer.size();
        SnapshotIndexContentSerializer.dataSerializerForVersionSealed(StockpileFormat.CURRENT)
            .serializeToEof(archive, chunkSerializer);
        int after = chunkSerializer.size();
        index.addMetric(localId, lastTsMillis, after - before);
        System.out.println("chunk "
            + index.getChunksCount()
            + ": "
            + StockpileLocalId.toString(localId)
            + ", "
            + DataSize.prettyString(after - before));
        expected.add(new MetricIdAndData(localId, lastTsMillis, 0, archive));
    }

    private void nextChunk() {
        if (chunkSerializer.size() > 0) {
            ChunkWithNo chunk = new ChunkWithNo(index.getChunksCount(), chunkSerializer.flush());
            index.finishChunk();
            chunks.add(chunk);
            System.out.println("finish chunk " + chunk.getNo() + ": " + DataSize.prettyString(chunk.getContent().length));
        }
    }

    private void assertRead() {
        AtomicInteger index = new AtomicInteger();
        var it = iterator();
        Async.forEach(it, result -> {
            if (result == null) {
                assertEquals(expected.size(), index.get());
                return;
            }

            var expect = expected.get(index.getAndIncrement());
            assertEquals(expect.localId(), result.localId());
            assertEquals(expect.lastTsMillis(), result.lastTsMillis());
            assertEquals(expect.archive(), result.archive());
            close(result.archive());
        }).join();

        assertNull(it.next().join());
        assertEquals(expected.size(), index.get());
    }

    private SnapshotIterator iterator() {
        var content = new SnapshotIndexContent(StockpileFormat.CURRENT,
            new SnapshotIndexProperties()
                .setCreatedAt(System.currentTimeMillis())
                .setRecordCount(expected.size())
                .setMetricCount(expected.size()),
            index);
        var i = new SnapshotIndex(SnapshotLevel.ETERNITY, 42, content);
        return new SnapshotIterator(i, new ChunkIterator());
    }

    private class ChunkIterator implements AsyncIterator<ChunkWithNo> {
        private AtomicInteger idx = new AtomicInteger();
        private AtomicBoolean inFlight = new AtomicBoolean();

        @Override
        public CompletableFuture<ChunkWithNo> next() {
            if (!inFlight.compareAndSet(false, true)) {
                return failedFuture(new IllegalStateException("previous fetch not completed yet"));
            }

            var future = CompletableFutures.supplyAsync(() -> {
                var i = idx.getAndIncrement();
                if (i >= chunks.size()) {
                    return null;
                }

                return chunks.get(i);
            }, executor);

            return future.whenComplete((chunkWithNo, throwable) -> {
                inFlight.set(false);
            });
        }
    }

}
