package ru.yandex.stockpile.server.shard.load;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;

import com.google.protobuf.ByteString;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KvReadRangeResult;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv.TKeyValueRequest.EPriority;
import ru.yandex.kikimr.proto.MsgbusKv.TKeyValueRequest.EStorageChannel;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.config.protobuf.stockpile.EInvalidArchiveStrategy;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.server.data.log.LongFileTypeLogEntry;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContent;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContentImmutable;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContentSerializer;
import ru.yandex.stockpile.server.data.log.StockpileLogEntrySerialized;

import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;


/**
 * @author Sergey Polovko
 */
public class KvLogParsingIteratorTest {

    private KikimrKvClientInMem kvClient = new KikimrKvClientInMem();
    private long tabletId;
    private long tabletGen;
    private ExecutorService executor;

    @Before
    public void before() {
        tabletId = kvClient.createKvTablet();
        tabletGen = kvClient.incrementGeneration(tabletId, 0).join();
        executor = ForkJoinPool.commonPool();
    }

    @Test
    public void noFiles() {
        KvLogParsingIterator it = parseIt(new TestIterator(NameRange.all()));

        StockpileLogEntryContentImmutable done = it.next().join();
        Assert.assertNull(done);
    }

    @Test
    public void emptyFile() {
        syncWriteLogEntry(1, new StockpileLogEntryContent());

        KvLogParsingIterator it = parseIt(new TestIterator(NameRange.all()));

        StockpileLogEntryContentImmutable l = it.next().join();
        Assert.assertNotNull(l);
        Assert.assertEquals(0, l.getMetricCount());

        StockpileLogEntryContentImmutable done = it.next().join();
        Assert.assertNull(done);
    }

    @Test
    public void nonEmptyFile() {
        StockpileLogEntryContent logEntry = new StockpileLogEntryContent();
        logEntry.addPointForTest(1, System.currentTimeMillis(), 3.14);
        logEntry.addPointForTest(2, System.currentTimeMillis(), 2.71);
        syncWriteLogEntry(2, logEntry);

        KvLogParsingIterator it = parseIt(new TestIterator(NameRange.all()));

        StockpileLogEntryContentImmutable l = it.next().join();
        Assert.assertNotNull(l);
        Assert.assertEquals(2, l.getMetricCount());
        Assert.assertEquals(logEntry.archiveRefByLocalId(1).toImmutable(), l.archiveRefByLocalId(1));
        Assert.assertEquals(logEntry.archiveRefByLocalId(2).toImmutable(), l.archiveRefByLocalId(2));

        StockpileLogEntryContentImmutable done = it.next().join();
        Assert.assertNull(done);
    }

    @Test
    public void multipleFiles() {
        StockpileLogEntryContent logEntry1 = new StockpileLogEntryContent();
        logEntry1.addPointForTest(1, System.currentTimeMillis(), 3.14);
        syncWriteLogEntry(3, logEntry1);

        StockpileLogEntryContent logEntry2 = new StockpileLogEntryContent();
        logEntry2.addPointForTest(2, System.currentTimeMillis(), 2.71);
        syncWriteLogEntry(4, logEntry2);

        KvLogParsingIterator it = parseIt(new TestIterator(NameRange.all()));

        StockpileLogEntryContentImmutable l1 = it.next().join();
        Assert.assertNotNull(l1);
        Assert.assertEquals(1, l1.getMetricCount());
        Assert.assertEquals(logEntry1.archiveRefByLocalId(1).toImmutable(), l1.archiveRefByLocalId(1));

        StockpileLogEntryContentImmutable l2 = it.next().join();
        Assert.assertNotNull(l2);
        Assert.assertEquals(1, l2.getMetricCount());
        Assert.assertEquals(logEntry2.archiveRefByLocalId(2).toImmutable(), l2.archiveRefByLocalId(2));

        StockpileLogEntryContentImmutable done = it.next().join();
        Assert.assertNull(done);
    }

    @Test
    public void hugeTxn() {
        StockpileLogEntryContent logEntry = prepareHugeTxn();
        syncWriteLogEntry(777, logEntry);

        KvLogParsingIterator it = parseIt(new TestIterator(NameRange.all()));

        StockpileLogEntryContentImmutable l = it.next().join();
        Assert.assertNotNull(l);
        Assert.assertEquals(logEntry.getMetricsCount(), l.getMetricCount());

        StockpileLogEntryContentImmutable done = it.next().join();
        Assert.assertNull(done);
    }

    private KvLogParsingIterator parseIt(KvReadRangeIterator it) {
        return new KvLogParsingIterator(executor, EInvalidArchiveStrategy.FAIL, it);
    }

    private static StockpileLogEntryContent prepareHugeTxn() {
        StockpileLogEntryContent logEntry = new StockpileLogEntryContent();

        long localId = 1;
        MetricArchiveMutable archive = new MetricArchiveMutable();
        archive.setType(MetricType.DGAUGE);
        archive.addRecord(randomPoint(MetricType.DGAUGE));

        int size = 0;
        while (true) {
            for (int i = 0; i < 1_000_000; i++) {
                logEntry.addArchive(localId++, archive);
                size += archive.bytesCount() + Long.SIZE;
            }

            System.out.printf("added %d metrics, size %.1f MiB\n", localId - 1, size / 1024. / 1024.);

            if (size > 2 * KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE) {
                break;
            }
        }

        archive.close();
        return logEntry;
    }

    private void syncWriteLogEntry(long txn, StockpileLogEntryContent deltaLogEntry) {
        writeLogEntry(txn, deltaLogEntry).join();
    }

    private CompletableFuture<Void> writeLogEntry(long txn, StockpileLogEntryContent deltaLogEntry) {
        ByteString content = StockpileLogEntryContentSerializer.S.serializeToByteString(deltaLogEntry);
        StockpileLogEntrySerialized serialized = new StockpileLogEntrySerialized(txn, content);

        List<KikimrKvClient.Write> writes = LongFileTypeLogEntry.I.makeWrites(
            serialized.split(), EStorageChannel.MAIN, EPriority.BACKGROUND);

        return kvClient.writeAndRenameAndDeleteAndConcat(
            tabletId, tabletGen,
            writes,
            Collections.emptyList(),
            Collections.emptyList(),
            Collections.emptyList(),
            0);
    }

    /**
     * TEST ITERATOR
     */
    final class TestIterator extends KvReadRangeIterator {
        TestIterator(NameRange nameRange) {
            super(nameRange);
        }

        @Override
        protected CompletableFuture<KvReadRangeResult> readNext(NameRange nameRange) {
            return kvClient.readRange(tabletId, tabletGen, nameRange, true, KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE,0);
        }
    }
}
