package ru.yandex.stockpile.server.shard.load;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.kikimr.client.kv.KikimrKvClient.KvEntryWithStats;
import ru.yandex.kikimr.client.kv.KvRange;
import ru.yandex.kikimr.client.kv.KvReadRangeResult;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv.TKeyValueRequest.EPriority;
import ru.yandex.kikimr.proto.MsgbusKv.TKeyValueRequest.EStorageChannel;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.concurrent.CompletableFutures;

import static java.util.stream.Collectors.toList;
import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;


/**
 * @author Sergey Polovko
 */
public class KvReadRangeIteratorTest {

    private KikimrKvClientInMem kvClient = new KikimrKvClientInMem();
    private long tabletId;
    private long tabletGen;

    @Before
    public void before() {
        kvClient = new KikimrKvClientInMem();
        tabletId = kvClient.createKvTablet();

        tabletGen = kvClient.incrementGeneration(tabletId, 0).join();

        syncWrite("a", bytes(1));
        syncWrite("a.2", new byte[1024]);
        syncWrite("a.3", new byte[1002]);
        syncWrite("b", bytes(2));
        syncWrite("b.2", new byte[1020]);
        syncWrite("c", bytes(3));
        syncWrite("d", bytes(4));
    }

    @Test
    public void readEmptyRange() {
        TestIterator it = new TestIterator(NameRange.inclusive("e", "z"));

        KvEntryWithStats done = it.next().join();
        Assert.assertNull(done);
    }

    @Test
    public void readSomeRange() {
        TestIterator it = new TestIterator(NameRange.inclusive("b", "c"));

        KvEntryWithStats b = it.next().join();
        assertEquals("b", b.getName());
        assertArrayEquals(bytes(2), b.getValue());

        KvEntryWithStats b2 = it.next().join();
        assertEquals("b.2", b2.getName());
        assertArrayEquals(new byte[1020], b2.getValue());

        KvEntryWithStats c = it.next().join();
        assertEquals("c", c.getName());
        assertArrayEquals(bytes(3), c.getValue());

        KvEntryWithStats done = it.next().join();
        Assert.assertNull(done);
    }

    @Test
    public void overrunRanges() {
        TestIterator it = new TestIterator(NameRange.inclusive("a", "c"), 1024);

        KvEntryWithStats a = it.next().join();
        assertEquals("a", a.getName());
        assertArrayEquals(bytes(1), a.getValue());

        KvEntryWithStats a2 = it.next().join();
        assertEquals("a.2", a2.getName());
        assertArrayEquals(new byte[1024], a2.getValue());

        KvEntryWithStats a3 = it.next().join();
        assertEquals("a.3", a3.getName());
        assertArrayEquals(new byte[1002], a3.getValue());

        KvEntryWithStats b = it.next().join();
        assertEquals("b", b.getName());
        assertArrayEquals(bytes(2), b.getValue());

        KvEntryWithStats b2 = it.next().join();
        assertEquals("b.2", b2.getName());
        assertArrayEquals(new byte[1020], b2.getValue());

        KvEntryWithStats c = it.next().join();
        assertEquals("c", c.getName());
        assertArrayEquals(bytes(3), c.getValue());

        KvEntryWithStats done = it.next().join();
        Assert.assertNull(done);
    }

    @Test
    public void readManyFilesWithDifferentSize() {
        var limit = 1 << 20;
        var random = ThreadLocalRandom.current();
        var expected = IntStream.range(1, 1000)
            .parallel()
            .mapToObj(idx -> {
                var name = String.format("c.l.%05dz", idx);
                var size = random.nextInt(1, limit);
                var bytes = new byte[size];
                random.nextBytes(bytes);

                KvEntryWithStats file = new KvEntryWithStats(name, bytes, bytes.length, System.currentTimeMillis());
                return write(file.getName(), file.getValue()).thenApply(ignore -> file);
            })
            .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOf))
            .join();

        AtomicInteger index = new AtomicInteger();

        var it = new TestIterator(NameRange.inclusive("c.l.00001z", "c.l.01000z"), limit);
        Async.forEach(it, file -> {
            var expectFile = expected.get(index.getAndIncrement());

            assertEquals(expectFile.getName(), file.getName());
            assertEquals(expectFile.getSize(), file.getSize());
            assertArrayEquals(expectFile.getValue(), file.getValue());
        }).join();
        assertEquals(expected.size(), index.get());
    }

    @Test
    public void name() {
        String path = "/Root/Test";
        var p = Path.of(path);
        System.out.println(p);
        System.out.println(p.getFileName());
        System.out.println(p.getParent());
    }

    private static byte[] bytes(int... v) {
        byte[] result = new byte[v.length];
        for (int i = 0; i < v.length; i++) {
            result[i] = (byte) v[i];
        }
        return result;
    }

    private void syncWrite(String key, byte[] value) {
        write(key, value).join();
    }

    private CompletableFuture<Void> write(String key, byte[] value) {
        return kvClient.write(tabletId, tabletGen, key, value, EStorageChannel.MAIN, EPriority.BACKGROUND, 0);
    }

    /**
     * TEST ITERATOR
     */
    final class TestIterator extends KvReadRangeIterator {
        final long bytesLimit;

        TestIterator(NameRange nameRange) {
            this(nameRange, KvRange.LEN_UNLIMITED);
        }

        TestIterator(NameRange nameRange, long bytesLimit) {
            super(nameRange);
            this.bytesLimit = bytesLimit;
        }

        @Override
        protected CompletableFuture<KvReadRangeResult> readNext(NameRange nameRange) {
            return kvClient.readRange(tabletId, tabletGen, nameRange, true, bytesLimit, 0);
        }
    }
}
