package ru.yandex.stockpile.server.shard.load;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.util.collection.array.LongArrayView;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.memState.MetricToArchiveMap;
import ru.yandex.stockpile.server.shard.InvalidArchiveStrategy;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.model.point.AggrPoints.point;
import static ru.yandex.solomon.util.CloseableUtils.close;


/**
 * @author Sergey Polovko
 */
public class MetricArchivesMergerTest {

    @Test
    public void merge() throws InterruptedException {
        ExecutorService mergerExecutor = Executors.newSingleThreadExecutor();
        try {
            final long ts = System.currentTimeMillis();

            MetricArchiveMutable archiveA = new MetricArchiveMutable();
            archiveA.addRecord(point(ts, Math.PI));

            MetricArchiveMutable archiveB = new MetricArchiveMutable();
            archiveB.addRecord(point(ts, Math.E));

            MetricArchiveMutable archiveC = new MetricArchiveMutable();
            archiveC.addRecord(point(ts + 1, Math.PI * 2));

            MetricArchivesMerger merger = new MetricArchivesMerger(52, InvalidArchiveStrategy.fail(), mergerExecutor, 1);
            merger.merge(new Long2ObjectOpenHashMap<MetricArchiveImmutable>() {{
                put(1, archiveA.toImmutable());
                put(2, archiveB.toImmutable());
            }}).join();

            merger.merge(new Long2ObjectOpenHashMap<MetricArchiveImmutable>() {{
                put(1, archiveC.toImmutable());
            }}).join();

            MetricToArchiveMap result = merger.combineResults();
            Assert.assertEquals(2, result.size());

            MetricArchiveMutable mergedArchive = new MetricArchiveMutable();
            mergedArchive.updateWith(archiveA);
            mergedArchive.updateWith(archiveC);
            Assert.assertEquals(mergedArchive, result.getById(1));

            Assert.assertEquals(archiveB, result.getById(2));
            result.release();
            close(archiveA, archiveB, archiveC, mergedArchive);
        } finally {
            mergerExecutor.shutdown();
            mergerExecutor.awaitTermination(1, TimeUnit.SECONDS);
        }
    }

    @Test
    public void threadSafety() throws InterruptedException, ExecutionException {
        final int producers = 8;
        final int points = 1000;

        ExecutorService mergerExecutor = Executors.newFixedThreadPool(2);
        ExecutorService producersExecutor = Executors.newFixedThreadPool(producers);
        try {
            MetricArchivesMerger merger = new MetricArchivesMerger(42, InvalidArchiveStrategy.fail(), mergerExecutor, 4);
            CountDownAwait countDown = new CountDownAwait(producers * points);

            Future[] futures = new Future[producers];
            for (int i = 0; i < producers; i++) {
                final int producerNum = i + 1;
                futures[i] = producersExecutor.submit(() -> {
                    for (int j = 0; j < points; j++) {
                        var archive = MetricArchiveImmutable.of(point(producerNum + producers * j, Math.PI));
                        Long2ObjectMap<MetricArchiveImmutable> metrics = new Long2ObjectOpenHashMap<>();
                        metrics.put(1, archive);
                        merger.merge(metrics, countDown);
                    }
                });
            }

            producersExecutor.shutdown();
            producersExecutor.awaitTermination(1, TimeUnit.SECONDS);

            // rethrow exceptions if any
            for (Future future : futures) {
                future.get();
            }

            countDown.await();

            MetricToArchiveMap result = merger.combineResults();
            Assert.assertEquals(1, result.size());

            MetricArchiveMutable mergedArchive = result.getById(1);
            AggrGraphDataArrayList graphData = mergedArchive.toAggrGraphDataArrayList();
            Assert.assertEquals(producers * points, graphData.length());

            LongArrayView timestamps = graphData.getTimestamps();
            for (int i = 0; i < timestamps.length(); i++) {
                Assert.assertEquals(i + 1, timestamps.at(i));
            }
            result.release();
        } finally {
            mergerExecutor.shutdown();
            mergerExecutor.awaitTermination(1, TimeUnit.SECONDS);
        }
    }

    @Test
    public void ignoreInvalidMetricType() throws InterruptedException {
        ExecutorService mergerExecutor = Executors.newSingleThreadExecutor();
        try {
            MetricArchiveMutable archiveA = new MetricArchiveMutable();
            archiveA.setType(MetricType.COUNTER);
            archiveA.addRecord(randomPoint(MetricType.COUNTER));

            MetricArchiveMutable archiveB = new MetricArchiveMutable();
            archiveB.setType(MetricType.HIST);
            archiveB.addRecord(randomPoint(MetricType.HIST));

            MetricArchiveMutable archiveC = new MetricArchiveMutable();
            archiveC.setType(MetricType.COUNTER);
            archiveC.addRecord(randomPoint(MetricType.COUNTER));

            final long localId = StockpileLocalId.random();
            MetricArchivesMerger merger = new MetricArchivesMerger(32, InvalidArchiveStrategy.fail(), mergerExecutor, 1);
            merger.merge(new Long2ObjectOpenHashMap<MetricArchiveImmutable>() {{
                put(localId, archiveA.toImmutable());
            }}).join();
            merger.merge(new Long2ObjectOpenHashMap<MetricArchiveImmutable>() {{
                put(localId, archiveB.toImmutable());
            }}).join();
            merger.merge(new Long2ObjectOpenHashMap<MetricArchiveImmutable>() {{
                put(localId, archiveC.toImmutable());
            }}).join();

            MetricToArchiveMap result = merger.combineResults();
            assertEquals(1, result.size());

            MetricArchiveMutable mergedArchive = new MetricArchiveMutable();
            mergedArchive.updateWith(archiveA);
            mergedArchive.updateWith(archiveC);
            assertEquals(mergedArchive, result.getById(localId));
            result.release();
            close(archiveA, archiveB, archiveC, mergedArchive);
        } finally {
            mergerExecutor.shutdown();
            mergerExecutor.awaitTermination(1, TimeUnit.SECONDS);
        }
    }
}
