package ru.yandex.stockpile.server.shard.merge;

import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

import static org.hamcrest.Matchers.instanceOf;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.stockpile.server.shard.merge.Utils.writeUntilCloseFrame;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class ArchiveItemIteratorTest {

    @Parameterized.Parameter
    public StockpileFormat format;

    @Parameterized.Parameters(name = "{0}")
    public static Object[] data() {
        return StockpileFormat.values();
    }

    @Test
    public void emptyArchive() {
        MetricArchiveMutable archive = archive();
        var it = iterator(archive);
        assertNull(it.next());
        assertNull(it.next());
        assertNull(it.next());
        close(archive);
    }

    @Test
    public void noFrames() {
        MetricArchiveMutable archive = archive();
        archive.setType(MetricType.DGAUGE);

        long ts0 = System.currentTimeMillis();
        AggrPoint point = new AggrPoint();
        point.setTsMillis(ts0);
        for (int index = 0; index < 10; index++) {
            point.setValue(index);
            archive.addRecord(point);
            point.setTsMillis(point.tsMillis + 10_000);
        }

        var it = iterator(archive);
        var item = it.next();
        assertNotNull(item);
        assertEquals(ts0, item.getFirstTsMillis());
        assertEquals(ts0 + (10_000 * 9), item.getLastTsMillis());
        assertEquals(archive.toAggrGraphDataArrayList(), AggrGraphDataArrayList.of(item.iterator()));
        assertEquals(archive.getCompressedDataRaw().bytesSize(), item.getElapsedBytes(), 10);
        assertThat(item, instanceOf(ItemIterator.class));
        assertNull(it.next());
        close(archive);
    }

    @Test
    public void oneClosedFrame() {
        MetricArchiveMutable archive = archive();
        archive.setType(MetricType.DGAUGE);

        var expected = writeUntilCloseFrame(archive);
        var it = iterator(archive);
        var item = it.next();
        assertNotNull(item);
        assertEquals(expected.getTsMillis(0), item.getFirstTsMillis());
        assertEquals(expected.getTsMillis(expected.length() - 1), item.getLastTsMillis());
        assertEquals(expected, AggrGraphDataArrayList.of(item.iterator()));
        assertEquals(archive.getCompressedDataRaw().bytesSize(), item.getElapsedBytes());
        assertThat(item, instanceOf(ItemFrame.class));
        assertNull(it.next());
        close(archive);
    }

    @Test
    public void oneClosedFrameAndUnclosed() {
        MetricArchiveMutable archive = archive();
        archive.setType(MetricType.DGAUGE);

        var expected = writeUntilCloseFrame(archive);

        var point = expected.getAnyPoint(expected.length() - 1);
        point.setTsMillis(point.tsMillis + 10_000);
        point.setValue(42);
        archive.addRecord(point);

        var it = iterator(archive);
        {
            var first = it.next();
            assertNotNull(first);
            assertEquals(expected.getTsMillis(0), first.getFirstTsMillis());
            assertEquals(expected.getTsMillis(expected.length() - 1), first.getLastTsMillis());
            assertEquals(expected, AggrGraphDataArrayList.of(first.iterator()));
            assertEquals(archive.getCompressedDataRaw().bytesSize(), first.getElapsedBytes(), 20);
            assertThat(first, instanceOf(ItemFrame.class));
        }

        {
            var second = it.next();
            assertNotNull(second);
            assertEquals(point.tsMillis, second.getFirstTsMillis());
            assertEquals(point.tsMillis, second.getLastTsMillis());
            assertEquals(AggrGraphDataArrayList.of(point), AggrGraphDataArrayList.of(second.iterator()));
            assertEquals(16, second.getElapsedBytes(), 10);
            assertThat(second, instanceOf(ItemIterator.class));
        }

        assertNull(it.next());
        close(archive);
    }

    @Test
    public void fewClosedFrame() {
        MetricArchiveMutable archive = archive();
        archive.setType(MetricType.DGAUGE);

        var frames = IntStream.range(0, 3)
            .mapToObj(ignore -> writeUntilCloseFrame(archive))
            .collect(Collectors.toList());

        var it = iterator(archive);
        for (var expected : frames) {
            var item = it.next();

            var frameArchive = archive();
            frameArchive.setType(MetricType.DGAUGE);
            frameArchive.addAll(expected);
            frameArchive.closeFrame();

            assertNotNull(item);
            assertEquals(expected.getTsMillis(0), item.getFirstTsMillis());
            assertEquals(expected.getTsMillis(expected.length() - 1), item.getLastTsMillis());
            assertEquals(expected, AggrGraphDataArrayList.of(item.iterator()));
            assertEquals(frameArchive.getCompressedDataRaw().bytesSize(), item.getElapsedBytes(), 100);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        assertNull(it.next());
        close(archive);
    }

    private MetricArchiveMutable archive() {
        return Utils.archive(format);
    }

    private Iterator iterator(MetricArchiveMutable archive) {
        return ArchiveItemIterator.of(archive);
    }
}
