package ru.yandex.stockpile.server.shard.merge;

import java.util.stream.Stream;

import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricType;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.stockpile.server.shard.merge.Utils.assertArchiveEquals;
import static ru.yandex.stockpile.server.shard.merge.Utils.assumeMultipleFormats;
import static ru.yandex.stockpile.server.shard.merge.Utils.writeUntilCloseFrame;

/**
 * @author Vladimir Gordiychuk
 */
public class CompressCollectorTest {
    private StockpileFormat format = StockpileFormat.CURRENT;

    @Test
    public void empty() {
        var archive = archive();
        var result = compress(format, archive);
        assertEquals(0, result.buffer.readableBits());
        assertEquals(archive.columnSetMask(), result.mask);
        assertEquals(archive.getFormat(), result.format);
        assertEquals(archive.getLastTsMillis(), result.lastTsMillis);
        assertEquals(archive.getRecordCount(), result.records);
        close(archive);
    }

    @Test
    public void onePoint() {
        var archive = archive();
        var point = randomPoint(archive.columnSetMask());
        archive.addRecord(point);
        var result = compress(format, archive);
        assertArchiveEquals(archive, result);
        close(archive);
    }

    @Test
    public void archiveAndEmpty() {
        var one = archive();
        writeUntilCloseFrame(archive());
        writeUntilCloseFrame(archive());

        var two = archive();
        var result = compress(format, one, two);
        assertArchiveEquals(one, result);
        close(one, two);
    }

    @Test
    public void fewClosedFrame() {
        var archive = archive();
        writeUntilCloseFrame(archive);
        writeUntilCloseFrame(archive);
        writeUntilCloseFrame(archive);

        var result = compress(format, archive);
        assertArchiveEquals(archive, result);
        close(archive);
    }

    @Test
    public void fewClosedAndUnclosed() {
        var archive = archive();
        writeUntilCloseFrame(archive);
        writeUntilCloseFrame(archive);

        AggrPoint point = new AggrPoint();
        point.setTsMillis(archive.getLastTsMillis() + 1);
        point.setValue(42);
        archive.addRecord(point);

        var result = compress(format, archive);
        assertArchiveEquals(archive, result);
        close(archive);
    }

    @Test
    public void unclosedClosedClosed() {
        var one = archive();
        one.addRecord(randomPoint(one.columnSetMask()));

        var two = archive();
        var twoOne = writeUntilCloseFrame(two, one.getLastTsMillis() + 1);
        var twoTwo = writeUntilCloseFrame(two, two.getLastTsMillis() + 1);

        var expected = archive();
        expected.addAll(one);
        expected.addAll(twoOne);
        expected.closeFrame();
        expected.addAll(twoTwo);
        expected.closeFrame();

        var result = compress(format, one, two);
        assertArchiveEquals(expected, result);
        close(one, two);
    }

    @Test
    public void fromNewFormatToOld() {
        assumeMultipleFormats();
        var one = archive();
        writeUntilCloseFrame(one);

        var expected = new MetricArchiveMutable(MetricHeader.defaultValue, StockpileFormat.MIN);
        expected.addAll(one);
        expected.closeFrame();

        var result = compress(StockpileFormat.MIN, one);
        assertArchiveEquals(expected, result);
        close(expected, one);
    }

    @Test
    public void fromOldToNewFormat() {
        var one = new MetricArchiveMutable(MetricHeader.defaultValue, StockpileFormat.MIN);
        one.setType(MetricType.DGAUGE);
        writeUntilCloseFrame(one);

        var expected = new MetricArchiveMutable(MetricHeader.defaultValue, StockpileFormat.MAX);
        expected.addAll(one);
        expected.closeFrame();

        var result = compress(StockpileFormat.MAX, one);
        assertArchiveEquals(expected, result);
        close(one, expected);
    }

    @Test
    public void firstTsMillisUndefined() {
        var one = archive();
        one.setType(MetricType.DGAUGE);
        writeUntilCloseFrame(one);

        var it = OneItemIterator.of(one.iterator(), MetricType.DGAUGE, 0, one.getLastTsMillis(), one.bytesCount());
        var result = CompressCollector.collect(format, MetricType.DGAUGE, TsColumn.mask | ValueColumn.mask, 16, it);
        assertArchiveEquals(one, result);
        close(one);
    }

    private CompressResult compress(StockpileFormat format, MetricArchiveMutable... archives) {
        var it = Stream.of(archives)
            .map(ArchiveItemIterator::of)
            .collect(collectingAndThen(toList(), MergeIterator::of));
        return CompressCollector.collect(format, MetricType.DGAUGE, TsColumn.mask | ValueColumn.mask, 16, it);
    }

    private MetricArchiveMutable archive() {
        return Utils.archive(StockpileFormat.CURRENT);
    }
}
