package ru.yandex.stockpile.server.shard.merge;

import java.util.stream.Stream;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.instanceOf;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.stockpile.server.shard.merge.Utils.writeUntilCloseFrame;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class ConcatIteratorTest {

    @Parameterized.Parameter
    public StockpileFormat format;

    @Parameterized.Parameters(name = "{0}")
    public static Object[] data() {
        return StockpileFormat.values();
    }

    @Test
    public void emptyArchives() {
        var it = iterator(archive(), archive(), archive());
        assertNull(it.next());
        assertNull(it.next());
        assertNull(it.next());
    }

    @Test
    public void emptyList() {
        var it = iterator();
        assertNull(it.next());
        assertNull(it.next());
        assertNull(it.next());
    }

    @Test
    public void closedFrames() {
        var one = archive();
        var expectedOne = writeUntilCloseFrame(one);
        var expectedOneTwo = writeUntilCloseFrame(one);

        var two = archive();
        var expectedTwo = writeUntilCloseFrame(two, one.getLastTsMillis() + 1);

        var it = iterator(one, two);
        { // archive one frame one
            var item = it.next();
            assertNotNull(item);
            assertEquals(expectedOne.getTsMillis(0), item.getFirstTsMillis());
            assertEquals(expectedOne.getTsMillis(expectedOne.length() - 1), item.getLastTsMillis());
            assertEquals(expectedOne, AggrGraphDataArrayList.of(item.iterator()));
            assertThat(item, instanceOf(ItemFrame.class));
        }
        { // archive one frame two
            var item = it.next();
            assertNotNull(item);
            assertEquals(expectedOneTwo.getTsMillis(0), item.getFirstTsMillis());
            assertEquals(expectedOneTwo.getTsMillis(expectedOneTwo.length() - 1), item.getLastTsMillis());
            assertEquals(expectedOneTwo, AggrGraphDataArrayList.of(item.iterator()));
            assertThat(item, instanceOf(ItemFrame.class));
        }
        { // archive two frame one
            var item = it.next();
            assertNotNull(item);
            assertEquals(expectedTwo.getTsMillis(0), item.getFirstTsMillis());
            assertEquals(expectedTwo.getTsMillis(expectedTwo.length() - 1), item.getLastTsMillis());
            assertEquals(expectedTwo, AggrGraphDataArrayList.of(item.iterator()));
            assertEquals(two.getCompressedDataRaw().bytesSize(), item.getElapsedBytes());
            assertThat(item, instanceOf(ItemFrame.class));
        }

        assertNull(it.next());
        assertNull(it.next());
        assertNull(it.next());
        close(one, two);
    }

    private MetricArchiveMutable archive() {
        return Utils.archive(format);
    }

    private Iterator iterator(MetricArchiveMutable... archives) {
        return Stream.of(archives)
            .map(ArchiveItemIterator::of)
            .collect(collectingAndThen(toList(), ConcatIterator::of));
    }
}
