package ru.yandex.stockpile.server.shard.merge;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.decim.DecimPolicy;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.instanceOf;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.stockpile.client.TestUtil.timeToMillis;
import static ru.yandex.stockpile.server.shard.merge.Utils.assertItemEquals;
import static ru.yandex.stockpile.server.shard.merge.Utils.writeUntilCloseFrame;

/**
 * @author Vladimir Gordiychuk
 */
public class DecimIteratorTest {
    private StockpileFormat format = StockpileFormat.CURRENT;

    private DecimPolicy policy = DecimPolicy.newBuilder()
        .after(Duration.ofDays(7)).to(Duration.ofMinutes(5))
        .after(Duration.ofDays(14)).to(Duration.ofMinutes(60))
        .build();

    @Test
    public void emptyList() {
        var it = iterator(policy);
        assertNull(it.next());
        assertNull(it.next());
        assertNull(it.next());
    }

    @Test
    public void emptyArchives() {
        var it = iterator(policy, archive(), archive(), archive());
        assertNull(it.next());
        assertNull(it.next());
        assertNull(it.next());
    }

    @Test
    public void skipNoPointsToDecim() {
        long now = System.currentTimeMillis();

        var one = archive();
        var oneOne = writeUntilCloseFrame(one);
        var oneTwo = writeUntilCloseFrame(one);

        var two = archive();
        var twoOne = writeUntilCloseFrame(two, one.getLastTsMillis() + 1);

        var it = iterator(policy, now, one, two);
        {
            var item = it.next();
            assertItemEquals(item, oneOne);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        {
            var item = it.next();
            assertItemEquals(item, oneTwo);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        {
            var item = it.next();
            assertItemEquals(item, twoOne);
            assertThat(item, instanceOf(ItemFrame.class));
        }

        assertNull(it.next());
        assertNull(it.next());
        close(one, two);
    }

    @Test
    public void skipAlreadyDecimed() {
        var archive = archive();
        var one = writeUntilCloseFrame(archive);
        var two = writeUntilCloseFrame(archive);

        long now = two.getTsMillis(two.length() - 1) + TimeUnit.DAYS.toMillis(35);
        long decimatedAt = now - TimeUnit.DAYS.toMillis(1);
        var it = iterator(policy, now, decimatedAt, archive);
        {
            var item = it.next();
            assertItemEquals(item, one);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        {
            var item = it.next();
            assertItemEquals(item, two);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        close(archive);
    }

    @Test
    public void decimOneItem() {
        DecimPolicy policy = DecimPolicy.newBuilder()
            .after(Duration.ofDays(7)).to(Duration.ofMinutes(5))
            .after(Duration.ofDays(30)).to(Duration.ofHours(1))
            .after(Duration.ofDays(60)).to(Duration.ofDays(1))
            .build();

        long now = timeToMillis("2018-08-09T15:15:18.010Z");
        AggrGraphDataArrayList source = AggrGraphDataArrayList.of(
            // to 1 day group
            point("2018-05-01T12:00:00Z"),
            point("2018-05-01T13:00:00Z"),
            point("2018-05-01T16:00:00Z"),
            point("2018-05-01T17:00:00Z"),
            point("2018-05-02T18:00:00Z"),

            // to 1 hours group
            point("2018-07-01T14:15:00Z"),
            point("2018-07-01T14:20:00Z"),
            point("2018-07-01T14:25:00Z"),
            point("2018-07-01T15:15:00Z"),
            point("2018-07-01T15:20:00Z"),

            // to 5 min group
            point("2018-08-01T14:17:15Z"),
            point("2018-08-01T14:17:30Z"),
            point("2018-08-01T14:17:45Z"),
            point("2018-08-01T14:21:45Z"),
            point("2018-08-01T14:23:45Z"),

            // no decim group
            point("2018-08-09T14:17:15Z"),
            point("2018-08-09T14:17:30Z"),
            point("2018-08-09T14:17:45Z")
        );

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
            // to 1 days group
            point("2018-05-01T00:00:00Z"),
            point("2018-05-02T00:00:00Z"),

            // to 1 hours group
            point("2018-07-01T14:00:00Z"),
            point("2018-07-01T15:00:00Z"),

            // to 5 min group
            point("2018-08-01T14:15:00Z"),
            point("2018-08-01T14:20:00Z"),

            // no decim group
            point("2018-08-09T14:17:15Z"),
            point("2018-08-09T14:17:30Z"),
            point("2018-08-09T14:17:45Z")
        ).cloneWithMask(StockpileColumn.TS.mask());

        var archive = archive();
        archive.addAll(source);

        var it = iterator(policy, now, archive);
        var item = it.next();
        assertNotNull(item);
        assertEquals(Instant.parse("2018-05-01T12:00:00Z"), Instant.ofEpochMilli(item.getFirstTsMillis()));
        assertEquals(Instant.parse("2018-08-09T14:17:45Z"), Instant.ofEpochMilli(item.getLastTsMillis()));
        assertNotEquals(0, item.getElapsedBytes());
        assertThat(item, instanceOf(ItemIterator.class));
        var result = AggrGraphDataArrayList.of(item.iterator()).cloneWithMask(StockpileColumn.TS.mask()).iterator();
        MetricsMatcher.assertEqualTo(expected, result);

        assertNull(it.next());
        assertNull(it.next());
        close(archive);
    }

    @Test
    public void decimMultipleItem() {
        DecimPolicy policy = DecimPolicy.newBuilder()
            .after(Duration.ofDays(7)).to(Duration.ofMinutes(5))
            .after(Duration.ofDays(30)).to(Duration.ofHours(1))
            .after(Duration.ofDays(60)).to(Duration.ofDays(1))
            .build();

        long now = timeToMillis("2018-08-09T15:15:18.010Z");
        var one = archive();
        one.addAll(AggrGraphDataArrayList.of(
            // to 1 day group
            point("2018-05-01T12:00:00Z"),
            point("2018-05-01T13:00:00Z"),
            point("2018-05-01T16:00:00Z"),
            point("2018-05-01T17:00:00Z"),
            point("2018-05-02T18:00:00Z")
        ));

        var two = archive();
        two.addAll(AggrGraphDataArrayList.of(
            // to 1 hours group
            point("2018-07-01T14:15:00Z"),
            point("2018-07-01T14:20:00Z"),
            point("2018-07-01T14:25:00Z"),
            point("2018-07-01T15:15:00Z"),
            point("2018-07-01T15:20:00Z")
        ));

        var tree = archive();
        tree.addAll(AggrGraphDataArrayList.of(
            // to 5 min group
            point("2018-08-01T14:17:15Z"),
            point("2018-08-01T14:17:30Z"),
            point("2018-08-01T14:17:45Z"),
            point("2018-08-01T14:21:45Z"),
            point("2018-08-01T14:23:45Z")
        ));

        var four = archive();
        four.addAll(AggrGraphDataArrayList.of(
            // no decim group
            point("2018-08-09T14:17:15Z"),
            point("2018-08-09T14:17:30Z"),
            point("2018-08-09T14:17:45Z")
        ));

        var it = iterator(policy, now, one, two, tree, four);
        {
            AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
                // to 1 days group
                point("2018-05-01T00:00:00Z"),
                point("2018-05-02T00:00:00Z"),

                // to 1 hours group
                point("2018-07-01T14:00:00Z"),
                point("2018-07-01T15:00:00Z"),

                // to 5 min group
                point("2018-08-01T14:15:00Z"),
                point("2018-08-01T14:20:00Z")
            ).cloneWithMask(StockpileColumn.TS.mask());

            var item = it.next();
            assertNotNull(item);
            assertEquals(Instant.parse("2018-05-01T12:00:00Z"), Instant.ofEpochMilli(item.getFirstTsMillis()));
            assertEquals(Instant.parse("2018-08-01T14:23:45Z"), Instant.ofEpochMilli(item.getLastTsMillis()));
            assertNotEquals(0, item.getElapsedBytes());
            assertThat(item, instanceOf(ItemIterator.class));
            var result = AggrGraphDataArrayList.of(item.iterator()).cloneWithMask(StockpileColumn.TS.mask()).iterator();
            MetricsMatcher.assertEqualTo(expected, result);
        }
        {
            AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
                // no decim group
                point("2018-08-09T14:17:15Z"),
                point("2018-08-09T14:17:30Z"),
                point("2018-08-09T14:17:45Z")
            ).cloneWithMask(StockpileColumn.TS.mask());

            var item = it.next();
            assertNotNull(item);
            assertEquals(Instant.parse("2018-08-09T14:17:15Z"), Instant.ofEpochMilli(item.getFirstTsMillis()));
            assertEquals(Instant.parse("2018-08-09T14:17:45Z"), Instant.ofEpochMilli(item.getLastTsMillis()));
            assertNotEquals(0, item.getElapsedBytes());
            assertThat(item, instanceOf(ItemIterator.class));
            var result = AggrGraphDataArrayList.of(item.iterator()).cloneWithMask(StockpileColumn.TS.mask()).iterator();
            MetricsMatcher.assertEqualTo(expected, result);
        }

        assertNull(it.next());
        assertNull(it.next());
        close(one, two, tree, four);
    }

    private AggrPoint point(String time) {
        AggrPoint point = randomPoint(TsColumn.mask | ValueColumn.mask);
        point.tsMillis = timeToMillis(time);
        return point;
    }

    private MetricArchiveMutable archive() {
        var archive = new MetricArchiveMutable(MetricHeader.defaultValue, format);
        archive.setType(MetricType.DGAUGE);
        return archive;
    }

    private Iterator iterator(DecimPolicy policy, MetricArchiveMutable... archives) {
        return iterator(policy, System.currentTimeMillis(), 0, archives);
    }

    private Iterator iterator(DecimPolicy policy, long now, MetricArchiveMutable... archives) {
        return iterator(policy, now, 0, archives);
    }

    private Iterator iterator(DecimPolicy policy, long now, long decimated, MetricArchiveMutable... archives) {
        var merged = Stream.of(archives)
            .map(ArchiveItemIterator::of)
            .collect(collectingAndThen(toList(), MergeIterator::of));

        return DecimIterator.of(merged, policy, now, decimated);
    }
}
