package ru.yandex.stockpile.server.shard.merge;

import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

import static org.hamcrest.Matchers.instanceOf;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.stockpile.server.shard.merge.Utils.assertItemEquals;
import static ru.yandex.stockpile.server.shard.merge.Utils.writeUntilCloseFrame;

/**
 * @author Vladimir Gordiychuk
 */
public class DeleteBeforeIteratorTest {
    private final StockpileFormat format = StockpileFormat.CURRENT;

    @Test
    public void empty() {
        var it = iterator(System.currentTimeMillis(), archive());
        assertNull(it.next());
        assertNull(it.next());
    }

    @Test
    public void deleteAll() {
        var archive = archive();
        writeUntilCloseFrame(archive);

        var it = iterator(DeleteBeforeField.DELETE_ALL, archive);
        assertNull(it.next());
        assertNull(it.next());
        close(archive);
    }

    @Test
    public void deleteBeforeSkip() {
        long ts0 = System.currentTimeMillis();
        var one = archive();
        var oneOne = writeUntilCloseFrame(one, ts0);

        var it = iterator(ts0, one);
        {
            var item = it.next();
            assertItemEquals(item, oneOne);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        assertNull(it.next());
        close(one);
    }

    @Test
    public void deleteBeforeDropChunks() {
        var archive = archive();
        var one = writeUntilCloseFrame(archive);
        var two = writeUntilCloseFrame(archive);

        var it = iterator(two.getTsMillis(0), archive);
        {
            var item = it.next();
            assertItemEquals(item, two);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        assertNull(it.next());
        close(archive);
    }

    @Test
    public void filterItemsIntoFrame() {
        var archive = archive();
        var one = writeUntilCloseFrame(archive);
        var it = iterator(one.getTsMillis(5), archive);

        var expected = AggrGraphDataArrayList.of(one.slice(5, one.length()));
        {
            var item = it.next();
            assertThat(item, instanceOf(ItemIterator.class));
            assertItemEquals(item, expected);
        }
        close(archive);
    }

    private MetricArchiveMutable archive() {
        var archive = new MetricArchiveMutable(MetricHeader.defaultValue, format);
        archive.setType(MetricType.DGAUGE);
        return archive;
    }

    private Iterator iterator(long deleteBefore, MetricArchiveMutable archive) {
        return DeleteBeforeIterator.of(deleteBefore, ArchiveItemIterator.of(archive));
    }
}
