package ru.yandex.stockpile.server.shard.merge;

import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Stream;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.MergingAggrGraphDataIterable;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.instanceOf;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.stockpile.server.shard.merge.MetricsMatcher.assertEqualTo;
import static ru.yandex.stockpile.server.shard.merge.Utils.assertItemEquals;
import static ru.yandex.stockpile.server.shard.merge.Utils.writeUntilCloseFrame;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class MergeItemIteratorTest {

    @Parameterized.Parameter
    public StockpileFormat format;

    @Parameterized.Parameters(name = "{0}")
    public static Object[] data() {
        return StockpileFormat.values();
    }

    @Test
    public void emptyList() {
        var it = iterator();
        assertNull(it.next());
        assertNull(it.next());
        assertNull(it.next());
    }

    @Test
    public void emptyArchives() {
        var it = iterator(archive(), archive(), archive());
        assertNull(it.next());
        assertNull(it.next());
        assertNull(it.next());
    }

    @Test
    public void mergeNoOverlaps() {
        var one = archive();
        var oneOne = writeUntilCloseFrame(one, System.currentTimeMillis());
        var oneTwo = writeUntilCloseFrame(one, one.getLastTsMillis() + 1);

        var two = archive();
        var twoOne = writeUntilCloseFrame(two, one.getLastTsMillis() + 1);

        var it = iterator(one, two);
        {
            var item = it.next();
            assertItemEquals(item, oneOne);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        {
            var item = it.next();
            assertItemEquals(item, oneTwo);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        {
            var item = it.next();
            assertItemEquals(item, twoOne);
            assertThat(item, instanceOf(ItemFrame.class));
        }

        assertNull(it.next());
        assertNull(it.next());
        close(one, two);
    }

    @Test
    public void mergeOverlaps() {
        var one = archive();
        var oneOne = writeUntilCloseFrame(one, System.currentTimeMillis());
        var oneTwo = writeUntilCloseFrame(one, one.getLastTsMillis() + 1);

        var two = archive();
        var twoOne = writeUntilCloseFrame(two, oneTwo.getTsMillis(0));

        var it = iterator(one, two);
        {
            var item = it.next();
            assertItemEquals(item, oneOne);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        {
            var item = it.next();
            assertItemEquals(item, oneTwo, twoOne);
            assertThat(item, instanceOf(ItemIterator.class));
        }

        assertNull(it.next());
        assertNull(it.next());
        close(one, two);
    }

    @Test
    public void mergeRandomOverlaps() {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        for (int iteration = 0; iteration < 10; iteration++) {
            var one = archive();
            {
                for (int index = 0; index < random.nextInt(1, 5); index++) {
                    writeUntilCloseFrame(one);
                }
            }

            var two = archive();
            {
                long since = random.nextLong(one.getFirstTsMillis(), one.getLastTsMillis());
                writeUntilCloseFrame(two, since);
                for (int index = 0; index < random.nextInt(1, 4); index++) {
                    writeUntilCloseFrame(two);
                }
            }

            var tree = archive();
            {
                long since = random.nextLong(two.getFirstTsMillis(), two.getLastTsMillis());
                writeUntilCloseFrame(tree, since);
                for (int index = 0; index < random.nextInt(1, 4); index++) {
                    writeUntilCloseFrame(tree);
                }
            }

            var it = new AggrPointIteratorWrapper(iterator(one, two, tree));
            var expected = MergingAggrGraphDataIterable.of(List.of(one, two, tree)).iterator();
            assertEqualTo(expected, it);
            close(one, two, tree);
        }
    }

    @Test
    public void mergePushIntoPastNoOverlaps() {
        var two = archive();
        var twoOne = writeUntilCloseFrame(two, System.currentTimeMillis());
        var twoTwo = writeUntilCloseFrame(two, two.getLastTsMillis() + 1);

        var one = archive();
        var oneOne = writeUntilCloseFrame(one, two.getLastTsMillis() + 1);

        var it = iterator(one, two);
        {
            var item = it.next();
            assertItemEquals(item, twoOne);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        {
            var item = it.next();
            assertItemEquals(item, twoTwo);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        {
            var item = it.next();
            assertItemEquals(item, oneOne);
            assertThat(item, instanceOf(ItemFrame.class));
        }

        assertNull(it.next());
        assertNull(it.next());
        close(one, two);
    }

    @Test
    public void mergePushIntoPastOverlaps() {
        var two = archive();
        var twoOne = writeUntilCloseFrame(two, System.currentTimeMillis());
        var twoTwo = writeUntilCloseFrame(two, two.getLastTsMillis() + 1);

        var one = archive();
        var oneOne = writeUntilCloseFrame(one, twoTwo.getTsMillis(2));

        var it = iterator(one, two);
        {
            var item = it.next();
            assertItemEquals(item, twoOne);
            assertThat(item, instanceOf(ItemFrame.class));
        }
        {
            var item = it.next();
            assertItemEquals(item, oneOne, twoTwo);
            assertThat(item, instanceOf(ItemIterator.class));
        }

        assertNull(it.next());
        assertNull(it.next());
        close(one, two);
    }

    private MetricArchiveMutable archive() {
        var archive = new MetricArchiveMutable(MetricHeader.defaultValue, format);
        archive.setType(MetricType.DGAUGE);
        return archive;
    }

    private Iterator iterator(MetricArchiveMutable... archives) {
        return Stream.of(archives)
            .map(ArchiveItemIterator::of)
            .collect(collectingAndThen(toList(), MergeIterator::of));
    }
}
