package ru.yandex.stockpile.server.shard.merge;

import java.util.stream.Stream;

import org.junit.Test;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricType;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static junit.framework.TestCase.assertEquals;
import static ru.yandex.solomon.util.CloseableUtils.close;
import static ru.yandex.stockpile.server.shard.merge.Utils.assertArchiveEquals;
import static ru.yandex.stockpile.server.shard.merge.Utils.writeUntilCloseFrame;

/**
 * @author Vladimir Gordiychuk
 */
public class SplitCollectorTest {

    @Test
    public void empty() {
        var archive = archive();
        var result = compress(System.currentTimeMillis(), archive);
        assertEquals(0, result.before.buffer.readableBits());
        assertEquals(0, result.after.buffer.readableBits());
        assertEquals(0, result.before.lastTsMillis);
        assertEquals(0, result.after.lastTsMillis);
        assertEquals(0, result.before.records);
        assertEquals(0, result.after.records);
        close(archive);
    }

    @Test
    public void nothingToAppend() {
        var archive = archive();
        var empty = archive();
        var one = writeUntilCloseFrame(archive);
        var two = writeUntilCloseFrame(archive);

        var result = compress(archive.getFirstTsMillis(), archive);
        assertArchiveEquals(empty, result.before);
        assertArchiveEquals(archive, result.after);
        close(archive, empty);
    }

    @Test
    public void appendAll() {
        var empty = archive();
        var archive = archive();
        var one = writeUntilCloseFrame(archive);
        var two = writeUntilCloseFrame(archive);

        var expected = archive();
        expected.addAll(one);
        expected.addAll(two);
        expected.closeFrame();

        var result = compress(archive.getLastTsMillis() + 1, archive);
        assertArchiveEquals(expected, result.before);
        assertArchiveEquals(empty, result.after);
        close(empty, archive);
    }

    @Test
    public void appendFirstFrame() {
        var archive = archive();
        var one = writeUntilCloseFrame(archive);
        var two = writeUntilCloseFrame(archive);

        var result = compress(two.getTsMillis(0), archive);

        var expectedBefore = archive();
        expectedBefore.addAll(one);
        expectedBefore.closeFrame();

        var expectedAfter = archive();
        expectedAfter.addAll(two);
        expectedAfter.closeFrame();

        assertArchiveEquals(expectedBefore, result.before);
        assertArchiveEquals(expectedAfter, result.after);
        close(archive);
    }

    @Test
    public void appendPartOfFrame() {
        var archive = archive();
        var one = writeUntilCloseFrame(archive);
        var two = writeUntilCloseFrame(archive);

        var result = compress(two.getTsMillis(5), archive);

        var expectedBefore = archive();
        expectedBefore.addAll(one);
        expectedBefore.addAll(two.slice(0, 5));
        expectedBefore.closeFrame();

        var expectedAfter = archive();
        expectedAfter.addAll(two.slice(5, two.length()));
        expectedAfter.closeFrame();

        assertArchiveEquals(expectedBefore, result.before);
        assertArchiveEquals(expectedAfter, result.after);
        close(archive, expectedAfter, expectedBefore);
    }

    private MetricArchiveMutable archive() {
        var archive = new MetricArchiveMutable();
        archive.setType(MetricType.DGAUGE);
        return archive;
    }

    private SplitCollector.Result compress(long tsMillis, MetricArchiveMutable... archives) {
        var it = Stream.of(archives)
            .map(ArchiveItemIterator::of)
            .collect(collectingAndThen(toList(), MergeIterator::of));
        int mask = TsColumn.mask | ValueColumn.mask;
        return SplitCollector.collect(StockpileFormat.CURRENT, MetricType.DGAUGE, mask, 16, tsMillis, it);
    }
}
