package ru.yandex.stockpile.server.shard.merge;

import java.time.Instant;
import java.util.Arrays;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListOrView;
import ru.yandex.solomon.model.timeseries.MergingAggrGraphDataIterator;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assume.assumeTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class Utils {
    public static AggrGraphDataArrayList writeUntilCloseFrame(MetricArchiveMutable archive) {
        long last = archive.getLastTsMillis();
        if (last != 0) {
            return writeUntilCloseFrame(archive, last + 1);
        }
        return writeUntilCloseFrame(archive, System.currentTimeMillis());
    }

    public static AggrGraphDataArrayList writeUntilCloseFrame(MetricArchiveMutable archive, long since) {
        var point = new AggrPoint();
        var random = ThreadLocalRandom.current();

        AggrGraphDataArrayList list = new AggrGraphDataArrayList();
        point.setValue(random.nextDouble());
        point.setTsMillis(since);
        list.addRecord(point);
        archive.addRecord(point);

        while (!archive.closeFrame()) {
            long delay = random.nextLong(TimeUnit.SECONDS.toMillis(1), TimeUnit.HOURS.toMillis(1));
            point.tsMillis = archive.getLastTsMillis() + delay;
            point.setValue(random.nextDouble());
            list.addRecord(point);
            archive.addRecord(point);
            if (archive.frameBytesCount() > (2 << 20)) {
                break;
            }
        }
        return list;
    }

    public static void assertItemEquals(@Nullable Item item, AggrGraphDataArrayList... frames) {
        assertNotNull(item);

        final  AggrGraphDataArrayList expected;
        if (frames.length == 1) {
            expected = frames[0];
        } else {
            var it = Stream.of(frames)
                .map(AggrGraphDataArrayListOrView::iterator)
                .collect(collectingAndThen(toList(), MergingAggrGraphDataIterator::ofCombineAggregate));
            expected = AggrGraphDataArrayList.of(it);
        }

        assertEquals(Instant.ofEpochMilli(expected.getTsMillis(0)), Instant.ofEpochMilli(item.getFirstTsMillis()));
        assertEquals(Instant.ofEpochMilli(expected.getTsMillis(expected.length() - 1)), Instant.ofEpochMilli(item.getLastTsMillis()));
        MetricsMatcher.assertEqualTo(expected, item.iterator());
        assertEquals(expectedBytesSize(StockpileFormat.CURRENT, frames), item.getElapsedBytes(), 128 << 10); // 128KiB Error
    }

    public static void assertArchiveEquals(MetricArchiveMutable expected, CompressResult result) {
        assertEquals(expected.getFormat(), result.format);
        assertEquals(expected.columnSetMask(), result.mask);
        assertEquals(expected.getLastTsMillis(), result.lastTsMillis);
        assertEquals(expected.getCompressedDataRaw(), result.buffer);
        assertEquals(expected.getRecordCount(), result.records);
    }

    public static void assumeMultipleFormats() {
        assumeTrue(Arrays.toString(StockpileFormat.values()), StockpileFormat.MIN != StockpileFormat.MAX);
    }

    public static int expectedBytesSize(StockpileFormat format, AggrGraphDataArrayList... frames) {
        var archive = archive(format);
        for (var frame : frames) {
            archive.addAllNoSortMergeFrom(frame.iterator());
        }
        return archive.getCompressedDataRaw().bytesSize();
    }

    public static MetricArchiveMutable archive(StockpileFormat format) {
        var archive = new MetricArchiveMutable(MetricHeader.defaultValue, format);
        archive.setType(MetricType.DGAUGE);
        return archive;
    }
}
