package ru.yandex.stockpile.server.shard.stat;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.encode.text.MetricTextEncoder;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.stockpile.api.EProjectId;

import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMask;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Vladimir Gordiychuk
 */
public class GlobalUsageStatsTest {

    private GlobalUsageStats global;

    @Before
    public void setUp() {
        global = new GlobalUsageStats();
    }

    @Test
    public void empty() {
        String result = toStringStats();
        assertEquals("", result);
    }

    @Test
    public void read() {
        global.read(EProjectId.SOLOMON, 42, RequestProducer.USER, MetricType.DGAUGE, 100);
        global.read(EProjectId.SOLOMON, 42, RequestProducer.USER, MetricType.DGAUGE, 10);
        global.read(EProjectId.SOLOMON, 42, RequestProducer.USER, MetricType.DGAUGE, 2);
        {
            String result = toStringStats();
            assertThat(result, containsString("RATE stockpile.read.records.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='USER', projectId='SOLOMON'} [112]"));
            assertThat(result, containsString("RATE stockpile.read.sensors.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='USER', projectId='SOLOMON'} [3]"));
        }

        global.read(EProjectId.SOLOMON, 44, RequestProducer.USER, MetricType.DGAUGE, 123);
        {
            String result = toStringStats();
            assertThat(result, containsString("RATE stockpile.read.records.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='USER', projectId='SOLOMON'} [112]"));
            assertThat(result, containsString("RATE stockpile.read.sensors.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='USER', projectId='SOLOMON'} [3]"));

            assertThat(result, containsString("RATE stockpile.read.records.rate{kind='DGAUGE', ownerId='44', " +
                    "producer='USER', projectId='SOLOMON'} [123]"));
            assertThat(result, containsString("RATE stockpile.read.sensors.rate{kind='DGAUGE', ownerId='44', " +
                    "producer='USER', projectId='SOLOMON'} [1]"));
        }

        global.read(EProjectId.SOLOMON, 42, RequestProducer.USER, MetricType.DGAUGE, 15);
        {
            String result = toStringStats();
            assertThat(result, containsString("RATE stockpile.read.records.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='USER', projectId='SOLOMON'} [127]"));
        }
    }

    @Test
    public void lazyRead() {
        int mask = randomMask(MetricType.DGAUGE);
        var source = AggrGraphDataArrayList.of(
                randomPoint(mask),
                randomPoint(mask),
                randomPoint(mask),
                randomPoint(mask));
        var header = new MetricHeader(0, EProjectId.SOLOMON.getNumber(), 42, 0, MetricType.DGAUGE);
        var lazy = global.lazyRead(RequestProducer.USER, header, source);
        {
            String result = toStringStats();
            assertEquals("", result);
        }

        // iterate over points
        {
            var it = lazy.iterator();
            var temp = new AggrPoint();
            int records = 0;
            while (it.next(temp)) {
                records++;
            }
            assertEquals(4, records);
            assertFalse(it.next(temp));
            assertFalse(it.next(temp));

            String result = toStringStats();
            assertThat(result, containsString("RATE stockpile.read.records.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='USER', projectId='SOLOMON'} [4]"));
            assertThat(result, containsString("RATE stockpile.read.sensors.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='USER', projectId='SOLOMON'} [1]"));
        }

        // repeat iteration not bill twice
        {
            var it = lazy.iterator();
            var temp = new AggrPoint();
            int records = 0;
            while (it.next(temp)) {
                records++;
            }
            assertEquals(4, records);

            String result = toStringStats();
            assertThat(result, containsString("RATE stockpile.read.records.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='USER', projectId='SOLOMON'} [4]"));
            assertThat(result, containsString("RATE stockpile.read.sensors.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='USER', projectId='SOLOMON'} [1]"));
        }
    }

    @Test
    public void write() {
        global.write(EProjectId.SOLOMON, 33, MetricType.DGAUGE, 100);
        global.write(EProjectId.SOLOMON, 33, MetricType.DGAUGE, 10);
        global.write(EProjectId.SOLOMON, 33, MetricType.DGAUGE, 5);
        {
            String result = toStringStats();
            assertThat(result, containsString("RATE stockpile.write.records.rate{kind='DGAUGE', ownerId='33', " +
                    "producer='total', projectId='SOLOMON'} [115]"));
            assertThat(result, containsString("RATE stockpile.write.sensors.rate{kind='DGAUGE', ownerId='33', " +
                    "producer='total', projectId='SOLOMON'} [3]"));
            assertThat(result, not(containsString("read")));
        }
    }

    @Test
    public void concurrentUpdate() {
        List<Integer> source = new ArrayList<>(10000);
        for (int index = 0; index < 10000; index++) {
            source.add(42);
        }
        for (int index = 1; index < 40; index++) {
            source.add(index);
        }
        Collections.shuffle(source);

        var r = source.parallelStream()
                .map(ownerId -> {
                    global.write(EProjectId.SOLOMON, ownerId, MetricType.DGAUGE, 1);
                    return null;
                })
                .collect(Collectors.toList());
        assertEquals(source.size(), r.size());

        String result = toStringStats();
        assertThat(result, containsString("RATE stockpile.write.records.rate{kind='DGAUGE', ownerId='42', " +
                "producer='total', projectId='SOLOMON'} [10000]"));
        assertThat(result, containsString("RATE stockpile.write.sensors.rate{kind='DGAUGE', ownerId='42', " +
                "producer='total', projectId='SOLOMON'} [10000]"));

        assertThat(result, containsString("RATE stockpile.write.records.rate{kind='DGAUGE', ownerId='30', " +
                "producer='total', projectId='SOLOMON'} [1]"));
        assertThat(result, containsString("RATE stockpile.write.records.rate{kind='DGAUGE', ownerId='20', " +
                "producer='total', projectId='SOLOMON'} [1]"));
        assertThat(result, containsString("RATE stockpile.write.records.rate{kind='DGAUGE', ownerId='10', " +
                "producer='total', projectId='SOLOMON'} [1]"));
    }

    @Test
    public void writeArchives() {
        MetricArchiveMutable one = new MetricArchiveMutable();
        one.setType(MetricType.DGAUGE);
        one.setOwnerProjectIdEnum(EProjectId.SOLOMON);
        one.setOwnerShardId(42);
        AggrPoint point1 = randomPoint(MetricType.DGAUGE);
        point1.setTsMillis(System.currentTimeMillis() - TimeUnit.SECONDS.toMillis(10));
        one.addRecord(point1);
        AggrPoint point2 = randomPoint(MetricType.DGAUGE);
        point2.setTsMillis((System.currentTimeMillis() - TimeUnit.SECONDS.toMillis(15)));
        one.addRecord(point2);

        global.write(List.of(one));
        {
            String result = toStringStats();
            assertThat(result, containsString("RATE stockpile.write.records.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='total', projectId='SOLOMON'} [2]"));
            assertThat(result, containsString("RATE stockpile.write.sensors.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='total', projectId='SOLOMON'} [1]"));
            assertThat(result, containsString("HIST_RATE stockpile.write.sensors.lag_seconds{kind='total', " +
                    "ownerId='total', producer='total', projectId='total'} " +
                    "[{1: 0, 2: 0, 4: 0, 8: 0, 16: 1, 32: 0, 64: 0, 128: 0, 256: 0, 512: 0, inf: 0}]"));
        }

        global.write(List.of(one, one, one, one));
        {
            String result = toStringStats();
            assertThat(result, containsString("RATE stockpile.write.records.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='total', projectId='SOLOMON'} [10]"));
            assertThat(result, containsString("RATE stockpile.write.sensors.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='total', projectId='SOLOMON'} [5]"));
            assertThat(result, containsString("HIST_RATE stockpile.write.sensors.lag_seconds{kind='total', " +
                    "ownerId='total', producer='total', projectId='total'} " +
                    "[{1: 0, 2: 0, 4: 0, 8: 0, 16: 5, 32: 0, 64: 0, 128: 0, 256: 0, 512: 0, inf: 0}]"));
        }

        MetricArchiveMutable two = new MetricArchiveMutable();
        two.setType(MetricType.DGAUGE);
        two.setOwnerProjectIdEnum(EProjectId.SOLOMON);
        two.setOwnerShardId(43);
        var point3 = randomPoint(MetricType.DGAUGE);
        point3.setTsMillis(System.currentTimeMillis() - TimeUnit.SECONDS.toMillis(3));
        two.addRecord(point3);
        global.write(List.of(one, two, two));
        {
            String result = toStringStats();
            assertThat(result, containsString("RATE stockpile.write.records.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='total', projectId='SOLOMON'} [12]"));
            assertThat(result, containsString("RATE stockpile.write.sensors.rate{kind='DGAUGE', ownerId='42', " +
                    "producer='total', projectId='SOLOMON'} [6]"));

            assertThat(result, containsString("RATE stockpile.write.records.rate{kind='DGAUGE', ownerId='43', " +
                    "producer='total', projectId='SOLOMON'} [2]"));
            assertThat(result, containsString("RATE stockpile.write.sensors.rate{kind='DGAUGE', ownerId='43', " +
                    "producer='total', projectId='SOLOMON'} [2]"));
            assertThat(result, containsString("HIST_RATE stockpile.write.sensors.lag_seconds{kind='total', " +
                    "ownerId='total', producer='total', projectId='total'} [{1: 0, 2: 0, 4: 2, 8: 0, 16: 6, 32: 0, " +
                    "64: 0, 128: 0, 256: 0, 512: 0, inf: 0}]"));
        }
        close(one, two);
    }

    private String toStringStats() {
        return toString(global.stats);
    }

    private String toString(MetricSupplier supplier) {
        StringWriter out = new StringWriter();
        try (MetricTextEncoder e = new MetricTextEncoder(out, true)) {
            supplier.supply(0, e);
        }

        String result = out.toString();
        System.out.println(result);
        return result;
    }
}
