#include "user_metrics.h"

#ifdef _unix_
#include <solomon/libs/cpp/sysmon/porto.h>
#endif

#include <solomon/libs/cpp/sysmon/proc.h>
#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/process_stats/process_stats.h>

#include <library/cpp/http/io/headers.h>
#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/threading/hot_swap/hot_swap.h>

#include <library/cpp/monlib/service/format.h>
#include <library/cpp/monlib/service/monservice.h>
#include <library/cpp/monlib/service/pages/pre_mon_page.h>
#include <library/cpp/monlib/service/pages/version_mon_page.h>
#include <library/cpp/monlib/service/pages/registry_mon_page.h>
#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/monlib/encode/legacy_protobuf/legacy_protobuf.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/encode/text/text.h>
#include <library/cpp/monlib/encode/json/json.h>

#include <util/string/ascii.h>
#include <util/generic/ptr.h>
#include <util/system/event.h>

#include <google/protobuf/text_format.h>
#include <contrib/libs/re2/re2/re2.h>

#include <atomic>
#include <thread>
#include <variant>

using namespace NSolomon;
using namespace NMonitoring;
using namespace NLastGetopt;

namespace {
class TArgumentException: public yexception { };

void FilterNetInterfaces(Sysmon& sysmon, const TString& reString) {
    re2::RE2 re{reString};
    Y_ENSURE_EX(re.ok(), TArgumentException() << "Unable to compile regex " << reString << ": " << re.error());

    auto& net = *sysmon.MutableNet();
    TVector<size_t> toDelete;

    for (auto i = 0u; i < net.IfsSize(); ++i) {
        auto&& intf = net.GetIfs(i);
        if (RE2::FullMatch({intf.GetDev()}, re)) {
            continue;
        }

        toDelete.push_back(i);
    }

    for (auto it = toDelete.rbegin(); it != toDelete.rend(); ++it) {
        auto idx = *it;
        net.MutableIfs()->erase(net.MutableIfs()->begin() + idx);
    }
}

void FilterMountPoints(Sysmon& sysmon, const TString& reString) {
    re2::RE2 re{reString};
    Y_ENSURE_EX(re.ok(), TArgumentException() << "Unable to compile regex " << reString << ": " << re.error());

    auto* fs = sysmon.MutableFilesystem();
    TVector<size_t> toDelete;

    auto* mountPoints = fs->MutablePoints();
    for (int i = 0; i < mountPoints->size(); ++i) {
        auto& mountPoint = mountPoints->Get(i);
        if (!RE2::FullMatch({mountPoint.GetDev()}, re)) {
            toDelete.push_back(i);
        }
    }

    for (auto it = toDelete.rbegin(); it != toDelete.rend(); ++it) {
        mountPoints->erase(mountPoints->begin() + *it);
    }
}

using TMetricRegistryPtr = TAtomicSharedPtr<TMetricRegistry>;

class TBackgroundCollector {
    static constexpr auto REFRESH_INTERVAL = TDuration::Seconds(15);

    using TResult = NSolomon::TErrorOr<Sysmon, NSolomon::TGenericError, true>;
    struct TContainer: TAtomicRefCount<TContainer> {
        TContainer(TResult&& r)
            : Result{std::move(r)}
        {
        }

        const TResult Result;
    };

    using TResultPtr = TIntrusivePtr<TContainer>;

public:
    explicit TBackgroundCollector(TMetricRegistryPtr registry)
        : Sysmon_{new TContainer{TResult::FromError("not initialized")}}
        , Registry_{registry}
    {
    }

    ~TBackgroundCollector() {
        Stop();
    }

    void Start() {
        Thr_ = MakeHolder<std::thread>([this] {
            Work();
        });
    }

    void Stop() {
        bool expected{false};
        if (Stopped_.compare_exchange_strong(expected, true)) {
            Thr_->join();
        }
    }

    void Work() noexcept {
        while (!Stopped_.load()) {
            if (ShouldRefresh()) {
                DoRefresh();
                LastRefresh_ = TInstant::Now();
                Evt_.Signal();
            }

            Sleep(TDuration::MilliSeconds(200));
        };
    }

    TResultPtr Get() const {
        return Sysmon_.AtomicLoad();
    }

    void WaitInitialized() {
        Evt_.WaitI();
    }

private:
    bool ShouldRefresh() const {
        return (TInstant::Now() - LastRefresh_) > REFRESH_INTERVAL;
    }

    void DoRefresh() noexcept {
        try {
            auto&& sysmon = FillSysmon();
            Sysmon_.AtomicStore(MakeIntrusive<TContainer>(TResult::FromValue(std::move(sysmon))));
        } catch (...) {
            Cerr << "Error while filling sysmon: " << CurrentExceptionMessage();
            Sysmon_.AtomicStore(MakeIntrusive<TContainer>(TResult::FromError(CurrentExceptionMessage())));
        }

        auto result = ProcessStatProvider_->GetSelfStats();
        if (result.Success()) {
            auto stats = result.Extract();
            WriteStats(stats);
            LastStats_ = std::move(stats);
        }
    }

    void WriteStats(const NSolomon::TProcessStats& stats) {
        Registry_->IntGauge({{"sensor", "process.memRssBytes"}})->Set(stats.MemRss);
        Registry_->IntGauge({{"sensor", "process.memLibBytes"}})->Set(stats.MemLib);
        Registry_->IntGauge({{"sensor", "process.memSwapBytes"}})->Set(stats.MemSwap);
        Registry_->IntGauge({{"sensor", "process.threadCount"}})->Set(stats.ThreadCount);
        Registry_->Rate({{"sensor", "process.majorPageFaults"}})->Add(stats.MajorPageFaults - LastStats_.MajorPageFaults);
        Registry_->Rate({{"sensor", "process.cpuUserMillis"}})->Add(stats.CpuUser - LastStats_.CpuUser);
        Registry_->Rate({{"sensor", "process.cpuSystemMillis"}})->Add(stats.CpuSystem - LastStats_.CpuSystem);
    }

private:
    THotSwap<TContainer> Sysmon_;
    NSolomon::IProcessStatProviderPtr ProcessStatProvider_{NSolomon::CreateProcessStatProvider()};
    TAtomicSharedPtr<TMetricRegistry> Registry_;
    NSolomon::TProcessStats LastStats_;

    std::atomic<bool> Stopped_{false};
    TInstant LastRefresh_{TInstant::Zero()};
    THolder<std::thread> Thr_;
    TAutoEvent Evt_;
};

const TString SPACK_CONTENT_TYPE {NMonitoring::NFormatContenType::SPACK};
const TString PB_CONTENT_TYPE {NMonitoring::NFormatContenType::PROTOBUF};
const TString TEXT_CONTENT_TYPE {NMonitoring::NFormatContenType::TEXT};
const TString JSON_CONTENT_TYPE {NMonitoring::NFormatContenType::JSON};

struct TSysmonSettings {
    bool Containerized = false;
    bool FetchPorto = false;
    TVector<TString> PortoMappings;
    TString PortoVolumeRegex;
    TString PortoSlotName;
    TString NetIntfRegex;
    TString MountPointRegex;
};

#ifdef _unix_
Sysmon MakeSysmon(TSysmonSettings settings, const TBackgroundCollector& collector) {
    Sysmon sysmon;

    // if no mappings -- just fetch all
    try {
        if (settings.FetchPorto) {
            auto&& portoStats = GetPortoStats(settings.PortoMappings);

            if (portoStats.ContainersSize() > 0) {
                sysmon.MutablePorto()->Swap(&portoStats);
            }
        }
    } catch (...) {
        Cerr << ::CurrentExceptionMessage();
    }

    // if containerized there's no sense in gathering host system metrics
    if (settings.Containerized) {
        try {
            auto portoStats = GetPortoSelf(settings.PortoVolumeRegex);
            sysmon.MutablePorto()->MergeFrom(portoStats);
        } catch (...) {
            Cerr << ::CurrentExceptionMessage();
        }
        try {
            auto portoSlotStats = GetPortoSlot(settings.PortoSlotName);
            sysmon.MutablePorto()->MergeFrom(portoSlotStats);
        } catch (...) {
            Cerr << ::CurrentExceptionMessage();
        }
    } else {
        auto&& result = collector.Get()->Result;
        Y_ENSURE(result.Success(),  result.Error().Message());
        sysmon.MergeFrom(result.Value());
    }

    if (!settings.NetIntfRegex.empty()) {
        FilterNetInterfaces(sysmon, settings.NetIntfRegex);
    }

    if (!settings.MountPointRegex.empty()) {
        FilterMountPoints(sysmon, settings.MountPointRegex);
    }

    return sysmon;
}
#else
Sysmon MakeSysmon(TSysmonSettings settings, const TBackgroundCollector& collector) {
    ythrow yexception() << "Sysmon is unavalable in Windows";
}
#endif

TSysmonSettings SettingsFromParams(const TCgiParameters& params) {
    TSysmonSettings result;

    TCgiParameters::const_iterator it, begin, end;
    std::tie(begin, end) = params.equal_range("porto.containerName");
    for (it = begin; it != end; ++it) {
        result.PortoMappings.push_back(it->second);
    }

    auto hasPorto = [&params] {
        auto it = params.Find("porto");
        return it != std::end(params) && it->second == "1";
    };

    if (!result.PortoMappings.empty() || hasPorto()) {
        result.FetchPorto = true;
    }

    if ((it = params.Find("porto.containerized")) != std::end(params) && it->second == "1") {
        result.Containerized = true;
    }

    if ((it = params.Find("porto.volumes")) != std::end(params)) {
        result.PortoVolumeRegex = it->second;
    }

    if ((it = params.Find("porto.slotName")) != std::end(params)) {
        result.PortoSlotName = it->second;
    } else {
        // Assuming that we are in 'iss_hook_start' container by default
        // https://wiki.yandex-team.ru/runtime-cloud/nanny/container-hierarchy/
        result.PortoSlotName = "self/../..";
    }

    if ((it = params.Find("net.intf")) != std::end(params)) {
        result.NetIntfRegex = it->second;
    }

    if ((it = params.Find("mount")) != std::end(params)) {
        result.MountPointRegex = it->second;
    }

    return result;
}

} // namespace

static TString GetParam(const TCgiParameters& params, const TString& paramName, const TString& defaultValue = "") {
    TCgiParameters::const_iterator it = params.Find(paramName);

    if (it == params.end() || it->second.empty()) {
        return defaultValue;
    }

    return it->second;
}

static void WriteStatusAndHeaders(IMonHttpRequest& request, size_t code, const TString& serverMessage, const THttpHeaders& headers) {
    request.Output() << "HTTP/1.1 " << code << " " << serverMessage << "\r\n";
    headers.OutTo(&request.Output());
    request.Output() << "\r\n";
}

static void SendResponse(IMonHttpRequest& request, size_t code, const TString& serverMessage, const TString& contentType, const TString& reply) {
    request.Output() << "HTTP/1.1 " << code << " " << serverMessage << "\r\nContent-Type: " << contentType << "\r\nConnection: Close\r\n\r\n";
    request.Output() << reply;
}

static void SendResponse_500(IMonHttpRequest& request, const TString& reply) {
    SendResponse(request, 500, "Internal Server Error", "text/plain", reply);
}

static void SendResponse_200(IMonHttpRequest& request, const TString& reply) {
    SendResponse(request, 200, "Ok", "text/plain", reply);
}

struct TInfoBinPage: public IMonPage {
    explicit TInfoBinPage(const TBackgroundCollector& collector)
        : IMonPage("pbbin", "system protobuf bin")
        , Collector_{collector}
    {
    }

    void Output(IMonHttpRequest& request) override {
        Sysmon data;

        try {
            data = MakeSysmon(SettingsFromParams(request.GetParams()), Collector_);
        } catch (const TArgumentException& e) {
            SendResponse(request, 400, "Bad Request", "text/plain", ::CurrentExceptionMessage());
            return;
        } catch (...) {
            SendResponse_500(request, ::CurrentExceptionMessage());
            return;
        }

        NMonitoring::EFormat format{NMonitoring::EFormat::UNKNOWN};

        THttpHeaders headers;
        headers.AddHeader("Connection", "Close");

        try {
            format = ParseFormat(request);
        } catch (...) {
            SendResponse(request, 400, "Bad Request", "text/plain", ::CurrentExceptionMessage());
            return;
        }

        if (format == EFormat::UNKNOWN) {
            headers.AddHeader("Content-Type", PB_CONTENT_TYPE);
            WriteStatusAndHeaders(request, 200, "OK", headers);
            data.SerializePartialToArcadiaStream(&request.Output());
        } else if (format == EFormat::SPACK) {
            const auto compression = ParseCompression(request);

            headers.AddHeader("Content-Type", SPACK_CONTENT_TYPE);
            headers.AddHeader("Content-Encoding", ContentEncodingByCompression(compression));

            WriteStatusAndHeaders(request, 200, "OK", headers);

            auto spackEncoder = NMonitoring::EncoderSpackV1(&request.Output(),
                NMonitoring::ETimePrecision::SECONDS,
                compression);

            NMonitoring::DecodeLegacyProto(data, spackEncoder.Get());
        } else if (format == EFormat::TEXT) {
            headers.AddHeader("Content-Type", TEXT_CONTENT_TYPE);
            WriteStatusAndHeaders(request, 200, "OK", headers);
            auto textEncoder = NMonitoring::EncoderText(&request.Output(), true);

            NMonitoring::DecodeLegacyProto(data, textEncoder.Get());
        } else if (format == EFormat::JSON) {
            headers.AddHeader("Content-Type", JSON_CONTENT_TYPE);
            WriteStatusAndHeaders(request, 200, "OK", headers);
            auto textEncoder = NMonitoring::EncoderJson(&request.Output(), true);

            NMonitoring::DecodeLegacyProto(data, textEncoder.Get());
        }
    }

private:
    const TBackgroundCollector& Collector_;
};

struct TInfoTxtPage: public TPreMonPage {
    explicit TInfoTxtPage(const TBackgroundCollector& collector)
        : TPreMonPage("pb", "system protobuf txt")
        , Collector_{collector}
    {
    }

    void OutputText(IOutputStream& out, NMonitoring::IMonHttpRequest& request) override {
        TString output;
        NProtoBuf::TextFormat::PrintToString(
            MakeSysmon(SettingsFromParams(request.GetParams()), Collector_),
            &output
        );

        out << output;
    }

private:
    const TBackgroundCollector& Collector_;
};

struct TUserShowPage: public TPreMonPage {
    TUserShowPage(TUserMetrics& metrics)
        : TPreMonPage("service-show.json", "show user metrics data")
        , Metrics(metrics)
    {
    }

    void OutputText(IOutputStream& out, NMonitoring::IMonHttpRequest&) override {
        for (const auto& services: Metrics.Services) {
            const TString& serviceName = services.first;
            out << "Service: " << serviceName << Endl;
            out << Metrics.GetMetricData(serviceName) << Endl << Endl;
        }
    }

private:
    TUserMetrics& Metrics;
};

struct TUserPullPage: public IMonPage {
    TUserPullPage(TUserMetrics& metrics)
        : IMonPage("service.json", "pull user metrics data")
        , Metrics(metrics)
    {
    }

    void Output(IMonHttpRequest& request) override {
        const TCgiParameters& params = request.GetParams();
        const TString& serviceName = GetParam(params, "service", "default");

        SendResponse_200(request, Metrics.GetAndFlushMetricData(serviceName));
    }

private:
    TUserMetrics& Metrics;
};

struct TUserUpdatePage: public IMonPage {
    TUserUpdatePage(TUserMetrics& metrics)
        : IMonPage("update", "update user metrics data")
        , Metrics(metrics)
    {
    }

    void Output(IMonHttpRequest& request) override {
        const TCgiParameters& params = request.GetParams();

        TString serviceName = GetParam(params, "service", "default");
        TString metricName = GetParam(params, "sensor");
        TString valueStr = GetParam(params, "value");
        TString ttlStr = GetParam(params, "expire", "300");
        TString modeName = GetParam(params, "mode");
        TString valueOp = GetParam(params, "op", "set");

        TUserMetricData::TLabels extraLabels;

        for (TCgiParameters::const_iterator it = params.begin(); it != params.end(); ++it) {
            const TString& name = it->first;
            if (name.find("label.") == 0) {
                TString labelName = name.substr(6);
                if (!labelName.empty()) {
                    extraLabels[labelName] = it->second;
                }
            }
        }

        if (params.empty() || metricName.empty() || valueStr.empty()) {
            SendResponse_200(request, "Usage: https://wiki.yandex-team.ru/solomon/sysmond/api");
            return;
        }

        try {
            Metrics.AddMetricData(serviceName, metricName, (double)FromString(valueStr), FromString(ttlStr), modeName, valueOp, extraLabels);
        } catch (const yexception& e) {
            SendResponse_500(request, TString("Unable to process metric data") + e.what());
            return;
        }

        SendResponse_200(request, "success");
    }

private:
    TUserMetrics& Metrics;
};

int main(int argc, char** argv) {
    unsigned port;

    TOpts opts;
    opts.AddLongOption('p', "port", "http port")
        .Required()
        .RequiredArgument("PORT")
        .StoreResult(&port);
    opts.SetFreeArgsMax(0);

    TOptsParseResult(&opts, argc, argv);

    TUserMetrics metrics;

    TMetricRegistryPtr registry{new TMetricRegistry};

    TBackgroundCollector collector{registry};
    collector.Start();
    TMonService2 monService(port);

    monService.Register(new TInfoTxtPage{collector});
    monService.Register(new TInfoBinPage{collector});
    monService.Register(new TUserShowPage(metrics));
    monService.Register(new TUserPullPage(metrics));
    monService.Register(new TUserUpdatePage(metrics));
    monService.Register(new TVersionMonPage);
    monService.Register(new TMetricRegistryPage{"self", "sysmond process metrics", registry});

    collector.WaitInitialized();
    bool ok = monService.Start();
    Y_VERIFY(ok);

    for (;;) {
        Sleep(TDuration::Hours(1));
    }

    return 0;
}
