#include "user_metrics.h"

#include <library/cpp/monlib/deprecated/json/writer.h>

#include <util/generic/map.h>
#include <util/string/vector.h>

namespace NSolomon {
    void TUserMetrics::AddMetricData(const TString& serviceName, const TString& metricName, double value, size_t ttl,
                                     const TString& modeName, const TString& valueOp,
                                     const TUserMetricData::TLabels& extraLabels, ui32 timestamp)
    {
        TGuard<TMutex> guard(Mutex);

        TUserMetricData& metric = Services[serviceName][metricName];
        if (valueOp == "set") {
            metric.Value = value;
        } else if (valueOp == "add") {
            metric.Value += value;
        } else {
            throw yexception() << ": op parameter should be one of ['set', 'add']";
        }
        metric = TUserMetricData(metric.Value, timestamp, ttl);

        for (const auto& label: extraLabels) {
            metric.Labels[label.first] = label.second;
        }

        metric.Labels["sensor"] = metricName;

        if (modeName == "deriv") {
            metric.Derived = true;
        }
    }

    TString TUserMetrics::GetJsonRepresentation(const TString& serviceName, bool format) const {
        TGuard<TMutex> guard(Mutex);

        TString result;
        TStringOutput so(result);

        Y_UNUSED(format);
        NMonitoring::TDeprecatedJsonWriter writer(&so);

        TServiceMap::const_iterator serviceIterator = Services.find(serviceName);

        if (serviceIterator == Services.end()) {
            return "";
        }

        const TMetricMap& metrics = serviceIterator->second;

        writer.OpenDocument();
        writer.OpenMetrics();

        for (TMetricMap::const_iterator it = metrics.begin(); it != metrics.end(); ++it) {
            const TUserMetricData& metric = it->second;
            writer.OpenMetric();
            writer.OpenLabels();
            const TUserMetricData::TLabels& labels = metric.Labels;
            for (TUserMetricData::TLabels::const_iterator et = labels.begin(); et != labels.end(); ++et) {
                writer.WriteLabel(et->first, et->second);
            }
            writer.CloseLabels();
            if (metric.Derived) {
                writer.WriteModeDeriv();
            }
            writer.WriteDoubleValue(metric.Value);
            writer.CloseMetric();
        }

        writer.CloseMetrics();
        writer.CloseDocument();

        return result;
    }

    TString TUserMetrics::GetMetricData(const TString& serviceName) const {
        return GetJsonRepresentation(serviceName, true);
    }

    TString TUserMetrics::GetAndFlushMetricData(const TString& serviceName) {
        TGuard<TMutex> guard(Mutex);

        TMetricMap& metrics = Services[serviceName];

        for (TMetricMap::iterator it = metrics.begin(); it != metrics.end();) {
            if (TInstant::Now().TimeT() < it->second.Timestamp + it->second.TTL) {
                ++it;
            } else {
                metrics.erase(it++);
            }
        }

        return GetJsonRepresentation(serviceName);
    }

} //namespace NSolomon

