package main

import (
	"flag"
	"log"
	"os"
	"path/filepath"
	"sort"
	"time"

	"golang.org/x/sys/unix"
)

// =================================================================================================

var CoreDirs = []string{
	"/coredumps/",
}

var DefaultCoreMaxLifetimeHours = 14 * 24
var DefaultCoreCleanerFsMaxPercent = 70.0
var DefaultCoreFreshMinCount = 5

// =================================================================================================

type File struct {
	Path string
	Info os.FileInfo
}

func findAndCleanCores(dirPaths []string,
	CoreMaxLifetime time.Duration,
	CoreCleanerFsMaxPercent float64,
	CoreFreshMinCount int) {

	oldestFileTime := time.Now().Add(-CoreMaxLifetime)
	fsCores := map[uint64][]File{}
	fsSpaceTotal := map[uint64]int64{}
	fsSpaceFree := map[uint64]int64{}
	fsDirs := map[uint64][]string{}

	for _, dir := range dirPaths {
		var stat unix.Statfs_t
		if err := unix.Statfs(dir, &stat); err != nil {
			log.Fatalf("cannot get stats of %s: %v", dir, err)
		}

		fsID := uint64(stat.Fsid.Val[1])<<32 + uint64(uint32(stat.Fsid.Val[0]))
		fsSpaceFree[fsID] = stat.Bsize * int64(stat.Bfree)
		fsSpaceTotal[fsID] = stat.Frsize * int64(stat.Blocks)

		cores := []File{}
		err := filepath.Walk(dir, func(path string, info os.FileInfo, err error) error {
			if err == nil && info.Mode().IsRegular() {
				cores = append(cores, File{path, info})
			}
			return nil
		})
		if err != nil {
			log.Fatal(err)
		}
		// sort oldest first
		sort.Slice(cores, func(i, j int) bool {
			return cores[i].Info.ModTime().Unix() < cores[j].Info.ModTime().Unix()
		})
		fsCores[fsID] = cores
		fsDirs[fsID] = append(fsDirs[fsID], dir)
	}
	for id, cores := range fsCores {
		dirs := fsDirs[id]
		coresSizeRemoved := int64(0)
		fsTotal := fsSpaceTotal[id]
		fsFree := fsSpaceFree[id]
		maxCoresSizeToRemove := fsTotal - fsFree - int64(float64(fsTotal)*CoreCleanerFsMaxPercent/100.0)
		coresCountRemoved := 0
		maxCoresCountToRemove := len(cores) - CoreFreshMinCount

		if maxCoresSizeToRemove < 0 {
			maxCoresSizeToRemove = 0
		}
		log.Printf("Start cleaning %v (totalSpace=%.1fMb freeSpace=%.1fMb toRemove=%.1fMb)",
			dirs,
			float64(fsTotal)/1024/1024,
			float64(fsFree)/1024/1024,
			float64(maxCoresSizeToRemove)/1024/1024,
		)
		for _, core := range cores {
			ts := core.Info.ModTime().Format("2006-01-02_15:04:05.000")
			size := core.Info.Size()
			if core.Info.ModTime().Before(oldestFileTime) {
				log.Printf("Removing too old file (ts=%s size=%.1fMb) %s", ts, float64(size)/1024/1024, core.Path)
			} else if coresCountRemoved >= maxCoresCountToRemove {
				break
			} else if coresSizeRemoved >= maxCoresSizeToRemove {
				break
			} else {
				log.Printf("Removing file (ts=%s size=%.1fMb) %s", ts, float64(size)/1024/1024, core.Path)
			}

			if err := os.Remove(core.Path); err != nil {
				log.Printf("Failed to remove %s: %v", core.Path, err)
			} else {
				coresSizeRemoved += size
				coresCountRemoved += 1
			}
		}
		log.Printf("Done cleaning %v: %d/%d files, %.fGb removed",
			dirs,
			coresCountRemoved,
			len(cores),
			float64(coresSizeRemoved)/1024/1024/1024,
		)
	}
}

// =================================================================================================

func main() {
	var CoreMaxLifetimeHours int
	var CoreCleanerFsMaxPercent float64
	var CoreFreshMinCount int

	flag.IntVar(&CoreMaxLifetimeHours, "L", DefaultCoreMaxLifetimeHours, "maximum core lifetime, hours")
	flag.Float64Var(&CoreCleanerFsMaxPercent, "p", DefaultCoreCleanerFsMaxPercent, "start cleaning when fs has over this much percent")
	flag.IntVar(&CoreFreshMinCount, "c", DefaultCoreFreshMinCount, "number of cores to leave")
	flag.Parse()

	CoreMaxLifetime := time.Duration(CoreMaxLifetimeHours) * time.Hour

	log.SetFlags(log.Ldate | log.Ltime | log.Lmicroseconds)

	findAndCleanCores(CoreDirs, CoreMaxLifetime, CoreCleanerFsMaxPercent, CoreFreshMinCount)
}
