#include "consumers.h"

static TString MetaStr(const TMetricMeta& meta) {
    return TStringBuilder() << "(" << meta.Name << ") " << meta.Id.ShardId << "|" << meta.Id.LocalId << "|" << meta.Labels;
}

static TString ConstructError(TStringBuf error, TStringBuf body, const TMetricMeta& meta) {
    TStringBuilder sb;
    sb << "ERROR {\n";
    sb << "    TYPE: " << error << '\n';
    sb << "    BODY: " << body << '\n';
    sb << "    META: " << MetaStr(meta) << "\n}\n";

    return sb;
}

static void Trace(IOutputStream& s, TActorMetaData& meta, TStringBuf mes) {
    s << TStringBuf(TStringBuilder() << MetaStr(*MetricMetaCast(meta)) << ": " << mes << '\n');
}

class TComparisonResolveDcConsumer: public IComparisonResolveDcConsumer {
public:
    TComparisonResolveDcConsumer(TDcFilter filter, IProgressMonitorPtr monitor, IOutputStream& s, IOutputStream& tr)
            : Filter_(std::move(filter))
            , Monitor_(std::move(monitor))
            , Stream_(s)
            , Trace_(tr)
    {
    }

    size_t Processed() const override {
        return Counter_.load();
    }

    size_t Errors() const override {
        return ErrC_.load();
    }

    size_t Dropped() const override {
        return Dropped_.load();
    }

private:
    void OnResolveGroupDc(TDcResolveResult dc, TStringBuf group, TActorMetaData meta) override {
        Y_VERIFY(MetricMetaCast(meta), "data lose");
        if (dc.Dc == NSolomon::EDc::UNKNOWN) {
            OnGroupError("unknown error", group, std::move(meta));
            return;
        }

        Monitor_->IncProgress(1u);

        if (!Filter_.ContainFlag(dc.Dc)) {
            Dropped_.fetch_add(1u);
            Trace(Trace_, meta, TStringBuilder() << "[dc filter] drop metric from dc " << DcToStr(dc.Dc));
            return;
        }

        Trace(Trace_, meta, TStringBuilder() << "[dc filter] resolved dc " << DcToStr(dc.Dc));
        Counter_.fetch_add(1u);

        Y_ENSURE(Callback_);
        Callback_(std::move(meta));
    }

    void OnResolveHostDc(TDcResolveResult dc, TStringBuf host, TActorMetaData meta) override {
        Y_VERIFY(MetricMetaCast(meta), "data lose");

        if (dc.Dc == NSolomon::EDc::UNKNOWN) {
            if (dc.Status == EDcResolveStatus::TIMEOUT) {
                OnHostError("request timed out", host, std::move(meta));
            } else if (dc.Status == EDcResolveStatus::NOT_FOUND) {
                OnHostError("host not found", host, std::move(meta));
            } else {
                OnHostError("unknown error", host, std::move(meta));
            }

            return;
        }

        Monitor_->IncProgress(1u);

        if (!Filter_.ContainFlag(dc.Dc)) {
            Trace(Trace_, meta, TStringBuilder() << "[dc filter] drop metric from dc " << DcToStr(dc.Dc));
            Dropped_.fetch_add(1u);
            return;
        }

        Trace(Trace_, meta, TStringBuilder() << "[dc filter] resolved dc " << DcToStr(dc.Dc));

        Counter_.fetch_add(1u);

        Y_ENSURE(Callback_);
        Callback_(std::move(meta));
    }

    void OnHostError(TString error, TStringBuf, TActorMetaData meta) override {
        Y_VERIFY(MetricMetaCast(meta), "data lose");

        Monitor_->IncProgress(1u);
        ErrC_.fetch_add(1u);

        Stream_ << ConstructError("failed to resolve host", error, *MetricMetaCast(meta));
        Trace(Trace_, meta, "[dc filter] error occurred");
    }

    void OnGroupError(TString error, TStringBuf, TActorMetaData meta) override {
        Y_VERIFY(MetricMetaCast(meta), "data lose");

        Monitor_->IncProgress(1u);
        ErrC_.fetch_add(1u);

        Stream_ << ConstructError("failed to resolve host", error, *MetricMetaCast(meta));
        Trace(Trace_, meta, "[dc filter] error occurred");
    }

private:
    TDcFilter Filter_;

    IProgressMonitorPtr Monitor_;

    std::atomic<size_t> Counter_{0};
    std::atomic<size_t> ErrC_{0};
    std::atomic<size_t> Dropped_{0};

    IOutputStream& Stream_;
    IOutputStream& Trace_;
};

class TComparisonReadMetaConsumer: public IComparisonReadMetricsMetaConsumer {
public:
    explicit TComparisonReadMetaConsumer(IProgressMonitorPtr monitor, IOutputStream& s, IOutputStream& tr)
            : Monitor_(std::move(monitor))
            , Stream_(s)
            , Trace_(tr)
    {
    }

    using TFunc = std::function<void(TActorMetaData meta)>;

    size_t Processed() const override {
        return Counter_.load();
    }

    size_t Errors() const override {
        return ErrC_.load();
    }

    size_t Dropped() const override {
        return Dropped_.load();
    }

private:
    void OnMetricMeta(TStockpileIds, TInstant lastUpdate, TActorMetaData meta) override {
        Y_VERIFY(MetricMetaCast(meta), "data lose");
        Monitor_->IncProgress(1u);

        if (!CheckAlive(lastUpdate)) {
            Dropped_.fetch_add(1u);

            Trace(
                    Trace_,
                    meta,
                    TStringBuilder() << "[read metrics meta] drop dead metric; last update (" << lastUpdate << ")");
            return;
        }

        Trace(
                Trace_,
                meta,
                TStringBuilder() << "[read metrics meta] last update (" << lastUpdate << ")");

        Counter_.fetch_add(1u);

        Y_ENSURE(Callback_);
        Callback_(std::move(meta));
    }

    void OnError(TString error, TStockpileIds, TActorMetaData meta) override {
        Y_VERIFY(MetricMetaCast(meta), "data lose");
        Monitor_->IncProgress(1u);
        ErrC_.fetch_add(1u);

        Trace(Trace_, meta, "[read metrics meta] error occurred");
        Stream_ << ConstructError("failed to read metric meta", error, *MetricMetaCast(meta));
    }

    static bool CheckAlive(TInstant ts) {
        return (TInstant::Now() - ts) < TDuration::Hours(12u);
    }

private:
    IProgressMonitorPtr Monitor_;

    std::atomic<size_t> Counter_{0};
    std::atomic<size_t> ErrC_{0};
    std::atomic<size_t> Dropped_{0};

    IOutputStream& Stream_;
    IOutputStream& Trace_;
};

class TComparisonDownloadMetricsConsumer: public IComparisonDownloadMetricsConsumer {
public:
    explicit TComparisonDownloadMetricsConsumer(IProgressMonitorPtr monitor, IOutputStream& s, IOutputStream& tr)
            : Monitor_(std::move(monitor))
            , Stream_(s)
            , Trace_(tr)
    {
    }

    using TFunc = std::function<void(TSeriesAndId&&, TActorMetaData)>;

    size_t Processed() const override {
        return Counter_.load();
    }

    size_t Errors() const override {
        return ErrC_.load();
    }

    size_t Dropped() const override {
        return Dropped_.load();
    }

private:
    void OnMetric(TSeriesAndId series, TActorMetaData meta) override {
        Y_VERIFY(MetricMetaCast(meta), "data lose");
        Monitor_->IncProgress(1u);

        Counter_.fetch_add(1u);
        Trace(
                Trace_,
                meta,
                TStringBuilder() << "[download] downloaded ts (size = " << series.Series->PointCount()
                             << ", type = " << NMonitoring::MetricTypeToStr(series.Series->Type()) << ")");

        Y_ENSURE(Callback_);
        Callback_(std::move(series), std::move(meta));
    }

    void OnError(TString error, TStockpileIds, TActorMetaData meta) override {
        Y_VERIFY(MetricMetaCast(meta), "data lose");
        Monitor_->IncProgress(1u);
        ErrC_.fetch_add(1u);

        Trace(Trace_, meta, "[download] error occurred");
        Stream_ << ConstructError("failed to download metric", error, *MetricMetaCast(meta));
    }

private:
    IProgressMonitorPtr Monitor_;

    std::atomic<size_t> Counter_{0};
    std::atomic<size_t> ErrC_{0};
    std::atomic<size_t> Dropped_{0};

    IOutputStream& Stream_;
    IOutputStream& Trace_;
};

IComparisonResolveDcConsumerPtr CreateComparisonResolveDcConsumer(
        TDcFilter filter,
        IProgressMonitorPtr monitor,
        IOutputStream& s,
        IOutputStream& tr) {
    return std::make_shared<TComparisonResolveDcConsumer>(std::move(filter), std::move(monitor), s, tr);
}

IComparisonReadMetricsMetaConsumerPtr CreateComparisonReadMetricsMetaConsumer(
        IProgressMonitorPtr monitor,
        IOutputStream& s,
        IOutputStream& tr) {
    return std::make_shared<TComparisonReadMetaConsumer>(std::move(monitor), s, tr);
}

IComparisonDownloadMetricsConsumerPtr CreateComparisonDownloadMetricsConsumer(
        IProgressMonitorPtr monitor,
        IOutputStream& s,
        IOutputStream& tr) {
    return std::make_shared<TComparisonDownloadMetricsConsumer>(std::move(monitor), s, tr);
}
