#include "consumers.h"
#include "proto_decoder.h"

#include <solomon/tools/data-comparison/lib/actors/actors.h>
#include <solomon/tools/data-comparison/lib/util/io.h>

#include <solomon/libs/cpp/config_includes/config_includes.h>
#include <solomon/libs/cpp/labels/known_keys.h>

#include <solomon/tools/data-comparison/lib/diff/diff.h>

#include <library/cpp/threading/future/async.h>
#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/getoptpb/getoptpb.h>

#include <util/stream/file.h>

#include <util/thread/pool.h>

#include <memory>
#include <sstream>
#include <regex>
#include <cctype>
#include <iomanip>

using NMonitoring::TLabels;

TString EncodeUri(TString string) {
    std::ostringstream escaped;
    escaped.fill('0');
    escaped << std::hex;

    for (char c: string) {
        if (std::isalnum(c) || c == '-' || c == '_' || c == '.' || c == '~') {
            escaped << c;
            continue;
        }

        // Any other characters are percent-encoded
        escaped << std::uppercase;
        escaped << '%' << std::setw(2) << int((unsigned char) c);
        escaped << std::nouppercase;
    }

    return TString{escaped.str().data(), escaped.str().size()};
}

TString LabelsShortStr(const NMonitoring::TLabels& labels) {
    TStringBuilder sb;
    for (size_t i = 0; i < labels.size(); ++i) {
        sb << labels[i].Name() << "=" << labels[i].Value();
        if (i + 1 != labels.size()) {
            sb << ",";
        }
    }

    return sb;
}

struct TLinkMakerOpts {
    ECluster ActualCluster;
    TString ActualPN;
    TString ActualClusterLabel;

    ECluster ExpectedCluster;
    TString ExpectedPN;
    TString ExpectedClusterLabel;

    TInstant From;
    TInstant To;

    bool Initialized() const {
        return ActualPN && ActualClusterLabel && ExpectedPN && ExpectedClusterLabel;
    }
};

struct TLinkMaker: public ILinkMaker {
    TLinkMaker(TLinkMakerOpts opts)
        : Opts(std::move(opts))
    {
    }

    TString MakeLink(NMonitoring::TLabels labels, ECluster cluster) const override {
        TString begin = "https://solomon.yandex-team.ru/admin/projects/";
        if (cluster == ECluster::PRE) {
            begin = "https://solomon-pre.yandex-team.ru/admin/projects/";
        }

        if (cluster == Opts.ActualCluster) {
            labels.Add(NSolomon::NLabels::LABEL_CLUSTER, Opts.ActualClusterLabel);
            begin += Opts.ActualPN;
        } else if (cluster == Opts.ExpectedCluster) {
            labels.Add(NSolomon::NLabels::LABEL_CLUSTER, Opts.ExpectedClusterLabel);
            begin += Opts.ExpectedPN;
        } else {
            return {};
        }

        labels.Add(NSolomon::NLabels::LABEL_SERVICE, "yasm");

        begin += "/stockpileMetric?labels=";
        begin += EncodeUri(LabelsShortStr(labels));

        begin += "&b=" + EncodeUri(TStringBuilder() << Opts.From);
        begin += "&e=" + EncodeUri(TStringBuilder() << Opts.To);

        if (cluster == ECluster::PROD_VLA) {
            begin += "&forceCluster=vla";
        }

        if (cluster == ECluster::PROD_SAS) {
            begin += "&forceCluster=sas";
        }

        return begin;
    }

    TLinkMakerOpts Opts;
};

struct TDownloadMonitors {
    TDownloadMonitors(
            IProgressMonitorPtr resolve,
            IProgressMonitorPtr readMeta,
            IProgressMonitorPtr download)
            : Resolve(std::move(resolve))
            , ReadMeta(std::move(readMeta))
            , Download(std::move(download))
    {
    }

    IProgressMonitorPtr Resolve;
    IProgressMonitorPtr ReadMeta;
    IProgressMonitorPtr Download;

    TDownloadMonitors Share() {
        return TDownloadMonitors(Resolve->Share(), ReadMeta->Share(), Download->Share());
    }
};

class TDownloader {
public:
    TDownloader(
            IActorEnginePtr engine,
            const TComparisonConfig& comparisonConfig,
            const TComparisonConfig::TShardConfig& shardConfig,
            TDownloadMonitors monitors,
            IShardComparisonSummary* comparator,
            IOutputStream& err,
            IOutputStream& tr)
            : Engine_(std::move(engine))
            , Conf_(comparisonConfig)
            , ShardConf_(shardConfig)
            , Monitors_(std::move(monitors))
            , Comparator_(comparator)
            , Err_(err)
            , Trace_(tr)
    {
        Name_ = ClusterToStr(ShardConf_.Cluster);
    }

    struct TStat {
        ui64 MetricsTotal{0};

        ui64 ReadMetricsMetaDropped{0};
        ui64 ReadMetricsMetaErrors{0};
        ui64 ReadMetricsMetaSuccess{0};

        ui64 ResolveDcMetricsDropped{0};
        ui64 ResolveDcMetricsErrors{0};
        ui64 ResolveDcMetricsSuccess{0};

        ui64 DownloadMetricsErrors{0};
        ui64 DownloadMetricsSuccess{0};
    };

    void InitializeActual() {
        Initialize();

        DownloadConsumer_->SetCallback([this](TSeriesAndId&& series, TActorMetaData&& meta) {
            const auto& labels = MetricMetaCast(meta)->Labels;
            auto name = MetricMetaCast(meta)->Name;

            Comparator_->AddActual(labels, std::move(series));
        });
    }

    void InitializeExpected() {
        Initialize();

        DownloadConsumer_->SetCallback([this](TSeriesAndId&& series, TActorMetaData&& meta) {
            const auto& labels = MetricMetaCast(meta)->Labels;
            auto name = MetricMetaCast(meta)->Name;

            Comparator_->AddExpected(labels, std::move(series));
        });
    }

    void Prepare() {
        ReadMetricsFromFile(ShardConf_.DumpsPath, [this](ui32 shardId, ui64 localId, TLabels labels) {
            Comparator_->AddLabels(labels);
            Metrics_.emplace_back(TMetric{shardId, localId, std::move(labels)});
        });
    }

    void DoCompare() {
        MetricsTotal = 0;
        for (auto&& metric: Metrics_) {
            ++MetricsTotal;
            DoCompareCallback(metric.ShardId, metric.LocalId, std::move(metric.Labels));
        }
    }

    void Wait() {
        Monitors_.ReadMeta->Start();
        MetaReader_->Close();

        Monitors_.Resolve->Start();
        Resolver_->Close();

        Monitors_.Download->Start();
        Downloader_->Close();
    }

    TStat Stat() const {
        TStat stat{
            .MetricsTotal = MetricsTotal,

            .ReadMetricsMetaDropped = MetaReadConsumer_->Dropped(),
            .ReadMetricsMetaErrors = MetaReadConsumer_->Errors(),
            .ReadMetricsMetaSuccess = MetaReadConsumer_->Processed(),

            .ResolveDcMetricsDropped = ResolveConsumer_->Dropped(),
            .ResolveDcMetricsErrors = ResolveConsumer_->Errors(),
            .ResolveDcMetricsSuccess = ResolveConsumer_->Processed(),

            .DownloadMetricsErrors = DownloadConsumer_->Errors(),
            .DownloadMetricsSuccess = DownloadConsumer_->Processed(),
        };

        return stat;
    }

private:
    void Initialize() {
        ResolveConsumer_ = CreateComparisonResolveDcConsumer(ShardConf_.DcFilter, Monitors_.Resolve, Err_, Trace_);
        MetaReadConsumer_ = CreateComparisonReadMetricsMetaConsumer(Monitors_.ReadMeta, Err_, Trace_);
        DownloadConsumer_ = CreateComparisonDownloadMetricsConsumer(Monitors_.Download, Err_, Trace_);

        auto clientFactory = [cluster = ShardConf_.Cluster]() {
            auto client = MakeStockpileClient();
            client->Open(cluster, TDuration::Minutes(30));
            return client;
        };

        auto hostResolveFactory = []() {
            return StaticHostResolver();
        };

        auto groupResolveFactory = [this]() {
            Y_ENSURE(Conf_.YasmConfigsDir, "empty yasm configs directory path");
            return StaticGroupResolver(Conf_.YasmConfigsDir);
        };

        MetaReader_ = CreateReadMetricsMetaActor(Engine_, *MetaReadConsumer_, clientFactory);
        Resolver_ = CreateResolveDcActor(Engine_, *ResolveConsumer_, hostResolveFactory, groupResolveFactory);
        Downloader_ = CreateDownloadMetricsActor(Engine_, *DownloadConsumer_, clientFactory, Conf_.FromMillis, Conf_.ToMillis);

        MetaReadConsumer_->SetCallback([this](TActorMetaData&& meta) {
            ReadMetaCallback(std::move(meta));
        });

        ResolveConsumer_->SetCallback([this](TActorMetaData&& meta) {
            auto id = MetricMetaCast(meta)->Id;
            Monitors_.Download->IncTotal(1u);
            Downloader_->Download(id, std::move(meta));
        });
    }

    static IGroupResolverPtr StaticGroupResolver(const TString& path) {
        static IGroupResolverPtr resolver = MakeGroupResolver(path);
        return resolver;
    }

    static IHostResolverPtr StaticHostResolver() {
        static IHostResolverPtr factory = MakeHostResolver();
        return factory;
    }

    static TComparisonConfig::EShardType DiscoverShardType(const NMonitoring::TLabels& labels) {
        if (labels.Has(NSolomon::NLabels::LABEL_GROUP)) {
            return TComparisonConfig::EShardType::GROUP;
        }

        return TComparisonConfig::EShardType::HOST;
    }

    static TString GetResolveLabel(const NMonitoring::TLabels& labels, TComparisonConfig::EShardType type) {
        Y_VERIFY(type != TComparisonConfig::EShardType::UNKNOWN);
        const TStringBuf label = type == TComparisonConfig::EShardType::HOST ? NSolomon::NLabels::LABEL_HOST: NSolomon::NLabels::LABEL_GROUP;

        auto mb = labels.Find(label);
        if (!mb) {
            return {};
        }

        return TString(mb->Value());
    }

    void DoCompareCallback(ui32 shardId, ui64 localId, TLabels labels) {
        labels.SortByName();
        TStockpileIds id{shardId, localId};
        TMetricMeta meta(std::move(labels), id, Name_);

        if (!ShardConf_.SkipALiveChecking) {
            MetaReader_->ReadMetricMeta(id, std::move(meta));
            Monitors_.ReadMeta->IncTotal(1u);
            return;
        }

        if (!ShardConf_.DcFilter.IsEmpty()) {
            Monitors_.Resolve->IncTotal(1u);
            ResolveDc(std::move(meta));
            return;
        }

        Monitors_.Download->IncTotal(1u);
        Downloader_->Download(id, std::move(meta));
    }

    void ReadMetaCallback(TActorMetaData&& meta) {
        Y_VERIFY(MetricMetaCast(meta));
        if (!ShardConf_.DcFilter.IsEmpty()) {
            Monitors_.Resolve->IncTotal(1u);
            ResolveDc(std::move(meta));
        } else {
            auto id = MetricMetaCast(meta)->Id;
            Monitors_.Download->IncTotal(1u);
            Downloader_->Download(id, std::move(meta));
        }
    }

    void ResolveDc(TActorMetaData&& meta) {
        const auto& labels = MetricMetaCast(meta)->Labels;

        auto type = Conf_.ShardType;

        if (type == TComparisonConfig::EShardType::UNKNOWN) {
            type = DiscoverShardType(labels);
        }

        Y_VERIFY(type != TComparisonConfig::EShardType::UNKNOWN);

        if (type == TComparisonConfig::EShardType::HOST) {
            Resolver_->ResolveHostDc(GetResolveLabel(labels, type), std::move(meta));
        } else if (type == TComparisonConfig::EShardType::GROUP) {
            Resolver_->ResolveGroupDc(GetResolveLabel(labels, type), std::move(meta));
        }
    }

    template <class TCallback>
    static void ReadMetricsFromFile(TStringBuf fileName, TCallback callback) {
        TFileInput file(TString{fileName});
        TString line;

        while (file.ReadLine(line)) {
            if (!line) {
                continue;
            }

            ui64 localId;
            ui32 shardId;
            TLabels labels;

            try {

                auto lineIterable = StringSplitter(line.begin(), line.end()).Split('|');
                auto lineIterator = lineIterable.begin();

                shardId = TIO::ParseShardId(lineIterator->Token());
                ++lineIterator;
                localId = TIO::ParseLocalId(lineIterator->Token());
                ++lineIterator;
                labels = TIO::ParseLabels(lineIterator->Token());
            } catch (...) {
                Cerr << "error while parsing input line: " << CurrentExceptionMessage() << "\n";
            }

            callback(shardId, localId, std::move(labels));
        }
    }

private:
    struct TMetric {
        ui32 ShardId;
        ui64 LocalId;
        TLabels Labels;
    };

    IActorEnginePtr Engine_;

    const TComparisonConfig& Conf_;
    const TComparisonConfig::TShardConfig& ShardConf_;
    TDownloadMonitors Monitors_;
    IShardComparisonSummary* Comparator_;
    IOutputStream& Err_;
    IOutputStream& Trace_;

    IResolveDcActorPtr Resolver_;
    IReadMetricsMetaActorPtr MetaReader_;
    IDownloadMetricsActorPtr Downloader_;

    IComparisonResolveDcConsumerPtr ResolveConsumer_;
    IComparisonReadMetricsMetaConsumerPtr MetaReadConsumer_;
    IComparisonDownloadMetricsConsumerPtr DownloadConsumer_;

    TVector<TMetric> Metrics_;

    TString Name_;
    ui64 MetricsTotal{0};
};

class TComparisonMaker {
public:
    TComparisonMaker(TShardsComparisonConfig config)
        : Config_(std::move(config))
    {
    }

    void DoCompare() {
        auto engine = CreateActorEngine(64u);

        TLinkMakerOpts linkMakerOpts {
            .ActualCluster = Config_.Actual.Cluster,
            .ActualPN = Config_.Actual.ProjectName,
            .ActualClusterLabel = Config_.Actual.ClusterLabel,

            .ExpectedCluster = Config_.Expected.Cluster,
            .ExpectedPN = Config_.Expected.ProjectName,
            .ExpectedClusterLabel = Config_.Expected.ClusterLabel,

            .From = Config_.FromMillis,
            .To = Config_.ToMillis,
        };

        TShardComparisonOptions comparisonOpts {
            .Comparator = MakeHolder<TMetricComparator>(),
            .ShardComparisonFlags = Config_.ComparisonFlags.ShardComparisonFlags,
            .SeriesComparisonFlags = Config_.ComparisonFlags.SeriesComparisonFlags,
            .Expected = {
                .Name = ClusterToStr(Config_.Expected.Cluster),
                .Cluster = Config_.Expected.Cluster,
            },
            .Actual = {
                .Name = ClusterToStr(Config_.Actual.Cluster),
                .Cluster = Config_.Actual.Cluster,
            },
        };

        if (linkMakerOpts.Initialized()) {
            comparisonOpts.LinkMaker = MakeHolder<TLinkMaker>(std::move(linkMakerOpts));
        }

        TDownloadMonitors monitors{
            CreateProgressMonitor("resolve dc"),
            CreateProgressMonitor("read metrics meta"),
            CreateProgressMonitor("downlod metrics"),
        };

        auto compMonitor = CreateProgressMonitor("comparing series");

        TUnbufferedFileOutput err(GetErrorsPath());
        TUnbufferedFileOutput out(GetOutputPath());
        TUnbufferedFileOutput tr(GetTracePath());

        auto comparator = CreateShardComparisonSummary(std::move(comparisonOpts), engine);

        TDownloader lhs(
                engine,
                Config_,
                Config_.Actual,
                monitors,
                comparator.Get(),
                err,
                tr);

        TDownloader rhs(
                engine,
                Config_,
                Config_.Expected,
                monitors.Share(),
                comparator.Get(),
                err,
                tr);

        lhs.InitializeActual();
        rhs.InitializeExpected();

        lhs.Prepare();
        rhs.Prepare();

        auto lhsF = NThreading::Async([&lhs]() {
           lhs.DoCompare();
           lhs.Wait();
        }, *engine);

        auto rhsF = NThreading::Async([&rhs]() {
            rhs.DoCompare();
            rhs.Wait();
        }, *engine);

        NThreading::WaitAll(lhsF, rhsF).Wait();

        WriteStat(ClusterToStr(Config_.Expected.Cluster), lhs.Stat(), out);
        out << "\n";
        WriteStat(ClusterToStr(Config_.Actual.Cluster), rhs.Stat(), out);
        out << "\n";

        comparator->Write(out);
    }

private:
    TString GetShardName(TStringBuf name) {
        if (Config_.ShardName) {
            return Config_.ShardName;
        }

        TStringBuf left;
        TStringBuf right;
        name.RSplit('.', left, right);

        TStringBuf l;
        TStringBuf r;

        left.RSplit('/', l, r);

        return TString(r);
    }

    static void WriteField(TStringBuf name, TStringBuf mes, ui64 value, IOutputStream& s) {
        s << "(" << name << ") " << mes << ": " << value << "\n";
    }

    static void WritePositiveField(TStringBuf name, TStringBuf mes, ui64 value, IOutputStream& s) {
        if (value > 0u) {
            WriteField(name, mes, value, s);
        }
    }

    void WriteStat(TStringBuf name, TDownloader::TStat stat, IOutputStream& stream) {
        WriteField(name, "metrics total", stat.MetricsTotal, stream);
        WritePositiveField(name, "read metrics meta successes", stat.ReadMetricsMetaSuccess, stream);
        WritePositiveField(name, "read metrics meta errors", stat.ReadMetricsMetaErrors, stream);
        WritePositiveField(name, "read metrics meta drops", stat.ReadMetricsMetaDropped, stream);

        WritePositiveField(name, "resolve dc successes", stat.ResolveDcMetricsSuccess, stream);
        WritePositiveField(name, "resolve dc errors", stat.ResolveDcMetricsErrors, stream);
        WritePositiveField(name, "resolve dc drops", stat.ResolveDcMetricsDropped, stream);

        WritePositiveField(name, "download data successes", stat.DownloadMetricsSuccess, stream);
        WritePositiveField(name, "download data errors", stat.DownloadMetricsErrors, stream);
    }

    TString GetShardName() {
        TString lhs = GetShardName(Config_.Expected.DumpsPath);
        TString rhs = GetShardName(Config_.Actual.DumpsPath);
        if (lhs.size() < rhs.size()) {
            return lhs;
        }

        return rhs;
    }

    TString GetOutputPath() {
        if (Config_.ComparisonOutputPath) {
            return Config_.ComparisonOutputPath;
        }

        TString name = GetShardName();
        return name + "-output.txt";
    }

    TString GetErrorsPath() {
        if (Config_.ComparisonErrPath) {
            return Config_.ComparisonErrPath;
        }

        TString name = GetShardName();
        return name + "-errors.txt";
    }

    TString GetTracePath() {
        if (Config_.ComparisonTracePath) {
            return Config_.ComparisonTracePath;
        }

        TString name = GetShardName();
        return name + "-trace.txt";
    }

private:
    TComparisonConfig Config_;
};

int main(int argc, const char** argv) {
    TShardsComparisonConfig conf;
    TString error;

    if (!NGetoptPb::GetoptPb(argc, argv, conf, error)) {
        Cerr << error;
    }

    NSolomon::MergeIncludes(conf);

    TComparisonMaker solver(conf);
    solver.DoCompare();

    return 0;
}
