#pragma once

#include "consumers.h"

#include <solomon/tools/data-comparison/config/comparison_config.pb.h>
#include <solomon/tools/data-comparison/lib/diff/diff.h>
#include "solomon/tools/data-comparison/lib/util/options.h"
#include "solomon/tools/data-comparison/lib/util/io.h"
#include <solomon/tools/data-comparison/lib/cluster/cluster.h>

struct TComparisonConfig {
    TComparisonConfig(const TShardsComparisonConfig& protoConf);

    enum class EShardType {
        HOST = 0,
        GROUP,
        UNKNOWN
    };

    struct TShardConfig {
        ECluster Cluster;
        TString DumpsPath;
        TString ProjectName;
        TString ClusterLabel;

        bool AlignHistogramBounds;
        bool SkipALiveChecking;
        TDcFilter DcFilter;
    };

    struct TComparisonFlags {
        TOptions SeriesComparisonFlags;
        TOptions ShardComparisonFlags;
    };

    EShardType ShardType;

    TShardConfig Expected;
    TShardConfig Actual;

    TComparisonFlags ComparisonFlags;

    TString ComparisonOutputPath;
    TString ComparisonErrPath;
    TString ComparisonTracePath;

    TString YasmConfigsDir;
    TString ShardName;

    TInstant FromMillis;
    TInstant ToMillis;

private:
    static EShardType FromProto(TShardsComparisonConfig::EShardType shardType) {
        switch (shardType) {
            case TShardsComparisonConfig::HOST: {
                return EShardType::HOST;
            }
            case TShardsComparisonConfig::GROUP: {
                return EShardType::GROUP;
            }
        }
    }

    static ECluster FromProto(::TShardConfig::ECluster cluster) {
        switch (cluster) {
            case ::TShardConfig::PRE: {
                return ECluster::PRE;
            }
            case ::TShardConfig::PROD_SAS: {
                return ECluster::PROD_SAS;
            }
            case ::TShardConfig::PROD_VLA: {
                return ECluster::PROD_VLA;
            }
        }
    }

    static void InitShard(TShardConfig& conf, const ::TShardConfig& protoConf) {
        conf.Cluster = ECluster::UNKNOWN;
        if (protoConf.HasCluster()) {
            conf.Cluster = FromProto(protoConf.GetCluster());
        }

        if (protoConf.HasProjectName()) {
            conf.ProjectName = TString(protoConf.GetProjectName().data(), protoConf.GetProjectName().size());
        }

        if (protoConf.HasClusterLabel()) {
            conf.ClusterLabel = TString(protoConf.GetClusterLabel().data(), protoConf.GetClusterLabel().size());
        }

        conf.DumpsPath = protoConf.GetDumpPath();
        conf.AlignHistogramBounds = false;

        conf.SkipALiveChecking = false;
        if (protoConf.HasSkipAliveChecking()) {
            conf.SkipALiveChecking = protoConf.GetSkipAliveChecking();
        }

        for (const auto& dcStr: protoConf.GetFilterDc()) {
            conf.DcFilter.AddFlag(StrToDc(dcStr));
        }
    }

    static ESeriesComparisonFlags SeriesComparisonFlagFromProto(int protoFlag) {
        switch (protoFlag) {
            case ::TComparisonFlags::MISSING_POINTS: {
                return ESeriesComparisonFlags::MISSING_POINTS;
            }
            case ::TComparisonFlags::VALUE_DIFFERENCE: {
                return ESeriesComparisonFlags::VALUE_DIFFERENCE;
            }
            case ::TComparisonFlags::MISSING_POINTS_SUMMARY: {
                return ESeriesComparisonFlags::MISSING_POINTS_SUMMARY;
            }
            case ::TComparisonFlags::VALUE_DIFFERENCE_SUMMARY: {
                return ESeriesComparisonFlags::VALUE_DIFFERENCE_SUMMARY;
            }

            default: {
                ythrow yexception() << "unknown series comparison flag";
            }
        }
    }

    static EShardComparisonFlags ShardComparisonFlagFromProto(int protoFlag) {
        switch (protoFlag) {
            case ::TComparisonFlags::MISSING_METRICS: {
                return EShardComparisonFlags::MISSING_METRICS;
            }
            case ::TComparisonFlags::SERIES_DIFF: {
                return EShardComparisonFlags::SERIES_DIFF;
            }
            case ::TComparisonFlags::STORAGE_LINK: {
                return EShardComparisonFlags::STORAGE_LINK;
            }
            case ::TComparisonFlags::TYPES_SUMMARY: {
                return EShardComparisonFlags::TYPES_SUMMARY;
            }
            case ::TComparisonFlags::TYPE_MISMATCH: {
                return EShardComparisonFlags::TYPE_MISMATCH;
            }
            default: {
                ythrow yexception() << "unknown shard comparison flag";
            }
        }
    }

    static void InitComparisonFlags(TComparisonFlags& flags, const ::TComparisonFlags& protoConf) {
        for (auto flag: protoConf.GetSeriesComparisonFlags()) {
            flags.SeriesComparisonFlags.AddFlag(SeriesComparisonFlagFromProto(flag));
        }
        for (auto flag: protoConf.GetShardComparisonFlags()) {
            flags.ShardComparisonFlags.AddFlag(ShardComparisonFlagFromProto(flag));
        }
    }

};

TComparisonConfig::TComparisonConfig(const TShardsComparisonConfig& protoConf) {
    ShardType = EShardType::UNKNOWN;
    if (protoConf.HasShardType()) {
        ShardType = FromProto(protoConf.GetShardType());
    }

    Y_ENSURE(protoConf.HasExpected());
    Y_ENSURE(protoConf.HasActual());

    InitShard(Expected, protoConf.GetExpected());
    InitShard(Actual, protoConf.GetActual());

    if (protoConf.HasComparisonFlags()) {
        InitComparisonFlags(ComparisonFlags, protoConf.GetComparisonFlags());
    }

    if (protoConf.HasShardName()) {
        ShardName = protoConf.GetShardName();
    }

    if (protoConf.GetComparisonOutputPath()) {
        ComparisonOutputPath = protoConf.GetComparisonOutputPath();
    }
    if (protoConf.GetComparisonErrPath()) {
        ComparisonErrPath = protoConf.GetComparisonErrPath();
    }
    if (protoConf.GetComparisonTracePath()) {
        ComparisonTracePath = protoConf.GetComparisonTracePath();
    }

    if (protoConf.HasYasmConfigsDir()) {
        YasmConfigsDir = protoConf.GetYasmConfigsDir();
    }

    Y_ENSURE(protoConf.HasFromMillis());
    Y_ENSURE(protoConf.HasToMillis());

    FromMillis = TIO::ParseTime(protoConf.GetFromMillis());
    ToMillis = TIO::ParseTime(protoConf.GetToMillis());
}
