#include <solomon/tools/data-comparison/lib/resolver/resolver.h>
#include <solomon/tools/data-comparison/lib/util/io.h>

#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/getopt/last_getopt.h>

#include <util/stream/file.h>
#include <util/string/split.h>

class THostParser: public TIO::TReadVisitor {
public:
    THostParser(bool asLabels)
        : AsLabels_(asLabels)
    {}

    TString ReleaseHost() {
        Y_ENSURE(Host_);
        return std::move(Host_);
    }

private:
    bool AsLabels_;
    TString Host_;

    void OnLabels(const NMonitoring::TLabels& labels) override {
        if (!AsLabels_) {
            return;
        }

        auto mb = labels.Find(TStringBuf("host"));
        Y_ENSURE(mb, "label \"host\" not found in labels " << labels);

        Host_ = mb->Value();
    }

    void OnStr(TStringBuf str) override {
        if (AsLabels_) {
            return;
        }

        Host_ = str;
    }
};

TString ParseHost(const TString& data, TStringBuf schema, bool asLabels) {
    THostParser parser(asLabels);
    TIO::ReadMetric(schema, data, parser);

    return parser.ReleaseHost();
}

struct THostMeta {
    TVector<TString> Lines;
    NSolomon::EDc Dc{NSolomon::EDc::UNKNOWN};
};

struct THostAndDc {
    TString Host;
    TDcResolveResult ResolveResult{};
};

void ResolveIteration(TVector<THostAndDc>& hosts, IHostResolver* resolver) {
    TVector<NThreading::TFuture<void>> fs;

    for (auto& hostAndDc: hosts) {
        NThreading::TFuture<void> f = resolver->ResolveDc(hostAndDc.Host).Apply([data=&hostAndDc](auto f) {
            TDcResolveResult resolvedDc = f.ExtractValueSync();
            data->ResolveResult = resolvedDc;
        });

        fs.emplace_back(std::move(f));
    }

    auto all = NThreading::WaitAll(fs);
    all.Wait();
}

TVector<THostAndDc> HostsToResolve(const THashMap<TString, THostMeta>& table) {
    TVector<THostAndDc> r;
    for (const auto& [host, meta]: table) {
        if (meta.Dc == NSolomon::EDc::UNKNOWN) {
            r.emplace_back(THostAndDc{host});
        }
    }

    return r;
}

void UpdateTable(THashMap<TString, THostMeta>& table, const TVector<THostAndDc>& hostsDc) {
    for (const auto& hostAndDc: hostsDc) {
        switch (hostAndDc.ResolveResult.Status) {
            case EDcResolveStatus::OK: {
                if (hostAndDc.ResolveResult.Dc == NSolomon::EDc::UNKNOWN) {
                    Cerr << TStringBuf(TStringBuilder() << "host:" << hostAndDc.Host << " is unknown\n");
                    table.erase(hostAndDc.Host);
                } else {
                    table[hostAndDc.Host].Dc = hostAndDc.ResolveResult.Dc;
                }
                break;
            }
            case EDcResolveStatus::NOT_FOUND: {
                Cerr << TStringBuf(TStringBuilder() << "host:" << hostAndDc.Host << " not found\n");
                table.erase(hostAndDc.Host);
                break;
            }
            case EDcResolveStatus::TIMEOUT: {
                break;
            }
            case EDcResolveStatus::UNKNOWN: {
                table.erase(hostAndDc.Host);
                Cerr << TStringBuf(TStringBuilder() << "unknown error while resolving host:" << hostAndDc.Host << '\n');
                break;
            }
        }
    }
}

int Main(IInputStream& in, IOutputStream& out, NSolomon::EDc dc, bool asLabels, TStringBuf schema) {
    auto resolver = MakeHostResolver();

    THashMap<TString, THostMeta> table;

    {
        TString data;
        while (in.ReadLine(data)) {
            if (!data) {
                continue;
            }

            table[ParseHost(data, schema, asLabels)].Lines.emplace_back(std::move(data));
        }
    }

    const size_t maxIterations = 5;
    for (size_t i = 0; i < maxIterations; ++i) {
        auto hosts = HostsToResolve(table);
        if (hosts.empty()) {
            break;
        }
        ResolveIteration(hosts, resolver.Get());
        UpdateTable(table, hosts);

        if (i + 1 == maxIterations) {
            Cerr << "[WARN] timeouts after all iterations\n";
        }
    }

    for (const auto& [_, v]: table) {
        if (v.Dc != dc) {
            continue;
        }

        for (const auto& data: v.Lines) {
            out << (data + '\n');
        }
    }

    return 0;
}

int Main(const NLastGetopt::TOptsParseResult& res) {
    TString dc = res.GetFreeArgs()[0];

    bool asLabels = res.Has("as-labels");
    TString schema = asLabels ? "l" : "s";
    if (res.Has("input-schema")) {
        schema = res.Get("input-schema");
    }

    if (res.Has("input-file") && res.Has("output-file")) {
        TFileInput in(res.Get("input-file"));
        TFileOutput out(res.Get("output-file"));

        return Main(in, out, StrToDc(dc), asLabels, schema);
    } else if (res.Has("input-file")) {
        TFileInput in(res.Get("input-file"));

        return Main(in, Cout, StrToDc(dc), asLabels, schema);
    } else if (res.Has("output-file")) {
        TFileOutput out(res.Get("output-file"));

        return Main(Cin, out, StrToDc(dc), asLabels, schema);
    }

    return Main(Cin, Cout, StrToDc(dc), asLabels, schema);
}

int main(int argc, const char* argv[]) {
    NLastGetopt::TOpts opts;
    opts.SetTitle("dc-filter");
    opts.SetFreeArgsNum(1);
    opts.SetFreeArgTitle(0, "<dc>", "dc to filter");
    opts.AddLongOption("input-file", "path to a file with input data")
            .RequiredArgument("[path]")
            .Optional();
    opts.AddLongOption("output-file", "path to a output file")
            .RequiredArgument("[path]")
            .Optional();
    opts.AddLongOption("as-labels", "input consists of number of labels with \"host\" label")
            .NoArgument()
            .Optional();
    opts.AddLongOption("input-schema", "schema of input file lines, default value is \"s\" (as-labels isn't set), \"l\" (as-labels is set)")
            .RequiredArgument("[path]")
            .Optional();

    try {
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);
        return Main(res);
    } catch (const NLastGetopt::TUsageException&) {
        opts.PrintUsage("dc-filter");
    } catch (...) {
        Cerr << "Unhandled exception: " << CurrentExceptionMessage() << Endl;
    }

    return 1;
}
