#include "diff.h"

#include <library/cpp/monlib/metrics/log_histogram_snapshot.h>

#include <cmath>

static double Diff(double x, double y) {
    return std::abs(x - y);
}

class TYasmHistogramComparator {
public:
    using TBucket = NSolomon::NTs::NValue::THistogram::TBucket;
    using TBuckets = std::vector<TBucket>;

    bool operator()(const TBuckets& lhs, const TBuckets& rhs) const {
        return Compare(lhs, rhs) || Compare(rhs, lhs);
    }

private:
    static bool Compare(const TBuckets& from, const TBuckets& to) {
        for (size_t i = 1; i + 1 < to.size(); ++i) {
            double boundBegin = to[i - 1].UpperBound;
            double boundEnd = to[i].UpperBound;

            auto end = std::lower_bound(
                    from.begin(),
                    from.end(),
                    TBucket{boundEnd, 0u},
                    [](const auto& lhs, const auto& rhs) { return lhs.UpperBound < rhs.UpperBound; });


            auto begin = std::upper_bound(
                    from.begin(),
                    from.end(),
                    TBucket{boundBegin, 0u},
                    [](const auto& lhs, const auto& rhs) { return lhs.UpperBound < rhs.UpperBound; });

            if (begin == from.begin()) {
                if (to[i].Value == 0) {
                    continue;
                }

                return false;
            }

            double sum = 0.0;
            for (auto it = begin; ; ++it) {
                sum += it->Value;

                if (it == end) {
                    break;
                }
            }

            double estimate = 0.0;
            if (boundBegin == (begin - 1)->UpperBound && boundEnd == end->UpperBound) {
                estimate = sum;
            } else {
                estimate = sum * (boundEnd - boundBegin) / (end->UpperBound - (begin - 1)->UpperBound);
            }

            if (Diff(estimate, (double) to[i].Value) > 1.0) {
                return false;
            }
        }

        return true;
    }
};

template <class TBucketProvider, class TPercentileCalculator>
double PercentileCompare(
        const TBucketProvider& lhs,
        const TBucketProvider& rhs,
        TPercentileCalculator calculator,
        double step)
{
    double delta = step;
    double max = 0.0;
    while (delta < 1.0) {
        double lp = calculator(lhs, delta);
        double rp = calculator(rhs, delta);
        double d = Diff(lp, rp);
        if (d > max) {
            max = d;
        }
        delta += step;
    }

    return max;
}

template <class TBucketProvider>
double Percentile(const TBucketProvider& provider, double p) {
    if (provider.Count() == 0u) {
        return 0.0;
    }

    double sum = 0.0;

    for (size_t i = 0; i < provider.Count(); ++i) {
        sum += provider.Value(i);
    }

    if (sum == 0.0) {
        return 0.0;
    }

    double delta = 0.0;

    for (size_t i = 0; i < provider.Count(); ++i) {
        double pn = delta + provider.Value(i) / sum;

        if (pn >= p) {
            if (i == 0) {
                return 0.0;
            }

            double len = provider.UpperBound(i) - provider.UpperBound(i - 1);

            return provider.UpperBound(i - 1) + len * (p - delta) / (pn - delta);
        }

        delta = pn;
    }

    return 0.0;
}

template <class TBucketProvider>
double LogPercentile(const TBucketProvider& provider, double p) {
    double sum = 0.0;
    for (size_t i = 0; i < provider.Count(); ++i) {
        sum += provider.Value(i);
    }

    sum += provider.Zeros();

    double delta = provider.Zeros() / sum;
    if (delta >= p) {
        return 0.0;
    }

    for (size_t i = 0; i < provider.Count(); ++i) {
        delta += provider.Value(i) / sum;
        if (delta >= p) {
            return provider.UpperBound(i);
        }
    }

    return 0.0;
}

struct THistogramBucketProvider {
    THistogramBucketProvider(const NSolomon::NTs::NValue::THistogram& base)
        : Hist(base)
    {
    }

    double Value(size_t i) const {
        return Hist.Buckets[i].Value;
    }

    double UpperBound(size_t i) const {
        return Hist.Buckets[i].UpperBound;
    }

    size_t Count() const {
        Y_ENSURE(Hist.Buckets.size() >= 1u);
        return Hist.Buckets.size() - 1;
    }

    const NSolomon::NTs::NValue::THistogram& Hist;
};

struct TLogHistogramBucketProvider {
    ui64 Zeros() const {
        return Base_.ZeroCount;
    }

    ui64 Count() const {
        return Base_.Values.size();
    }

    double Value(size_t i) const {
        return Base_.Values[i];
    }

    double UpperBound(size_t i) const {
        return std::pow(Base_.Base, Base_.StartPower + (int) i);
    }

    const NSolomon::NTs::NValue::TLogHistogram& Base_;
};

bool TMetricComparator::Equal(
        const NSolomon::NTs::NValue::THistogram& lhs,
        const NSolomon::NTs::NValue::THistogram& rhs) const
{
    THistogramBucketProvider providerLhs(lhs);
    THistogramBucketProvider providerRhs(rhs);
    TYasmHistogramComparator comparator;

    return (PercentileCompare(providerLhs, providerRhs, Percentile<THistogramBucketProvider>, 0.05) < DEFAULT_EPS
            || comparator(lhs.Buckets, rhs.Buckets));
}

bool TMetricComparator::Equal(
        const NSolomon::NTs::NValue::TSummaryDouble& lhs,
        const NSolomon::NTs::NValue::TSummaryDouble& rhs) const
{
    double maxD = std::max({
        Diff(lhs.Sum, rhs.Sum),
        Diff(lhs.Min, rhs.Min),
        Diff(lhs.Max, rhs.Max),
        Diff(lhs.Last, rhs.Last),
        Diff(lhs.CountValue, rhs.CountValue)
    });

    return (maxD < DEFAULT_EPS);
}

bool TMetricComparator::Equal(
        const NSolomon::NTs::NValue::TLogHistogram& lhs,
        const NSolomon::NTs::NValue::TLogHistogram& rhs) const
{
    TLogHistogramBucketProvider lhsP{lhs};
    TLogHistogramBucketProvider rhsP{rhs};

    return (PercentileCompare(lhsP, rhsP, LogPercentile<TLogHistogramBucketProvider>, 0.05) < DEFAULT_EPS);
}
