#include "diff.h"

#include <solomon/tools/data-comparison/lib/util/wait_context.h>

#include <unordered_map>
#include <optional>
#include <atomic>

using NMonitoring::ISummaryDoubleSnapshot;
using NMonitoring::IHistogramSnapshot;
using NMonitoring::TLogHistogramSnapshot;
using NMonitoring::TLabels;

using NMonitoring::EMetricType;
using NMonitoring::MetricTypeToStr;

IOutputStream& operator<<(IOutputStream& s, const NSolomon::NTs::NValue::THistogram& h) {
    s << "{";
    for (size_t i = 0; i < h.Buckets.size(); ++i) {
        if (h.Buckets[i].UpperBound == NMonitoring::HISTOGRAM_INF_BOUND) {
            s << "inf";
        } else {
            s << h.Buckets[i].UpperBound;
        }

        s << ": " << h.Buckets[i].Value;

        if (i + 1 != h.Buckets.size()) {
            s << ", ";
        }
    }

    s << "}";

    return s;
}

IOutputStream& operator<<(IOutputStream& s, const NSolomon::NTs::NValue::TLogHistogram& h) {
    s << "{";
    s << "zeros: " << h.ZeroCount << ", startPower: " << h.StartPower << ", values: [";
    for (size_t i = 0; i < h.Values.size(); ++i) {
        s << h.Values[i];
        if (i + 1 != h.Values.size()) {
            s << ", ";
        }
    }

    s << "]";

    s << "}";

    return s;
}

IOutputStream& operator<<(IOutputStream& s, const NSolomon::NTs::NValue::TSummaryDouble& h) {
    s << "{" << "sum: " << h.Sum << ", min: " << h.Min << ", max: " << h.Max << ", last: " << h.Last << ", count: " << h.CountValue << "}";
    return s;
}

IOutputStream& operator<<(IOutputStream& s, const NSolomon::NTs::TVariantPoint& point) {
    try {
        const auto& h = point.Get<NSolomon::NTs::NValue::THistogram>();
        s << h;
        return s;
    } catch (...) {}

    try {
        const auto& h = point.Get<NSolomon::NTs::NValue::TLogHistogram>();
        s << h;
        return s;
    } catch (...) {}

    try {
        const auto& h = point.Get<NSolomon::NTs::NValue::TSummaryDouble>();
        s << h;
        return s;
    } catch (...) {}

    ythrow yexception() << "unexpected type";
}

class TLineStream {
public:
    TLineStream(IOutputStream& base)
        : Base_(base)
    {
    }

    void ResetWritten() {
        Written_ = false;
    }

    void PrintLine(TStringBuf line) {
        for (const auto& trigger: Triggers_) {
            if (trigger.Begin) {
                WriteSpaces(trigger.Spaces);
                Base_ << trigger.Begin << '\n';
            }
        }

        Written_ = true;
        WriteSpaces(Spaces_);
        Base_ << line << '\n';

        Triggers_.clear();
    }

    bool Written() const {
        return Written_;
    }

    void NewLine() {
        Base_ << "\n";
    }

    void Tab() {
        Spaces_ += TAB_SIZE;
    }

    void UnTab() {
        Y_ENSURE(Spaces_ >= TAB_SIZE);
        Spaces_ -= TAB_SIZE;
    }

    void AddTrigger(TString begin) {
        Triggers_.emplace_back(TTrigger{std::move(begin), Spaces_});
    }

    void AddTrigger(TString begin, ui8 spaces) {
        Triggers_.emplace_back(TTrigger{std::move(begin), spaces});
    }

    void DropTriggers() {
        Triggers_.clear();
    }

    IOutputStream& Stream() {
        return Base_;
    }

private:
    void WriteSpaces(ui8 spaces) {
        for (size_t i = 0; i < spaces; ++i) {
            Base_ << ' ';
        }
    }

private:
    IOutputStream& Base_;
    ui8 Spaces_{0u};
    bool Written_{false};

    struct TTrigger {
        TString Begin;
        ui8 Spaces;
    };

    TVector<TTrigger> Triggers_;

    static constexpr ui8 TAB_SIZE = 4u;
};

struct TSeriesComparisonOptions {
    TOptions ComparisonFlags;

    TString Expected;
    TString Actual;

    TLineStream* Stream;

    const TMetricComparator* Comparator;
};

class TSeriesComparisonSummary {
public:
    TSeriesComparisonSummary(TSeriesComparisonOptions opts)
        : Opts_(std::move(opts))
    {
    }

    TSeriesComparisonResult Compare(const ITimeseries& expected, const ITimeseries& actual, const NMonitoring::TLabels& labels) {
        Labels_ = &labels;
        Y_ENSURE(expected.Type() == actual.Type(), "timeseries types are not equal");
        Opts_.Stream->ResetWritten();

        OnComparisonBegin(expected.Type());

        NSolomon::NTs::TVariantPoint lhsPoint;
        NSolomon::NTs::TVariantPoint rhsPoint;

        auto lhsIt = expected.Iterator();
        auto rhsIt = actual.Iterator();

        bool lhsHasNext = lhsIt->Next(&lhsPoint);
        bool rhsHasNext = rhsIt->Next(&rhsPoint);

        while (lhsHasNext && rhsHasNext) {
            if (lhsPoint.Time == rhsPoint.Time) {
                OnTimeEqual(lhsPoint, rhsPoint,  expected.Type());
                lhsHasNext = lhsIt->Next(&lhsPoint);
                rhsHasNext = rhsIt->Next(&rhsPoint);
            } else if (lhsPoint.Time < rhsPoint.Time) {
                OnMissingActualPoint(lhsPoint.Time);
                lhsHasNext = lhsIt->Next(&lhsPoint);
            } else {
                OnMissingExpectedPoint(rhsPoint.Time);
                rhsHasNext = rhsIt->Next(&rhsPoint);
            }
        }

        while (lhsHasNext) {
            OnMissingActualPoint(lhsPoint.Time);
            lhsHasNext = lhsIt->Next(&lhsPoint);
        }

        while (rhsHasNext) {
            OnMissingExpectedPoint(rhsPoint.Time);
            rhsHasNext = rhsIt->Next(&rhsPoint);
        }

        WriteSummary();

        OnComparisonEnd();

        return Summary_;
    }

private:
    void OnComparisonBegin(NMonitoring::EMetricType type) {
        Y_ENSURE(Labels_);
        Opts_.Stream->AddTrigger(TStringBuilder() << *(Labels_) << " (" << type << "): {");
        Opts_.Stream->Tab();
    }

    void OnComparisonEnd() {
        Opts_.Stream->UnTab();

        if (Opts_.Stream->Written()) {
            Opts_.Stream->PrintLine("}");
        }
    }

    void OnTimeEqual(const NSolomon::NTs::TVariantPoint& expected, const NSolomon::NTs::TVariantPoint& actual, NMonitoring::EMetricType type) {
        ++Summary_.CommonPoints;
        ++Summary_.Actual.TotalPoints;
        ++Summary_.Expected.TotalPoints;

        bool equal = false;

        switch (type) {
            case NMonitoring::EMetricType::DSUMMARY: {
                const auto& expectedSummary = expected.Get<NSolomon::NTs::NValue::TSummaryDouble>();
                const auto& actualSummary = actual.Get<NSolomon::NTs::NValue::TSummaryDouble>();
                equal = Opts_.Comparator->Equal(actualSummary, expectedSummary);
                break;
            }
            case NMonitoring::EMetricType::HIST: {
                const auto& expectedHist = expected.Get<NSolomon::NTs::NValue::THistogram>();
                const auto& actualHist = actual.Get<NSolomon::NTs::NValue::THistogram>();
                equal = Opts_.Comparator->Equal(actualHist, expectedHist);
                break;
            }
            case NMonitoring::EMetricType::LOGHIST: {
                const auto& expectedLogHist = expected.Get<NSolomon::NTs::NValue::TLogHistogram>();
                const auto& actualLogHist = actual.Get<NSolomon::NTs::NValue::TLogHistogram>();
                equal = Opts_.Comparator->Equal(actualLogHist, expectedLogHist);
                break;
            }

            default: {
                ythrow yexception() << "unexpected type: " << type;
            }
        }

        if (!equal) {
            OnPointsNotEqual(expected, actual);
        } else {
            OnPointsEqual();
        }
    }

    void OnMissingActualPoint(TInstant ts) {
        OnMissingPoint(ts, Opts_.Actual, Summary_.Actual);
    }

    void OnMissingExpectedPoint(TInstant ts) {
        OnMissingPoint(ts, Opts_.Expected, Summary_.Expected);
    }

    void WriteSummary() {
        if (Opts_.Stream->Written()) {
            Opts_.Stream->AddTrigger("\n", 0u);
        }

        WriteMissingPointsSummary(Opts_.Actual, Summary_.Actual);
        WriteMissingPointsSummary(Opts_.Expected, Summary_.Expected);

        if (!Opts_.ComparisonFlags.ContainFlag(ESeriesComparisonFlags::VALUE_DIFFERENCE_SUMMARY)) {
            return;
        }

        if (Summary_.CommonPoints == Summary_.EqualPoints) {
            return;
        }

        TStringBuilder sb;
        sb << "points are not equal: " << Summary_.CommonPoints - Summary_.EqualPoints << " / " << Summary_.CommonPoints;

        Opts_.Stream->PrintLine(sb);
    }

private:
    void OnPointsEqual() {
        ++Summary_.EqualPoints;
    }

    void OnPointsNotEqual(const NSolomon::NTs::TVariantPoint& expected, const NSolomon::NTs::TVariantPoint& actual) {
        if (!Opts_.ComparisonFlags.ContainFlag(ESeriesComparisonFlags::VALUE_DIFFERENCE)) {
            return;
        }

        TStringBuilder sb;
        sb << expected.Time << ": points are not equal: " << "(" << Opts_.Expected << ") " << expected << " != " << "(" << Opts_.Actual << ") " << actual;
        Opts_.Stream->PrintLine(sb);
    }

    void OnMissingPoint(TInstant ts, TStringBuf name, TSeriesComparisonResult::TShardResult& r) {
        ++r.MissingPoints;
        if (!Opts_.ComparisonFlags.ContainFlag(ESeriesComparisonFlags::MISSING_POINTS)) {
            return;
        }

        Opts_.Stream->PrintLine(TStringBuilder() << ts << ": missing point (" << name << ")");
    }

    void WriteMissingPointsSummary(TStringBuf name, TSeriesComparisonResult::TShardResult& r) {
        if (!Opts_.ComparisonFlags.ContainFlag(ESeriesComparisonFlags::MISSING_POINTS_SUMMARY)) {
            return;
        }

        if (!r.MissingPoints) {
            return;
        }

        TStringBuilder sb;
        sb << "missing points (" << name << "): " << r.MissingPoints;

        Opts_.Stream->PrintLine(sb);
    }

private:
    TSeriesComparisonResult Summary_;
    TSeriesComparisonOptions Opts_;

    const NMonitoring::TLabels* Labels_;
};

TSeriesComparisonResult CompareSeries(
        const ITimeseries& lhs,
        const ITimeseries& rhs,
        const NMonitoring::TLabels& labels,
        const TSeriesComparisonOptions& opts)
{
    TSeriesComparisonSummary summary(opts);
    return summary.Compare(lhs, rhs, labels);
}


struct TShardComparisonSummaryOutput {
    struct TItem {
        ui64 AliveMetrics{0};

        ui64 EmptyMetrics{0};
        ui64 MissingMetrics{0};

        ui64 MissingPoints{0};
        ui64 MetricsWithMissingPoints{0};

        ui64 Points{0};
    };

    TItem Expected;
    TItem Actual;

    ui64 CommonPoints{0};
    ui64 EqualPoints{0};

    ui64 EqualMetrics{0};
    ui64 CommonMetrics{0};

    ui64 MetricsWithDifferentTypes{0};
};

class TEngine {
public:
    using TTask = std::function<void()>;

    TEngine(IEnginePtr base, size_t batchSize)
        : Base_(std::move(base))
        , BatchSize_(batchSize)
        , WC_(MakeWaitContext())
    {
    }

    void Add(TTask&& task) {
        TasksQueue_.emplace_back(std::move(task));
        if (TasksQueue_.size() == BatchSize_) {
            Base_->SafeAddFunc([queue = std::move(TasksQueue_), wc = WC_]() {
               for (auto&& task: queue) {
                   task();
               }
            });
        }
    }

    void Flush() {
        if (TasksQueue_.empty()) {
            return;
        }

        Base_->SafeAddFunc([queue = std::move(TasksQueue_), wc = WC_]() {
            for (auto&& task: queue) {
                task();
            }
        });
    }

    void Close() {
        Flush();
        auto f = WC_->GetFuture();
        WC_.Reset(nullptr);
        f.Wait();
    }

private:
    IEnginePtr Base_;
    size_t BatchSize_;

    TVector<TTask> TasksQueue_;
    TWaitContextPtr WC_;
};

class TShardComparisonSummary: public IShardComparisonSummary {
public:
    TShardComparisonSummary(TShardComparisonOptions opts, IEnginePtr engine)
        : Opts_(std::move(opts))
        , Engine_(std::move(engine))
    {
    }

    void AddActual(const NMonitoring::TLabels& labels, TSeriesAndId&& series) override {
        auto it = Map_.find(labels);

        Y_ENSURE(it != Map_.end());
        auto& value = (it->second);

        value.Actual = std::move(series.Series);
        value.ActualId = series.Id;
    }

    void AddExpected(const NMonitoring::TLabels& labels, TSeriesAndId&& series) override {
        auto it = Map_.find(labels);

        Y_ENSURE(it != Map_.end());
        auto& value = (it->second);

        value.Expected = std::move(series.Series);
        value.ExpectedId = series.Id;
    }

    void AddLabels(const NMonitoring::TLabels& labels) override {
        Map_[labels] = TValue{};
    }

    void Write(IOutputStream& stream) override {
        TEngine engine(Engine_, BATCH_SIZE);
        compareMon = CreateProgressMonitor("compare metrics");
        compareMon->IncTotal(Map_.size());
        compareMon->Start();

        for (auto& [key, value]: Map_) {
            engine.Add([this, key = &key, value = &value]() {
                Compare(*key, *value);
            });
        }

        engine.Close();

        TLineStream s(stream);
        OnComparisonBegin(s);

        mergeMon = CreateProgressMonitor("merge results");
        mergeMon->IncTotal(Map_.size());
        mergeMon->Start();

        for (auto& [_, value]: Map_) {
            Merge(value);
            mergeMon->IncProgress(1u);
        }

        WriteSummary("total", Total_, s);

        s.NewLine();
        if (Opts_.ShardComparisonFlags.ContainFlag(EShardComparisonFlags::TYPES_SUMMARY)) {
            WriteSummary("summary type", Summary_, s);
            s.NewLine();
            WriteSummary("histogram type", Histogram_, s);
            s.NewLine();
            WriteSummary("log histogram type", LogHistogram_, s);
            s.NewLine();
        }

        if (Missing_) {
            s.Stream() << Missing_;
            s.NewLine();
        }

        if (Comparison_) {
            s.Stream() << Comparison_;
            s.NewLine();
        }

        if (TypeMismatch_) {
            s.Stream() << TypeMismatch_;
            s.NewLine();
        }

        OnComparisonEnd(s);
    }

private:
    struct TValue {
        ITimeseriesPtr Expected;
        ITimeseriesPtr Actual;

        TStockpileIds ExpectedId{0, 0};
        TStockpileIds ActualId{0, 0};

        TString Summary;
        TString Missing;
        TString TypeMismatch;
        TShardComparisonSummaryOutput Result;
    };

    static bool Defined(ITimeseries* ts) {
        if (ts && ts->PointCount() > 0u) {
            return true;
        }

        return false;
    }

    void Compare(const NMonitoring::TLabels& labels, TValue& value) {
        compareMon->IncProgress(1u);

        if (!Defined(value.Expected.get()) && !Defined(value.Actual.get())) {
            return;
        }

        if (Defined(value.Expected.get()) && !Defined(value.Actual.get())) {
            if (value.Actual.get()) {
                OnEmptyMetric(labels, value.Result.Actual, value.Result.Expected, Opts_.Actual.Name, value);
            } else {
                OnMissingMetric(labels, value.Result.Actual, value.Result.Expected, Opts_.Actual.Name, value);
            }

            return;
        }

        if (!Defined(value.Expected.get()) && Defined(value.Actual.get())) {
            if (value.Expected.get()) {
                OnEmptyMetric(labels, value.Result.Expected, value.Result.Actual, Opts_.Expected.Name, value);
            } else {
                OnMissingMetric(labels, value.Result.Expected, value.Result.Actual, Opts_.Expected.Name, value);
            }

            return;
        }

        Y_ENSURE(Defined(value.Expected.get()) && Defined(value.Actual.get()));

        if (value.Expected->Type() != value.Actual->Type()) {
            OnTypeMismatch(labels, value);
            return;
        }

        CompareImpl(labels, value);
    }

    void CompareImpl(const NMonitoring::TLabels& labels, TValue& value) {
        Y_ENSURE(Defined(value.Expected.get()) && Defined(value.Actual.get()));

        TStringOutput stream(value.Summary);
        TLineStream lineStream(stream);

        lineStream.Tab();
        if (Opts_.LinkMaker && Opts_.ShardComparisonFlags.ContainFlag(EShardComparisonFlags::STORAGE_LINK)) {
            lineStream.AddTrigger(TStringBuilder() << "(" << Opts_.Expected.Name << ") " << Opts_.LinkMaker->MakeLink(labels, Opts_.Expected.Cluster));
            lineStream.AddTrigger(TStringBuilder() << "(" << Opts_.Actual.Name << ") " << Opts_.LinkMaker->MakeLink(labels, Opts_.Actual.Cluster));
        }

        TSeriesComparisonOptions opts {
                .ComparisonFlags = Opts_.SeriesComparisonFlags,
                .Expected = Opts_.Expected.Name,
                .Actual = Opts_.Actual.Name,
                .Stream = &lineStream,
                .Comparator = Opts_.Comparator.Get(),
        };

        auto res = CompareSeries(*value.Expected, *value.Actual, labels, opts);

        MergeSeriesResult(value.Result, res);
    }

    void MergeSeriesResult(TShardComparisonSummaryOutput& shardRes, const TSeriesComparisonResult& res) {
        ++shardRes.CommonMetrics;
        shardRes.CommonPoints += res.CommonPoints;
        shardRes.EqualPoints += res.EqualPoints;

        if (res.CommonPoints == res.EqualPoints) {
            ++shardRes.EqualMetrics;
        }

        MergeSeriesResult(shardRes.Actual, res.Actual);
        MergeSeriesResult(shardRes.Expected, res.Expected);
    }

    void MergeSeriesResult(TShardComparisonSummaryOutput::TItem& shardRes, const TSeriesComparisonResult::TShardResult& res) {
        ++shardRes.AliveMetrics;

        shardRes.Points += res.TotalPoints;
        shardRes.MissingPoints += res.MissingPoints;

        if (res.TotalPoints != res.MissingPoints) {
            ++shardRes.MetricsWithMissingPoints;
        }
    }

    void OnEmptyMetric(
            const NMonitoring::TLabels& labels,
            TShardComparisonSummaryOutput::TItem& missing,
            TShardComparisonSummaryOutput::TItem& existing,
            TStringBuf name,
            TValue& value)
    {
        ++missing.EmptyMetrics;
        ++existing.AliveMetrics;

        if (!Opts_.ShardComparisonFlags.ContainFlag(EShardComparisonFlags::MISSING_METRICS)) {
            return;
        }

        TStringOutput ss(value.Missing);
        TLineStream s(ss);

        s.Tab();
        s.PrintLine(TStringBuilder() << "(" << name << ") empty metric: " << labels);
    }

    void OnMissingMetric(
            const NMonitoring::TLabels& labels,
            TShardComparisonSummaryOutput::TItem& missing,
            TShardComparisonSummaryOutput::TItem& existing,
            TStringBuf name,
            TValue& value)
    {
        ++missing.MissingMetrics;
        ++existing.AliveMetrics;

        if (!Opts_.ShardComparisonFlags.ContainFlag(EShardComparisonFlags::MISSING_METRICS)) {
            return;
        }

        TStringOutput ss(value.Missing);
        TLineStream s(ss);

        s.Tab();
        s.PrintLine(TStringBuilder() << "(" << name << ") missing metric: " << labels);
    }

    void OnTypeMismatch(const NMonitoring::TLabels& labels, TValue& value) {
        ++value.Result.MetricsWithDifferentTypes;
        ++value.Result.Actual.AliveMetrics;
        ++value.Result.Expected.AliveMetrics;

        if (!Opts_.ShardComparisonFlags.ContainFlag(EShardComparisonFlags::TYPE_MISMATCH)) {
            return;
        }

        TStringOutput ss(value.TypeMismatch);
        TLineStream s(ss);

        s.Tab();
        s.PrintLine(TStringBuilder()
            << "different types: " << labels << ": (" << Opts_.Expected.Name << ") "
            << value.Expected->Type() << " != "
            << "(" << Opts_.Actual.Name << ") " << value.Actual->Type());
    }


    void OnComparisonBegin(TLineStream& stream) {
        stream.PrintLine("comparison results: {");
        stream.Tab();
    }

    void OnComparisonEnd(TLineStream& stream) {
        stream.UnTab();
        stream.PrintLine("}");
    }

    void Merge(TValue& value) {
        if (value.Missing) {
            Missing_ += value.Missing;
        }

        if (value.Summary && Opts_.ShardComparisonFlags.ContainFlag(EShardComparisonFlags::SERIES_DIFF)) {
            Comparison_ += "\n\n" + value.Summary;
        }

        if (value.TypeMismatch) {
            TypeMismatch_ += TypeMismatch_;
        }

        if (value.Result.MetricsWithDifferentTypes) {
            ++Total_.MetricsWithDifferentTypes;
        }

        if (value.Result.CommonMetrics) {
            switch (value.Expected->Type()) {
                case NMonitoring::EMetricType::DSUMMARY: {
                    Merge(value.Result, Summary_);
                    break;
                }

                case NMonitoring::EMetricType::HIST: {
                    Merge(value.Result, Histogram_);
                    break;
                }

                case NMonitoring::EMetricType::LOGHIST: {
                    Merge(value.Result, LogHistogram_);
                    break;
                }

                default: {
                    ythrow yexception() << "unknown type: " << value.Expected->Type();
                }
            }
        }

        Merge(value.Result, Total_);
    }

    void Merge(const TShardComparisonSummaryOutput& from, TShardComparisonSummaryOutput& to) {
        to.EqualMetrics += from.EqualMetrics;
        to.EqualPoints += from.EqualPoints;
        to.CommonPoints += from.CommonPoints;
        to.CommonMetrics += from.CommonMetrics;

        Merge(from.Actual, to.Actual);
        Merge(from.Expected, to.Expected);
    }

    void Merge(const TShardComparisonSummaryOutput::TItem& from, TShardComparisonSummaryOutput::TItem& to) {
        to.MissingPoints += from.MissingPoints;
        to.MetricsWithMissingPoints += from.MetricsWithMissingPoints;
        to.Points += from.Points;
        to.AliveMetrics += from.AliveMetrics;
        to.MissingMetrics += from.MissingMetrics;
        to.EmptyMetrics += from.EmptyMetrics;
    }

    template <class TValue>
    static void WriteField(TStringBuf cluster, TStringBuf name, TValue value, TLineStream& stream) {
        stream.PrintLine(TStringBuilder() << "(" << cluster << ") " << name << ": " << value);
    }

    template <class TValue>
    static void WriteField(TStringBuf name, TValue value, TLineStream& stream) {
        stream.PrintLine(TStringBuilder() << name << ": " << value);
    }

    void WriteSummary(TStringBuf name, const TShardComparisonSummaryOutput& summary, TLineStream& stream) {
        stream.ResetWritten();
        stream.DropTriggers();

        stream.PrintLine(TStringBuilder() << name << " comparison result: {");
        stream.Tab();


        WriteField("number of common metrics", summary.CommonMetrics, stream);
        WriteField("number of equal metrics", summary.EqualMetrics, stream);

        stream.NewLine();
        WriteField("number of common points", summary.CommonPoints, stream);
        WriteField("number of equal metrics", summary.EqualPoints, stream);

        if (summary.MetricsWithDifferentTypes) {
            stream.NewLine();
            WriteField("number of metrics with different type", summary.MetricsWithDifferentTypes, stream);
        }

        stream.NewLine();
        WriteClusterSummary(Opts_.Expected.Name, summary.Expected, stream);
        WriteClusterSummary(Opts_.Actual.Name, summary.Actual, stream);

        stream.UnTab();
        stream.PrintLine("}");
    }

    void WriteClusterSummary(TStringBuf cluster, const TShardComparisonSummaryOutput::TItem& summary, TLineStream& stream) {
        WriteField(cluster, "number of alive metrics", summary.AliveMetrics, stream);
        WriteField(cluster, "number of missing metrics", summary.MissingMetrics, stream);
        WriteField(cluster, "number of empty metrics", summary.EmptyMetrics, stream);
        WriteField(cluster, "number of points", summary.Points, stream);
        WriteField(cluster, "number of missing points", summary.MissingPoints, stream);
        WriteField(cluster, "number of metrics with missing points", summary.MetricsWithMissingPoints, stream);
    }

private:
    TShardComparisonOptions Opts_;

    TShardComparisonSummaryOutput Summary_;
    TShardComparisonSummaryOutput Histogram_;
    TShardComparisonSummaryOutput LogHistogram_;
    TShardComparisonSummaryOutput Total_;

    IProgressMonitorPtr compareMon;
    IProgressMonitorPtr mergeMon;

    TString Missing_;
    TString Comparison_;
    TString TypeMismatch_;

    static constexpr size_t BATCH_SIZE = 1024 * 16;

    std::unordered_map<NMonitoring::TLabels, TValue, THash<NMonitoring::TLabels>> Map_;
    IEnginePtr Engine_;
};


IShardComparisonSummaryPtr CreateShardComparisonSummary(
        TShardComparisonOptions config,
        IEnginePtr engine)
{
    return MakeHolder<TShardComparisonSummary>(std::move(config), std::move(engine));
}
