#include "metabase_helper.h"

#include <solomon/tools/data-comparison/lib/util/wait_context.h>

#include <solomon/services/dataproxy/lib/metabase/rpc.h>

#include <solomon/libs/cpp/grpc/status/code.h>
#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/host_resolver/host_resolver.h>
#include <solomon/libs/cpp/error_or/error_or.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <memory>

using namespace NSolomon;
using namespace NDataProxy;
using namespace yandex::solomon::metabase;
using namespace yandex::solomon::model;
using namespace yandex::solomon::config;

constexpr ui16 METABASE_DEFAULT_PORT = 5710;

static MatchType MatcherTypeToProto(EMatcherType type, bool negative) {
    switch (type) {
        case EMatcherType::EXACT: return negative ? MatchType::NOT_EXACT : MatchType::EXACT;
        case EMatcherType::REGEX: return negative ? MatchType::NOT_REGEX : MatchType::REGEX;
        case EMatcherType::ANY: return MatchType::ANY;
        case EMatcherType::ABSENT: return MatchType::ABSENT;

        case EMatcherType::GLOB:
        case EMatcherType::MULTI:
            return negative ? MatchType::NOT_GLOB : MatchType::GLOB;
    }
}

static void FillMetabaseConfig(TStringBuf hosts, rpc::TGrpcClientConfig* metabaseConfig) {
    TAddressSet addresses;
    ConductorGroupResolver()->Resolve(hosts, &addresses);

    for (const auto& address: addresses) {
        if (TString::npos == address.rfind(':')) {
            metabaseConfig->add_addresses(TStringBuilder{} << address << ':' << METABASE_DEFAULT_PORT);
        } else {
            metabaseConfig->add_addresses(address);
        }
    }

    auto setTimeoutSeconds = [](auto* timeout, int value) {
        timeout->set_value(value);
        timeout->set_unit(TimeUnit::SECONDS);
    };

    setTimeoutSeconds(metabaseConfig->mutable_readtimeout(), 20);
    setTimeoutSeconds(metabaseConfig->mutable_connecttimeout(), 5);
}

template <class T>
TMeta ConvertToMeta(const T& m) {
    TMeta meta;
    for (int j = 0; j < m.labels_size(); ++j) {
        const auto& l = m.labels(j);
        meta.Labels.Add(l.key(), l.value());
    }
    meta.Id = TStringBuilder() << m.metric_id().shard_id() << TStringBuf("/") << m.metric_id().local_id();
    meta.TypeStr = MetricType_Name(m.type());
    return meta;
}

TString SelectorsToStr(const NSolomon::TSelectors& selectors) {
    TStringBuilder s;
    s << "{";
    for (size_t i = 0; i < selectors.size(); ++i) {
        s << selectors[i].Key() << "=" << selectors[i].Pattern();
        if (i + 1 != selectors.size()) {
            s << ",";
        }
    }
    s  << "}\n";

    return s;
}

class TMetaProvider: public IMetaProvider {
public:
    TAsyncTMBResponse GetMeta(NSolomon::TSelectors selectors, bool verbose) override {
        Y_ENSURE(Rpc_, "rpc is not opened");
        CorrectProject(selectors, Hosts_, PrePrefix_);

        FindRequest request;
        for (const auto& s: selectors) {
            auto* sProto = request.add_selectors();
            sProto->set_key(TString{s.Key()});
            sProto->set_pattern(TString{s.Pattern()});
            sProto->set_match_type(MatcherTypeToProto(s.Type(), s.Negative()));
        }

        const auto& addresses = Rpc_->Addresses();

        auto responses = std::make_shared<TVector<TAsyncFindResponse>>();
        responses->reserve(addresses.size());
        for (const auto& address: addresses) {
            if (verbose) {
                Cerr << "sending request to " << address << Endl;
            }
            auto* nodeRpc = Rpc_->Get(address);
            responses->emplace_back(nodeRpc->Find(request));
        }

        auto doneFuture = NThreading::WaitAll(*responses).Apply([responses = std::move(responses), wc = WC_, this](const auto& f) {
            Y_UNUSED(f);
            TMBResponseBuilder response;
            const auto& addresses = Rpc_->Addresses();
            for (size_t i = 0; i < responses->size(); ++i) {
                const auto& responseOrError = responses->at(i).GetValueSync();
                if (responseOrError.Fail()) {
                    const auto& error = responseOrError.Error();
                    TStringBuilder builder;
                    builder << "got error response from " << addresses[i]
                            << " { rpcCode: " << NGrpc::StatusCodeToString(error.RpcCode)
                            << ", metabaseCode: " << EMetabaseStatusCode_Name(error.MetabaseCode)
                            << ", message: " << error.Message
                            << " }"
                            << "\n";
                    response.OnError(builder);
                } else {
                    const auto& r = responseOrError.Value();
                    for (const auto& m: r->metrics()) {
                        response.OnSuccess(ConvertToMeta(m));
                    }
                }
            }

            return response.Build();
        });

        return doneFuture;
    }

    void Open(TStringBuf hosts, TStringBuf prePrefix) override {
        Y_ENSURE(hosts == PRE || hosts == PROD_SAS || hosts == PROD_VLA, "unexpected hosts" << hosts);
        Y_ENSURE(!Rpc_, "rpc is opened already");
        rpc::TGrpcClientConfig metabaseConfig;
        FillMetabaseConfig(hosts, &metabaseConfig);
        Rpc_ = CreateMetabaseRpc(metabaseConfig, Registry_);
        WC_ = MakeWaitContext();
        Hosts_ = hosts;
        PrePrefix_ = prePrefix;
    }

    void SyncClose() {
        if (IsClosed()) {
            return;
        }
        auto f = WC_->GetFuture();
        WC_.Reset(nullptr);
        f.Wait();
        Rpc_ = {};
    }

    void Close() override {
        SyncClose();
    }

    ~TMetaProvider() {
        SyncClose();
    }

private:
    static constexpr TStringBuf PROD_PREFIX = "yasm_";

    static void CorrectProject(TSelectors& selectors, TStringBuf hosts, TStringBuf prePrefix) {
        auto projectIt = selectors.Find(NLabels::LABEL_PROJECT);
        Y_ENSURE(projectIt != selectors.end(), "please provide project selector, selectors:" << SelectorsToStr(selectors));
        Y_ENSURE(projectIt->Type() == EMatcherType::EXACT, "project selector must have exact matcher");
        Y_ENSURE(hosts == PRE || hosts == PROD_SAS || hosts == PROD_VLA, "hosts must be either prod or pre conductor");

        TStringBuf value = projectIt->Pattern();
        Y_ENSURE(value.StartsWith(PROD_PREFIX) || value.StartsWith(prePrefix), "Invalid project:" << value);

        if (value.StartsWith(prePrefix) && hosts == PRE) {
            return;
        }
        if (!value.StartsWith(prePrefix) && value.StartsWith(PROD_PREFIX) && (hosts == PROD_SAS || hosts == PROD_VLA)) {
            return;
        }

        if (value.StartsWith(prePrefix)) {
            value = TString(PROD_PREFIX) + TString(value.data() + prePrefix.size(), value.end());
        } else {
            value = TString(prePrefix) + TString(value.data() + PROD_PREFIX.size(), value.end());
        }

        selectors.Override(NLabels::LABEL_PROJECT, value);
    }

    bool IsClosed() const {
        return !WC_;
    }

private:
    IMetabaseClusterRpcPtr Rpc_;
    NMonitoring::TMetricRegistry Registry_;
    TString Hosts_;
    TWaitContextPtr WC_;
    TString PrePrefix_;
};

IMetaProviderPtr MakeMetaProvider() {
    return MakeHolder<TMetaProvider>();
}
