#pragma once

#include <util/generic/string.h>

#include <library/cpp/threading/future/future.h>
#include <library/cpp/monlib/metrics/labels.h>

#include <solomon/libs/cpp/selectors/selectors.h>

class TMBResponseBuilder;

struct TMeta {
    NMonitoring::TLabels Labels;
    TString TypeStr;
    TString Id;
};

class TMBResponse {
public:
    TStringBuf GetErrors() const {
        return Errors_;
    }

    auto begin() const {
        return Metas_.begin();
    }

    auto end() const {
        return Metas_.end();
    }

    const TVector<TMeta>& AsVector() const {
        return Metas_;
    }

    friend class TMBResponseBuilder;

private:
    TVector<TMeta> Metas_;
    TString Errors_;

    TMBResponse(TVector<TMeta> metas, TString errors)
        : Metas_(std::move(metas))
        , Errors_(std::move(errors))
    {}
};

class TMBResponseBuilder {
public:
    void OnSuccess(TMeta response) {
        Metas_.emplace_back(std::move(response));
    }

    void OnError(TStringBuf error) {
        Errors_ += error;
    }

    TMBResponse Build() {
        return TMBResponse(std::move(Metas_), std::move(Errors_));
    }
private:
    TVector<TMeta> Metas_;
    TString Errors_;
};

using TAsyncTMBResponse = NThreading::TFuture<TMBResponse>;

class IMetaProvider {
public:
    static constexpr TStringBuf PRE = "conductor_group://solomon_pre_fetcher";
    static constexpr TStringBuf PROD_SAS = "conductor_group://solomon_prod_fetcher_sas";
    static constexpr TStringBuf PROD_VLA = "conductor_group://solomon_prod_fetcher_sas";

public:
    virtual TAsyncTMBResponse GetMeta(NSolomon::TSelectors selectors, bool verbose) = 0;

    virtual void Open(TStringBuf hosts, TStringBuf prePrefix) = 0;

    virtual void Close() = 0;

    virtual ~IMetaProvider() = default;
};

using IMetaProviderPtr = THolder<IMetaProvider>;

IMetaProviderPtr MakeMetaProvider();
