#include "monitor.h"

#include <util/generic/yexception.h>

class TProgressMonitor: public IProgressMonitor, public std::enable_shared_from_this<TProgressMonitor> {
public:
    explicit TProgressMonitor(TStringBuf body)
            : Body_(body)
    {
    }

    void Start() override {
        auto val = Started_.fetch_add(1u) + 1;
        if (val == Count_.load()) {
            Write(0, Processed_.load());
        }
    }

    void IncTotal(size_t delta) override {
        Y_ENSURE(!Started());
        Total_.fetch_add(delta);
    }

    void IncProgress(size_t delta) override {
        size_t old = Processed_.fetch_add(delta);

        Write(old, delta);
    }

    std::shared_ptr<IProgressMonitor> Share() override {
        Count_.fetch_add(1u);
        return shared_from_this();
    }

private:
    bool Started() const {
        return Started_.load() >= Count_.load();
    }

    void Write(size_t old, size_t delta) {
        if (!Started()) {
            return;
        }

        size_t now = GetProgress(old + delta);

        if (old && GetProgress(old) == now) {
            return;
        }

        Cerr << Body_ << " progress: " << now << "%\n";
    }

    size_t GetProgress(size_t processed) const {
        if (Total_.load() == processed) {
            return 100u;
        }

        return static_cast<size_t>((static_cast<double>(processed) / Total_.load()) * 100);
    }

private:
    std::atomic<size_t> Total_{0};
    std::atomic<size_t> Processed_{0};
    std::atomic<size_t> Started_{0};

    std::atomic<size_t> Count_{1u};

    TString Body_;
};

IProgressMonitorPtr CreateProgressMonitor(TStringBuf body) {
    return std::make_shared<TProgressMonitor>(body);
}
