#include "resolver.h"

#include <library/cpp/yaml/as/tstring.h>

#include <util/generic/hash.h>
#include <util/folder/iterator.h>
#include <util/stream/file.h>

#include <string>

class THostResolver: public IHostResolver {
public:
    THostResolver()
            : DcMatcher_(NSolomon::CreateDefaultIpv6Matcher())
            , DnsClient_(NSolomon::CreateDnsClient())
    {}

    NThreading::TFuture<TIpv6AddressesSet> ResolveIp(const TString& host) override {
        return DnsClient_->GetAddresses(host, true).Apply([g = Lock()](NThreading::TFuture<TIpv6AddressesSet> f) {
            return f;
        });
    }

    NThreading::TFuture<TDcResolveResult> ResolveDc(const TString& host) override {
        return  ResolveIp(host).Apply([this](auto f) {
            return ResolveDc(std::move(f));
        });
    }

private:
    NSolomon::IDcMatcherPtr DcMatcher_;
    NSolomon::IDnsClientPtr DnsClient_;

    TDcResolveResult ResolveDc(NThreading::TFuture<TIpv6AddressesSet> f) {
        TIpv6AddressesSet ips;
        try {
            ips = f.ExtractValueSync();
        } catch (const NSolomon::TDnsRequestTimeoutError&) {
            return TDcResolveResult{NSolomon::EDc::UNKNOWN, EDcResolveStatus ::TIMEOUT};
        } catch (const NSolomon::TDnsRecordNotFound&) {
            return TDcResolveResult{NSolomon::EDc::UNKNOWN, EDcResolveStatus::NOT_FOUND};
        } catch(...) {
            return TDcResolveResult{NSolomon::EDc::UNKNOWN, EDcResolveStatus::UNKNOWN};
        }

        if (ips.size() != 1u) {
            return TDcResolveResult{NSolomon::EDc::UNKNOWN, EDcResolveStatus::UNKNOWN};
        }

        for (const auto& ip: ips) {
            return TDcResolveResult{DcMatcher_->DcByAddress(ip), EDcResolveStatus::OK};
        }

        Y_FAIL("unreachable code");
    }

    TIntrusivePtr<THostResolver> Lock() {
        return TIntrusivePtr<THostResolver>(this);
    }
};

IHostResolverPtr MakeHostResolver() {
    return MakeIntrusive<THostResolver>();
}

TString DcToStr(NSolomon::EDc dc) {
    switch (dc) {
        case NSolomon::EDc::UNKNOWN: {
            return "unknown";
        }
        case NSolomon::EDc::AMS: {
            return "ams";
        }
        case NSolomon::EDc::ASH: {
            return "ash";
        }
        case NSolomon::EDc::FOL: {
            return "fol";
        }
        case NSolomon::EDc::IVA: {
            return "iva";
        }
        case NSolomon::EDc::MAN: {
            return "man";
        }
        case NSolomon::EDc::MYT: {
            return "myt";
        }
        case NSolomon::EDc::SAS: {
            return "sas";
        }
        case NSolomon::EDc::UGR: {
            return "ugr";
        }
        case NSolomon::EDc::VEG: {
            return "veg";
        }
        case NSolomon::EDc::VLA: {
            return "vla";
        }
        default: {
            return "unknown";
        }
    }
}

NSolomon::EDc StrToDc(TStringBuf dcStr) {
    if (dcStr == TStringBuf("unknown")) {
        return NSolomon::EDc::UNKNOWN;
    }
    if (dcStr == TStringBuf("ams")) {
        return NSolomon::EDc::AMS;
    }
    if (dcStr == TStringBuf("man")) {
        return NSolomon::EDc::MAN;
    }
    if (dcStr == TStringBuf("ash")) {
        return NSolomon::EDc::ASH;
    }
    if (dcStr == TStringBuf("fol")) {
        return NSolomon::EDc::FOL;
    }
    if (dcStr == TStringBuf("iva")) {
        return NSolomon::EDc::IVA;
    }
    if (dcStr == TStringBuf("myt")) {
        return NSolomon::EDc::MYT;
    }
    if (dcStr == TStringBuf("sas")) {
        return NSolomon::EDc::SAS;
    }
    if (dcStr == TStringBuf("ugr")) {
        return NSolomon::EDc::UGR;
    }
    if (dcStr == TStringBuf("veg")) {
        return NSolomon::EDc::VEG;
    }
    if (dcStr == TStringBuf("vla")) {
        return NSolomon::EDc::VLA;
    }

    ythrow yexception() << "invalid dc name: " << dcStr;
}

class TGroupResolver: public IGroupResolver {
public:
    TGroupResolver(const TString& configsDir) {
        LoadTable(configsDir);
        CheckDc();
    }

    NSolomon::EDc ResolveDc(TStringBuf group) override {
        return Group2Metagroup_[group].Dc;
    }

    TString ResolveMetagroup(TStringBuf group) override {
        return Group2Metagroup_[group].Metagroup;
    }

    std::unordered_map<std::string, std::vector<std::string>> Metagroups() const override {
        std::unordered_map<std::string, std::vector<std::string>> dump;
        for (const auto& [group, value]: Group2Metagroup_) {
            dump[value.Metagroup.data()].emplace_back(group.data());
        }

        return dump;
    }

private:
    static constexpr size_t MAX_RETRIES = 10;
    struct TValue {
        TString Metagroup{};
        NSolomon::EDc Dc{NSolomon::EDc::UNKNOWN};
    };

    THashMap<TString, TValue> Group2Metagroup_;

    void LoadTable(const TString& configsDir) {
        auto hostsResolver = MakeHostResolver();
        TVector<NThreading::TFuture<void>> futures;
        for (const auto& f: TDirIterator(configsDir)) {
            if (f.fts_pathlen == f.fts_namelen || f.fts_info != FTS_F) {
                continue;
            }

            VerifyFileName(f.fts_path);
            futures.emplace_back(HandleYamlFile(f.fts_path, hostsResolver.Get()));
        }

        auto all = NThreading::WaitAll(futures);
        all.Wait();
    }

    NThreading::TFuture<void> HandleYamlFile(TStringBuf file, IHostResolver* hostsResolver) {
        TFileInput in(TString{file});
        TString data = in.ReadAll();
        YAML::Node config = YAML::Load(std::string(data.data(), data.size()));

        TVector<NThreading::TFuture<void>> futures;

        TString metaGroup = config["front_properties"]["metagroup"].as<TString>();
        Y_ENSURE(metaGroup, "empty metagroup");

        for (const auto& group: config["instances"]) {
            TString g = group["aggr_name"].as<TString>();
            Y_ENSURE(g, "empty group");
            auto [it, _] = Group2Metagroup_.emplace(std::make_pair(g, TValue{metaGroup}));

            for (const auto& host: group["hosts"]) {
                futures.emplace_back(ResolveDc(hostsResolver, host.as<TString>(), it->second));
                break;
            }
        }

        return NThreading::WaitAll(futures);
    }

    static void VerifyFileName(TStringBuf name) {
        TStringBuf rPart;
        TStringBuf _;
        name.RSplit('.', _, rPart);

        if (rPart != TStringBuf("yaml")) {
            ythrow yexception() << "file from configs dir \"" << name << "\" must have YAML extension";
        }
    }

    static NThreading::TFuture<void> ResolveDc(IHostResolver* resolver, const TString& host, TValue& value, size_t retryNum = 0) {
        if (retryNum == MAX_RETRIES) {
            ythrow yexception() << "cannot resolve host, retries limit exceeded:" << host << '\n';
        }
        return resolver->ResolveDc(host).Apply([resolver, &host, &value, retryNum](auto f) {
            auto v = f.ExtractValueSync();
            if (v.Status == EDcResolveStatus::TIMEOUT) {
                auto future = ResolveDc(resolver, host, value, retryNum + 1);
                future.Wait();
                return;
            }
            if (v.Dc == NSolomon::EDc::UNKNOWN) {
                ythrow yexception() << "cannot resolve host:" << host << '\n';
            }

            value.Dc = v.Dc;
        });
    }

    void CheckDc() {
        for (const auto& [key, value]: Group2Metagroup_) {
            Y_ENSURE(value.Dc != NSolomon::EDc::UNKNOWN, "not able to resolve " << key << " dc");
        }
    }
};

IGroupResolverPtr MakeGroupResolver(const TString& confogsDir) {
    return MakeIntrusive<TGroupResolver>(confogsDir);
}
