#include "stockpile_client.h"

#include <solomon/tools/data-comparison/lib/util/wait_context.h>

#include <solomon/services/dataproxy/lib/stockpile/rpc.h>
#include <solomon/services/dataproxy/lib/timeseries/protobuf.h>

#include <solomon/libs/cpp/host_resolver/host_resolver.h>
#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/stream/null.h>
#include <util/stream/format.h>

constexpr ui16 STOCKPILE_DEFAULT_PORT = 5700;

using namespace NSolomon;
using namespace yandex::solomon;

static config::rpc::TGrpcClientConfig RpcConfig(TStringBuf group, ui16 port, TDuration connectionTimeout) {
    TAddressSet addresses;
    ConductorGroupResolver()->Resolve(group, &addresses);

    config::rpc::TGrpcClientConfig config;
    for (const auto& address: addresses) {
        config.add_addresses(TStringBuilder{} << address << ':' << port);
    }

    auto setTimeoutSeconds = [](auto* timeout, int value) {
        timeout->set_value(value);
        timeout->set_unit(config::TimeUnit::SECONDS);
    };

    setTimeoutSeconds(config.mutable_readtimeout(), connectionTimeout.Seconds());
    setTimeoutSeconds(config.mutable_connecttimeout(), 5);
    return config;
}

template <typename T>
TString ExceptionMessage(const TErrorOr<T, NDataProxy::TStockpileError>& responseOrError) {
    if (responseOrError.Fail()) {
        return TString(TStringBuilder() << "cannot read data " << responseOrError.Error());
    }

    const T& response = responseOrError.Value();
    stockpile::EStockpileStatusCode status = response.GetStatus();
    if (status != stockpile::EStockpileStatusCode::OK) {
        return TString(TStringBuilder()
            << "cannot read data " << NDataProxy::TStockpileError(grpc::StatusCode::OK, status, response.GetStatusMessage()));
    }
    Y_ENSURE(status == stockpile::EStockpileStatusCode::OK,
             "cannot read data " << NDataProxy::TStockpileError(grpc::StatusCode::OK, status, response.GetStatusMessage()));

    return {};
}

static TString ClusterToHosts(ECluster cluster) {
    switch (cluster) {
        case ECluster::PRE: {
            return "conductor_group://solomon_pre_stockpile";
        }
        case ECluster::PROD_SAS: {
            return "conductor_group://solomon_prod_stockpile_sas";
        }
        case ECluster::PROD_VLA: {
            return "conductor_group://solomon_prod_stockpile_vla";
        }
        default: {
            return "";
        }
    }
}

class TStockpileClient: public IStockpileClient {
public:
    void Open(ECluster cluster, TDuration connectionTimeout) override {
        Y_ENSURE(!Rpc_, "rpc is opened already");
        TString hosts = ClusterToHosts(cluster);
        Y_ENSURE(hosts, "unknown cluster:" << ClusterToStr(cluster));

        config::rpc::TGrpcClientConfig rpcConfig = RpcConfig(hosts, STOCKPILE_DEFAULT_PORT, connectionTimeout);
        Rpc_ = NDataProxy::CreateStockpileRpc(rpcConfig, Registry_);
        LoadShardLocations();
        WC_ = MakeWaitContext();
    }

    TAsyncReadManyResponse CompressedReadMany(ui32 shardId, const TVector<ui64>& localIds, ui64 fromMillis, ui64 toMillis) override {
        TStringBuf address = ShardLocations_[shardId];
        Y_ENSURE(address, "cannot find shard " << shardId << " location");

        stockpile::TReadManyRequest readRequest;
        readRequest.SetBinaryVersion(static_cast<i32>(NStockpile::EFormat::IDEMPOTENT_WRITE_38));
        readRequest.SetShardId(shardId);
        for (auto v: localIds) {
            readRequest.AddLocalIds(v);
        }

        if (fromMillis) {
            readRequest.SetFromMillis(fromMillis);
        }
        if (toMillis) {
            readRequest.SetToMillis(toMillis);
        }

        return CompressedReadMany(Rpc_->Get(address), readRequest);
    }

    TAsyncReadMetricsMetaResponse ReadMetricsMeta(ui32 shardId, const TVector<ui64>& localIds) override {
        TStringBuf address = ShardLocations_[shardId];
        Y_ENSURE(address, "cannot find shard " << shardId << " location");

        stockpile::ReadMetricsMetaRequest request;
        request.SetShardId(shardId);
        for (auto localId: localIds) {
            request.AddLocalIds(localId);
        }

        return ReadMetricsMeta(Rpc_->Get(address), request);
    }

    void SyncClose() {
        if (IsClosed()) {
            return;
        }
        auto f = WC_->GetFuture();
        WC_.Reset(nullptr);
        f.Wait();
        Rpc_ = {};
    }

    void Close() override {
        SyncClose();
    }

    ~TStockpileClient() {
        SyncClose();
    }

private:
    TAsyncCompressedReadManyResponse CompressedReadMany(
            NDataProxy::IStockpileRpc* nodeRpc,
            const stockpile::TReadManyRequest& readRequest)
    {
        auto future = nodeRpc->ReadCompressedMany(readRequest);

        return future.Apply([wc = WC_](auto f) {
            auto value = f.ExtractValue();
            TString error = ExceptionMessage(value);
            if (error) {
                return TReadManyResponseOrError::FromError(std::move(error));
            }

            return TReadManyResponseOrError::FromValue(FromProto(value.Value()));
        });
    }

    TAsyncReadMetricsMetaResponse ReadMetricsMeta(
            NDataProxy::IStockpileRpc* nodeRpc,
            const stockpile::ReadMetricsMetaRequest& request)
    {
        NDataProxy::TAsyncReadMetricsMetaResponse future = nodeRpc->ReadMetricsMeta(request);

        return future.Apply([wc = WC_](auto f) {
            auto value = f.ExtractValue();
            TString error = ExceptionMessage(value);
            if (error) {
                return TReadMetricsMetaResponseOrError::FromError(std::move(error));
            }

            return TReadMetricsMetaResponseOrError::FromValue(FromProto(value.Value()));
        });
    }

    static TCompressedReadManyResponse FromProto(const stockpile::TCompressedReadManyResponse& response) {
        TCompressedReadManyResponse r;
        r.reserve(response.metrics().size());

        for (const auto& metric: response.metrics()) {
            // there is no better way here to get fallback type for metric
            auto type = NSolomon::FromProto(metric.type());
            TSeriesAndId m{
                .Id = TStockpileIds{metric.GetShardId(), metric.GetLocalId()},
                .Series = NSolomon::NDataProxy::FromProto(type, metric),
            };

            r.emplace_back(std::move(m));
        }

        return r;
    }

    static TReadMetricsMetaResponse FromProto(const stockpile::ReadMetricsMetaResponse& response) {
        TReadMetricsMetaResponse meta;
        if (response.GetStatus() != stockpile::EStockpileStatusCode::OK) {
            ythrow yexception() << TString(response.GetStatusMessage().data(), response.GetStatusMessage().size());
        }

        meta.reserve(response.get_arr_meta().size());
        for (const auto& m: response.get_arr_meta()) {
            meta.emplace_back();
            meta.back().LocalId = m.GetlocalId();
            meta.back().LastTsMillis = TInstant::MilliSeconds(m.GetlastTsMillis());
        }

        return meta;
    }

    bool IsClosed() const {
        return !WC_;
    }

    void LoadShardLocations() {
        stockpile::TServerStatusRequest serverStatusRequest;
        serverStatusRequest.set_deadline(TDuration::Seconds(20).ToDeadLine().MilliSeconds());

        std::vector<NDataProxy::TAsyncStockpileStatusResponse> statusFutures;
        const auto& addresses = Rpc_->Addresses();
        for (TStringBuf address: addresses) {
            auto* nodeRpc = Rpc_->Get(address);
            statusFutures.push_back(nodeRpc->ServerStatus(serverStatusRequest));
        }

        for (auto done = NThreading::WaitExceptionOrAll(statusFutures); !done.Wait(TDuration::Seconds(1)); ) {
        }

        ShardLocations_.clear();

        for (size_t i = 0; i < statusFutures.size(); ++i) {
            const auto& responseOrError = statusFutures[i].GetValueSync();
            if (responseOrError.Fail()) {
                Cerr << "cannot get status response from " << addresses[i] << ' ' << responseOrError.Error() << Endl;
                continue;
            }

            const stockpile::TServerStatusResponse& response = responseOrError.Value();
            if (response.GetStatus() != stockpile::EStockpileStatusCode::OK) {
                NDataProxy::TStockpileError error{grpc::StatusCode::OK, response.GetStatus(), response.GetStatusMessage()};
                Cerr << "cannot get status response from " << addresses[i] << ' ' << error << Endl;
                continue;
            }

            for (const auto& shardStatus: response.GetShardStatus()) {
                ShardLocations_[shardStatus.GetShardId()] = addresses[i];
            }
        }
    }

private:
    NDataProxy::IStockpileClusterRpcPtr Rpc_;
    std::unordered_map<ui32, TStringBuf> ShardLocations_;
    NMonitoring::TMetricRegistry Registry_;
    TWaitContextPtr WC_;
};

IStockpileClientPtr MakeStockpileClient() {
    return MakeIntrusive<TStockpileClient>();
}
