#pragma once

#include <library/cpp/monlib/metrics/labels.h>

#include <util/string/split.h>
#include <util/stream/file.h>
#include <util/generic/vector.h>
#include <util/datetime/base.h>

class TIO {
public:
    class TReadVisitor {
    public:
        virtual void OnLabels(const NMonitoring::TLabels&) {}
        virtual void OnShardId(ui32) {}
        virtual void OnLocalId(ui64) {}
        virtual void OnTime(TInstant) {}
        virtual void OnStr(TStringBuf) {}

        virtual ~TReadVisitor() = default;
    };

    static void ReadMetric(TStringBuf schema, TStringBuf line, TReadVisitor& visitor) {
        auto schemaIterable = StringSplitter(schema.begin(), schema.end()).Split('|');
        auto lineIterable = StringSplitter(line.begin(), line.end()).Split('|');

        auto schemaIt = schemaIterable.begin();
        auto lineIt = lineIterable.begin();

        while (schemaIt != schemaIterable.end() && lineIt != lineIterable.end()) {
            if (schemaIt->Token()) {
                HandleToken(schemaIt->Token(), lineIt->Token(), visitor);
            }

            ++lineIt;
            ++schemaIt;
        }

        Y_ENSURE(
                lineIt == lineIterable.end() && schemaIt == schemaIterable.end(),
                "line:" << line << " don't match schema:" << schema);
    }

    static ui64 ParseLocalId(TStringBuf localIdStr) {
        ui64 localId;
        Y_ENSURE_EX(TryFromString(localIdStr, localId), TBadArgumentException()
                << "cannot parse localId from: " << localIdStr);
        return localId;
    }

    static ui32 ParseShardId(TStringBuf shardIdStr) {
        return ParseLocalId(shardIdStr);
    }

    static TInstant ParseTime(TStringBuf timeStr) {
        static const ui64 NOT_BEFORE = TInstant::ParseIso8601("2000-01-01T00:00:00Z").MilliSeconds();
        static const ui64 NOT_AFTER = TInstant::ParseIso8601("2038-01-19T03:14:07Z").MilliSeconds();

        try {
            ui64 ts = TInstant::ParseIso8601(timeStr).MilliSeconds();
            if (ts >= NOT_BEFORE && ts <= NOT_AFTER) {
                return TInstant::MilliSeconds(ts);
            }
        } catch (...) {
        }

        ui64 ts = std::stoull(std::string(timeStr.data(), timeStr.data() + timeStr.size()));

        if (ts < NOT_BEFORE) {
            ts *= 1000u;
        }

        if (ts >= NOT_BEFORE && ts <= NOT_AFTER) {
            return TInstant::MilliSeconds(ts);
        }

        ythrow yexception() << "invalid ts:" << timeStr << '\n';
    }

    static NMonitoring::TLabels ParseLabels(TStringBuf labelsStr) {
        NMonitoring::TLabels labels;
        try {
            if (NMonitoring::TryLoadLabelsFromString(labelsStr, labels)) {
                return labels;
            }
        } catch (...) {
            // labels have &label&label&... format
        }

        for (auto it: StringSplitter(labelsStr.begin(), labelsStr.end()).Split('&')) {
            TStringBuf labelStr = it.Token();
            if (!labelStr) {
                continue;
            }
            labels.Add(NMonitoring::TLabel::FromString(it.Token()));
        }

        return labels;
    }

    static std::pair<ui32, ui64> ParseMetricId(TStringBuf str) {
        TStringBuf shardIdStr, localIdStr;
        Y_ENSURE_EX(str.TrySplit('/', shardIdStr, localIdStr), TBadArgumentException()
                << "invalid metricId format (must be <shardId>/<localId>), but got: " << str);

        ui32 shardId;
        Y_ENSURE_EX(TryFromString(shardIdStr, shardId), TBadArgumentException()
                << "cannot parse shardId from: " << shardIdStr);

        ui64 localId;
        Y_ENSURE_EX(TryFromString(localIdStr, localId), TBadArgumentException()
                << "cannot parse localId from: " << localIdStr);

        return {shardId, localId};
    }

private:
    static void HandleToken(TStringBuf token, TStringBuf data, TReadVisitor& visitor) {
        if (token == "l") {
            visitor.OnLabels(ParseLabels(data));
        } else if (token == "sid") {
            visitor.OnShardId(ParseShardId(data));
        } else if (token == "lid") {
            visitor.OnLocalId(ParseLocalId(data));
        } else if (token == "t") {
            visitor.OnTime(ParseTime(data));
        } else if (token == "s") {
            visitor.OnStr(data);
        } else {
            ythrow yexception() << "invalid schema token:" << token;
        }
    }
};
