package main

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net"
	"os"
	"os/signal"
	"syscall"
	"time"

	"a.yandex-team.ru/solomon/libs/go/filelogger"

	"a.yandex-team.ru/solomon/tools/discovery/internal/bus"
	"a.yandex-team.ru/solomon/tools/discovery/internal/config"
	"a.yandex-team.ru/solomon/tools/discovery/internal/units"
	"a.yandex-team.ru/solomon/tools/discovery/internal/unroller"

	"a.yandex-team.ru/solomon/tools/discovery/proto"
)

// ==========================================================================================

// Config with default values
//
var Config = config.MainConfig{
	GRPCAddr:               "[::]:8011",
	ServiceAddr:            "[::]:8010",
	DNSAddr:                "[::]:5353",
	DNSTTL:                 units.Duration{Duration: 30 * time.Second},
	ClientTimeout:          units.Duration{Duration: 5 * time.Second},
	SafeChangeUpFraction:   0.5,
	SafeChangeUpCount:      10,
	SafeChangeDownFraction: 0.2,
	SafeChangeDownCount:    2,
	GrowthIsAlwaysSafe:     true,
	EmptyGroupIsOk:         true,
	BestEffortUnroll:       false,
	LogFileSize:            units.Size{Size: 10000000},
	LogFileCount:           20,
	VerboseLevel:           1,
	CacheDumpFile:          "/dev/shm/discovery.db",
	CacheDumpFileCount:     24,
	CacheDumpInterval:      units.Duration{Duration: 60 * time.Minute},
	DataSaveDir:            "/dev/shm/discovery",
	DataUpdateInterval:     units.Duration{Duration: 60 * time.Second},
	UseIPDC:                true,
	ConductorCacheConfig: config.CacheConfig{
		CacheGoodTime:   units.Duration{Duration: 10 * time.Minute},
		CacheBadTime:    units.Duration{Duration: 30 * time.Second},
		PrefetchTime:    units.Duration{Duration: 60 * time.Second},
		CleanUpInterval: units.Duration{Duration: 20 * time.Second},
		RequestTimeout:  units.Duration{Duration: 1 * time.Second},
		CacheMaxSize:    500,
		Workers:         10,
		ServeStale:      true,
	},
	KubeCacheConfig: config.CacheConfig{
		CacheGoodTime:   units.Duration{Duration: 2 * time.Minute},
		CacheBadTime:    units.Duration{Duration: 5 * time.Second},
		PrefetchTime:    units.Duration{Duration: 10 * time.Second},
		CleanUpInterval: units.Duration{Duration: 5 * time.Second},
		RequestTimeout:  units.Duration{Duration: 5 * time.Second},
		CacheMaxSize:    500,
		Workers:         10,
		ServeStale:      true,
	},
	EdsCacheConfig: config.CacheConfig{
		CacheGoodTime:   units.Duration{Duration: 2 * time.Minute},
		CacheBadTime:    units.Duration{Duration: 5 * time.Second},
		PrefetchTime:    units.Duration{Duration: 10 * time.Second},
		CleanUpInterval: units.Duration{Duration: 5 * time.Second},
		RequestTimeout:  units.Duration{Duration: 5 * time.Second},
		CacheMaxSize:    500,
		Workers:         10,
		ServeStale:      true,
	},
	ResolverConfig: config.CacheConfig{
		CacheGoodTime:   units.Duration{Duration: 10 * time.Minute},
		CacheBadTime:    units.Duration{Duration: 20 * time.Second},
		PrefetchTime:    units.Duration{Duration: 10 * time.Second},
		CleanUpInterval: units.Duration{Duration: 5 * time.Second},
		CacheMaxSize:    5000,
		Workers:         30,
		ServeStale:      true,
	},
	IPDCConfig: config.CacheConfig{
		CacheGoodTime:   units.Duration{Duration: 6 * time.Hour},
		CacheBadTime:    units.Duration{Duration: 60 * time.Second},
		PrefetchTime:    units.Duration{Duration: 40 * time.Second},
		CleanUpInterval: units.Duration{Duration: 10 * time.Second},
		RequestTimeout:  units.Duration{Duration: 1 * time.Second},
		CacheMaxSize:    10,
		ServeStale:      true,
	},
}

// ==========================================================================================

func readConfig(path string) error {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		return err
	}
	if err = json.Unmarshal(data, &Config); err != nil {
		return err
	}
	return nil
}

// ==========================================================================================

func main() {
	log.SetFlags(log.Ldate | log.Ltime | log.Lmicroseconds)
	if len(os.Args) != 2 {
		log.Fatalf("Usage: %s <config_file>", os.Args[0])
	}

	configFileName := os.Args[1]
	if err := readConfig(configFileName); err != nil {
		log.Fatalf("Bad config file %s: %s", configFileName, err.Error())
	}

	// Setup logs
	if Config.LogFile == "" {
		filelogger.SetStdLogger()
	} else {
		filelogger.SetFileLogger(Config.LogFile, Config.LogFileSize.Size, Config.LogFileCount)
	}
	log.Println("=========================================================================================")
	log.Println("Start!")

	// Start unroll
	unroll, err := unroller.NewUnroller(&Config)
	if err != nil {
		log.Fatalf("Failed to create unroller: %v", err)
	}

	// Start DNS server
	dns, err := NewDNSServer(Config.DNSAddr, Config.VerboseLevel)
	if err != nil {
		log.Fatalf("Failed to create DNS server: %v", err)
	}

	// Create main processor
	proc, err := NewProcessor(&Config, unroll)
	if err != nil {
		log.Fatalf("Failed to create processor: %v", err)
	}
	proc.AddOnUpdateCallback(func(_ []*pbData.DataRef) {
		dns.SetDataClean(proc.GetDNSMap(), uint32(Config.DNSTTL.Duration.Seconds()))
	})

	// Create cluster gbus
	var gbus *bus.Bus
	if Config.MyCluster == nil {
		log.Println("My cluster is not specified in config, not starting GRPC bus server")
	} else {
		getCluster := func() (map[string]net.IP, error) {
			emptyOk := false
			// XXX
			// If best effort is used here, result could be empty
			bestEffortUnroll := false
			hds, _, err := unroll.GetHostDataList(Config.MyCluster, "", nil, emptyOk, bestEffortUnroll)
			if err != nil {
				return nil, fmt.Errorf("failed to unroll my cluster nodes: %v", err)
			}
			cluster := map[string]net.IP{}
			for _, hd := range hds {
				cluster[hd.FQDN] = hd.Address
			}
			return cluster, nil
		}

		gbus, err = bus.NewBus(Config.GRPCAddr, getCluster, Config.VerboseLevel)
		if err != nil {
			log.Fatalf("Failed to start bus: %v", err)
		}
		proc.AddOnUpdateCallback(func(refs []*pbData.DataRef) {
			// Must be async to avoid distributed deadlocks
			_, _ = gbus.DoNotifyClients(refs, true)
		})
		gbus.AddOnNotifyCallback(func(refs []*pbData.DataRef, name string) {
			proc.ForceUpdate(refs, name)
		})
	}

	sigChan := make(chan os.Signal, 2)
	signal.Notify(sigChan, syscall.SIGTERM, syscall.SIGINT, syscall.SIGHUP, syscall.SIGUSR1, syscall.SIGUSR2)

	// Start http server
	server := HTTPServer(&Config, proc, dns)
	go func() {
		log.Printf("[http] server %v", server.ListenAndServe())
		sigChan <- syscall.SIGHUP
	}()

	// Block until a signal is received
	for {
		s := <-sigChan
		log.Printf("Got signal=%v", s)
		if s == syscall.SIGUSR1 {
			if err := proc.Save(); err != nil {
				log.Printf("Failed to save discovery data, %v", err)
			}
		} else if s == syscall.SIGUSR2 {
			if err := unroll.Dump(); err != nil {
				log.Printf("Failed to dump discovery db, %v", err)
			}
		} else {
			break
		}
	}

	// Shutdown all
	if err := server.Shutdown(context.Background()); err != nil {
		log.Printf("Failed while shutting HTTP server down, %v", err)
	}
	proc.Shutdown()
	if gbus != nil {
		gbus.Shutdown()
	}
	if err := dns.Shutdown(); err != nil {
		log.Printf("Failed while shutting DNS server down, %v", err)
	}
	unroll.Shutdown()

	log.Printf("Done!")
}
