package main

import (
	"log"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/solomon/tools/discovery/internal/config"
	"a.yandex-team.ru/solomon/tools/discovery/internal/metrics"
)

// ==========================================================================================

type HTTPManager struct {
	LogPrefix    string
	VerboseLevel int
	processor    *Processor
	dns          *DNSServer
}

func NewHTTPManager(p *Processor, d *DNSServer, verboseLevel int) *HTTPManager {
	return &HTTPManager{
		LogPrefix:    "[http] ",
		VerboseLevel: verboseLevel,
		processor:    p,
		dns:          d,
	}
}

func (h *HTTPManager) log(lvl int, ts *time.Time, format string, v ...interface{}) {
	if h.VerboseLevel >= lvl {
		tsStr := ""
		if ts != nil {
			tsStr = ", " + time.Since(*ts).String()
		}
		log.Printf(h.LogPrefix+format+tsStr, v...)
	}
}

func (h *HTTPManager) httpHandler(w http.ResponseWriter, r *http.Request) {
	reqTime := time.Now()
	h.log(2, nil, "connection from %s: %s", r.RemoteAddr, r.URL.Path)

	// Common response generator
	resp := func(data []byte, contentType string, status int, format string, v ...interface{}) {
		w.Header().Set("Content-type", contentType)
		w.Header().Set("Connection", "Close")
		w.WriteHeader(status)
		if status >= http.StatusBadRequest {
			h.log(0, &reqTime, "request failed "+strconv.Itoa(status)+" ("+r.RemoteAddr+"): "+format, v...)
		} else {
			h.log(1, &reqTime, "request ok "+strconv.Itoa(status)+" ("+r.RemoteAddr+"): "+format, v...)
		}
		if _, err := w.Write(data); err != nil {
			h.log(0, &reqTime, "failed to send response %d (%s): %v", status, r.RemoteAddr, err)
		} else {
			_, _ = w.Write([]byte{'\n'})
		}
	}

	// Filter out bad requests
	if len(r.URL.Path) > 256 || len(r.URL.RawQuery) > 128 {
		resp([]byte("too long"), "text/plain", http.StatusRequestURITooLong, "URI too long path=%s query=%s", r.URL.Path, r.URL.RawQuery)
		return
	}

	// Parse path
	idxPrev := 0
	elems := []string{}
	for idx, c := range r.URL.Path {
		if c == '/' {
			if idx > 0 {
				elems = append(elems, r.URL.Path[idxPrev:idx])
			}
			idxPrev = idx + 1
		} else if ('a' > c || c > 'z') && ('A' > c || c > 'Z') && ('0' > c || c > '9') && strings.IndexByte(":._-", byte(c)) < 0 {
			resp([]byte("bad chars in URL"), "text/plain", http.StatusBadRequest, "URI has bad char=%v path=%s", c, r.URL.Path)
			return
		}
	}
	if idxPrev < len(r.URL.Path) {
		elems = append(elems, r.URL.Path[idxPrev:])
	}

	// Parse query
	query, err := url.ParseQuery(r.URL.RawQuery)
	if err != nil {
		resp([]byte("bad query"), "text/plain", http.StatusBadRequest, "bad query, %v", err)
		return
	}

	// Reply based on
	// - elems
	// - query
	// https://st.yandex-team.ru/SOLOMON-8218
	// discovery/<env>/<service>[:<port>][?dc=xxx&port=yyy]
	//
	if len(elems) == 0 {
		resp([]byte("empty path"), "text/plain", http.StatusNotFound, "empty path")
	} else if elems[0] == "metrics" {
		data := metrics.MetricsSum(h.processor.GetMetrics(), h.dns.GetMetrics()).Bytes()
		resp(data, "application/json", http.StatusOK, "metrics response len=%d", len(data))
	} else if elems[0] == "discovery" {
		var env, service, rtype, port, dc string

		if len(elems) > 1 {
			env = elems[1]
		}
		if len(elems) > 2 {
			service = elems[2]

			// Parse service, port
			idx := strings.IndexRune(service, ':')
			if idx > 0 {
				service, port = service[:idx], service[idx+1:]
			}
			idx = strings.IndexRune(service, '.')
			if idx > 0 {
				service, rtype = service[:idx], service[idx+1:]
			} else {
				rtype = "json"
			}

			// Set variables from query
			if dcList, ok := query["dc"]; ok && len(dcList) > 0 {
				dc = dcList[0]
			}
			if portList, ok := query["port"]; ok && len(portList) > 0 {
				port = portList[0]
			}
		}
		reqTarget := "env=" + env + " service=" + service + " rtype=" + rtype + " port=" + port + " dc=" + dc

		if env == "" {
			if data, err := h.processor.GetDataStruct(); err == nil {
				// discovery -> <env>/<service> map
				//
				resp(data, "application/json", http.StatusOK, "%s, data map response len=%d", reqTarget, len(data))
			} else {
				resp([]byte(err.Error()), "text/plain", http.StatusNotFound, "%s, %v", reqTarget, err)
			}
		} else {
			if service == "" {
				if data, err := h.processor.GetServicesByEnv(env); err == nil {
					// discovery/<env> -> list of services for <env>
					//
					resp(data, "application/json", http.StatusOK, "%s, env response len=%d", reqTarget, len(data))
				} else if data, err2 := h.processor.GetEnvsByService(env); err2 == nil {
					// discovery/<service> -> list of environments for <service>
					//
					resp(data, "application/json", http.StatusOK, "%s, services slice response len=%d", reqTarget, len(data))
				} else {
					resp([]byte(err.Error()), "text/plain", http.StatusNotFound, "%s, %v, %v", reqTarget, err, err2)
				}
			} else if rtype == "list" {
				if data, err := h.processor.GetHostsList(env, service, dc); err == nil {
					// discovery/<env>/<service>.list -> hosts list
					//
					resp(data, "text/plain", http.StatusOK, "%s, hosts list response len=%d", reqTarget, len(data))
				} else {
					resp([]byte(err.Error()), "text/plain", http.StatusNotFound, "%s, %v", reqTarget, err)
				}
			} else {
				if data, err := h.processor.GetDiscoveryData(env, service, dc, port); err == nil {
					// discovery/<env>/<service>(.json)?(:port)? -> discovery json
					//
					resp(data, "application/json", http.StatusOK, "%s, discovery response len=%d", reqTarget, len(data))
				} else {
					resp([]byte(err.Error()), "text/plain", http.StatusNotFound, "%s, %v", reqTarget, err)
				}
			}
		}
	} else {
		resp([]byte("not found"), "text/plain", http.StatusNotFound, "path="+r.URL.Path)
	}
}

// ==========================================================================================

func HTTPServer(c *config.MainConfig, p *Processor, d *DNSServer) *http.Server {
	m := NewHTTPManager(p, d, c.VerboseLevel)

	return &http.Server{
		Addr:           c.ServiceAddr,
		Handler:        http.HandlerFunc(m.httpHandler),
		ReadTimeout:    c.ClientTimeout.Duration,
		WriteTimeout:   c.ClientTimeout.Duration,
		MaxHeaderBytes: 1 << 14,
	}
}
