package conductor

import (
	"context"
	"log"
	"time"

	"a.yandex-team.ru/solomon/libs/go/cache"
	"a.yandex-team.ru/solomon/libs/go/conductor"
	"a.yandex-team.ru/solomon/libs/go/workerpool"
)

// ==========================================================================================

type ConductorData struct {
	HostsList []string
	Error     error
}

type workerJob struct {
	Data          *ConductorData
	Request       string
	MinLastUpdate *time.Time
}

// ==========================================================================================

// Conductor cache. Able to serve stale items and prefetch.
//
// type ConductorCache interface {
//     Get(groups []string) map[string]*ConductorData
//     Purge()
//     Destroy()
//     Dump() ([]byte, error)
//     Restore([]byte) error
// }
//

type ConductorCache struct {
	LogPrefix    string
	VerboseLevel int
	workerpool   *workerpool.WorkerPool
	cache        *cache.Cache
	client       *conductor.Client
}

func NewConductorCache(goodCacheTime, badCacheTime, prefetchTime, cleanUpInterval, requestTimeout time.Duration,
	cacheSize, workers int,
	serveStale bool,
	verboseLevel int) *ConductorCache {

	c := &ConductorCache{
		LogPrefix:    "[conductor] ",
		VerboseLevel: verboseLevel,
		client:       conductor.NewClientWithTimeout(requestTimeout),
	}
	c.cache = cache.NewCache(
		"conductor",
		func(req interface{}) (interface{}, error) {
			return c.cacheConductor(req.(string))
		},
		goodCacheTime,
		badCacheTime,
		prefetchTime,
		cleanUpInterval,
		serveStale,
		verboseLevel,
		cacheSize,
	)
	c.workerpool = workerpool.NewWorkerPool(
		"conductor",
		workers,
		func(req interface{}) {
			c.poolWorker(req.(*workerJob))
		},
		verboseLevel > 1,
	)
	return c
}

func (c *ConductorCache) log(lvl int, ts *time.Time, format string, v ...interface{}) {
	if c.VerboseLevel >= lvl {
		tsStr := ""
		if ts != nil {
			tsStr = ", " + time.Since(*ts).String()
		}
		log.Printf(c.LogPrefix+format+tsStr, v...)
	}
}

func (c *ConductorCache) cacheConductor(name string) ([]string, error) {
	hosts, err := c.client.GroupToHosts(context.Background(), name)
	if err != nil {
		return nil, err
	}
	response := make([]string, 0, len(hosts))
	for _, host := range hosts {
		if len(host) > 0 {
			response = append(response, host)
		}
	}
	return response, nil
}

func (c *ConductorCache) poolWorker(job *workerJob) {
	var resp interface{}
	var cacheError error

	if job.MinLastUpdate != nil {
		resp, cacheError = c.cache.GetForceFresh(job.Request, job.MinLastUpdate)
	} else {
		resp, cacheError = c.cache.Get(job.Request)
	}
	job.Data.HostsList = resp.([]string)
	job.Data.Error = cacheError
}

func (c *ConductorCache) Get(groups []string, minLastUpdate *time.Time) map[string]*ConductorData {
	reqTime := time.Now()
	jobs := make([]interface{}, len(groups))
	result := make(map[string]*ConductorData, len(groups))

	for i, group := range groups {
		d := &ConductorData{}
		jobs[i] = &workerJob{
			Request:       group,
			Data:          d,
			MinLastUpdate: minLastUpdate,
		}
		result[group] = d
	}
	c.workerpool.Do(jobs)
	c.log(2, &reqTime, "unrolled %d records", len(groups))

	return result
}

func (c *ConductorCache) Purge() {
	c.log(1, nil, "purging")
	c.cache.Purge()
}

func (c *ConductorCache) Destroy() {
	c.log(1, nil, "destroying")
	c.workerpool.Stop(true)
	c.cache.Destroy()
}

func (c *ConductorCache) Dump(onlyFresh bool) ([]byte, error) {
	c.log(1, nil, "dumping %d records (only fresh = %v)", c.cache.Len(), onlyFresh)
	return c.cache.Dump(onlyFresh)
}

func (c *ConductorCache) Restore(inData []byte) error {
	c.log(1, nil, "restoring")
	bumpEOL := false
	err := c.cache.Restore(inData, bumpEOL, "", []string{})
	if err != nil {
		return err
	}
	c.log(1, nil, "restored %d records", c.cache.Len())
	return nil
}
