package holder

import (
	"bytes"
	"encoding/json"
	"fmt"
	"hash"
	"hash/crc64"
	"log"
	"net"
	"os"
	"sort"
	"sync"
	"time"

	"a.yandex-team.ru/solomon/tools/discovery/internal/unroller"
)

// ==========================================================================================

func mkDir(p string, mode os.FileMode) error {
	if stat, err := os.Stat(p); os.IsNotExist(err) {
		if err := os.MkdirAll(p, 0755); err != nil {
			return err
		}
	} else if err != nil {
		return err
	} else if !stat.IsDir() {
		return fmt.Errorf("%s exists and is not a directory", p)
	}
	if mode != 0 {
		if err := os.Chmod(p, mode); err != nil {
			return err
		}
	}
	return nil
}

// ==========================================================================================

type dataHash struct {
	h hash.Hash64
}

func NewDataHash() *dataHash {
	return &dataHash{
		h: crc64.New(crc64.MakeTable(crc64.ISO)),
	}
}

func (d *dataHash) GetHash(hds []*unroller.HostData, ports map[string]int) uint64 {
	d.h.Reset()

	for _, hd := range hds {
		_, _ = d.h.Write([]byte(hd.FQDN))
		_, _ = d.h.Write(hd.Address)
		_, _ = d.h.Write([]byte(hd.Cluster))
	}
	keys := make([]string, 0, len(ports))
	for name := range ports {
		keys = append(keys, name)
	}
	sort.Strings(keys)
	for _, name := range keys {
		_, _ = d.h.Write([]byte(name))
		p := ports[name]
		_, _ = d.h.Write([]byte{byte(p >> 8), byte(p)})
	}
	return d.h.Sum64()
}

// ==========================================================================================

type DiscoveryFormat struct {
	EnvType     string               `json:"envType"`
	ServiceName string               `json:"serviceName"`
	Ports       map[string]int       `json:"ports"`
	Hosts       []*unroller.HostData `json:"hosts"`
}

type Source struct {
	Ports     map[string]int                 // { <portName>: <portNumber>, ... }
	PortsInv  map[string]string              // { <portNumberString>: <portName>, ... }
	Endpoints map[string]map[string][]string // { <dc>: { <src>: [ <srcData>, ... ], ... }, ... }
}

type HolderMetrics struct {
	HostsCount          uint64
	StaleSeconds        float64
	RequestsTotal       int64
	UpdatesTotal        int64
	UpdatesForced       int64
	UpdatesFailedUnsafe int64
	UpdatesFailedUnroll int64
}

type SafetyParams struct {
	SafeChangeUpFraction   float64
	SafeChangeDownFraction float64
	SafeChangeUpCount      int
	SafeChangeDownCount    int
	GrowthIsAlwaysSafe     bool
	EmptyGroupIsOk         bool
	BestEffortUnroll       bool
}

type CacheKey struct {
	DC   string
	Port int
}

type CacheValue struct {
	ListBytes      []byte
	DiscoveryBytes []byte
}

type HostsData struct {
	Hash  uint64
	Hosts []*unroller.HostData
}

// ==========================================================================================

type Holder struct {
	LogPrefix           string
	EnvType             string
	ServiceName         string
	VerboseLevel        int
	hash                uint64
	prevHash            uint64
	ports               map[string]int
	portsInv            map[string]string
	dcHosts             map[string]*HostsData // map from DC to HostsData
	eol                 time.Time
	bytesCache          map[CacheKey]*CacheValue
	dataSaveDir         string
	safetyParams        *SafetyParams
	updatesFailedUnsafe int64
	updatesFailedUnroll int64
	updatesForced       int64
	updatesTotal        int64
	requestsTotal       int64
	mutex               sync.RWMutex
	mutexUpdate         sync.Mutex
	dataHash            *dataHash
	unroller            *unroller.Unroller
}

func NewHolder(envType, serviceName string,
	dataSaveDir string,
	safetyParams *SafetyParams,
	unroller *unroller.Unroller,
	verboseLevel int) *Holder {

	h := &Holder{
		LogPrefix:    fmt.Sprintf("[holder] {%s %s} ", envType, serviceName),
		EnvType:      envType,
		ServiceName:  serviceName,
		VerboseLevel: verboseLevel,
		bytesCache:   make(map[CacheKey]*CacheValue),
		dataSaveDir:  dataSaveDir,
		safetyParams: safetyParams,
		dataHash:     NewDataHash(),
		unroller:     unroller,
	}
	return h
}

func (h *Holder) log(lvl int, ts *time.Time, format string, v ...interface{}) {
	if h.VerboseLevel >= lvl {
		tsStr := ""
		if ts != nil {
			tsStr = ", " + time.Since(*ts).String()
		}
		log.Printf(h.LogPrefix+format+tsStr, v...)
	}
}

// ==========================================================================================

func (h *Holder) GetHashes() (uint64, uint64) {
	h.mutex.RLock()
	defer h.mutex.RUnlock()

	return h.hash, h.prevHash
}

func (h *Holder) GetHolderMetrics() *HolderMetrics {
	h.mutex.RLock()
	defer h.mutex.RUnlock()

	hostsCount := 0
	for _, hd := range h.dcHosts {
		hostsCount += len(hd.Hosts)
	}
	m := &HolderMetrics{
		HostsCount:          uint64(hostsCount),
		RequestsTotal:       h.requestsTotal,
		UpdatesTotal:        h.updatesTotal,
		UpdatesForced:       h.updatesForced,
		UpdatesFailedUnsafe: h.updatesFailedUnsafe,
		UpdatesFailedUnroll: h.updatesFailedUnroll,
	}
	if !h.eol.IsZero() {
		// XXX
		// Could be negative, means someone forced updating, update failed and EOL remained unchanged
		m.StaleSeconds = time.Since(h.eol).Seconds()
	}
	return m
}

func (h *Holder) UpdateDNSMap(dnsMap map[string]net.IP) {
	h.mutex.RLock()
	defer h.mutex.RUnlock()

	for _, dh := range h.dcHosts {
		for _, h := range dh.Hosts {
			dnsMap[h.FQDN] = h.Address
		}
	}
}

// ==========================================================================================

// Return (updated, failed) flags
//
func (h *Holder) Update(src *Source, minLastUpdate *time.Time, safetyCheck bool) (bool, bool) {
	var updated bool
	var failedUnroll, failedUnsafe int64
	var newHash uint64

	reqTime := time.Now()

	// Lock for updates, one at a time
	h.mutexUpdate.Lock()
	defer h.mutexUpdate.Unlock()
	h.log(1, &reqTime, "start updating (forced=%v): got lock", minLastUpdate != nil)

	if minLastUpdate != nil {
		h.updatesForced++
	}

	// Reset EOL, otherwise it will be stuck forever
	minEOL := time.Time{}
	dcHosts := map[string]*HostsData{}

	for dc, srcMap := range src.Endpoints {
		udx := h.dcHosts[dc]
		// Dangerous setting!!!
		// Must be used only if the dc was never here before. Otherwise bad unroll could be considered as successful!
		//
		bestEffortUnroll := udx == nil && h.safetyParams.BestEffortUnroll

		hds, eol, err := h.unroller.GetHostDataList(
			srcMap,
			dc,
			minLastUpdate,
			h.safetyParams.EmptyGroupIsOk,
			bestEffortUnroll,
		)
		if err != nil {
			failedUnroll = 1
		} else {
			// udx == nil means no previous data present, safety check makes no sense
			if udx != nil && safetyCheck && !h.isSafeChange(len(udx.Hosts), len(hds)) {
				failedUnsafe = 1
				err = fmt.Errorf("unsafe hosts size change %d->%d", len(udx.Hosts), len(hds))
			}
		}

		if err != nil {
			if udx != nil {
				dcHosts[dc] = udx
				h.log(0, nil, "not updating dc=%s: %v", dc, err)
			} else {
				h.log(0, nil, "not adding new dc=%s: %v", dc, err)
				continue
			}
		} else {
			// Must sort keys to get same CRC
			sort.SliceStable(hds, func(i, j int) bool { return hds[i].FQDN < hds[j].FQDN })

			dcHosts[dc] = &HostsData{
				Hash:  h.dataHash.GetHash(hds, src.Ports),
				Hosts: hds,
			}
		}

		if eol != nil && (eol.Before(minEOL) || minEOL.IsZero()) {
			minEOL = *eol
		}
		newHash += dcHosts[dc].Hash
	}
	if newHash != h.hash {
		updated = true

		h.mutex.Lock()
		h.log(1, &reqTime, "storing metrics and data")

		// update relevant fields
		h.hash, h.prevHash = newHash, h.hash
		h.updatesTotal++
		h.updatesFailedUnroll += failedUnroll
		h.updatesFailedUnsafe += failedUnsafe
		h.ports = src.Ports
		h.portsInv = src.PortsInv
		h.dcHosts = dcHosts
		h.eol = minEOL
		h.bytesCache = make(map[CacheKey]*CacheValue)

		h.mutex.Unlock()
	}
	return updated, failedUnroll+failedUnsafe > 0
}

func (h *Holder) isSafeChange(curr, next int) bool {
	if curr == next {
		return true
	} else if curr > next {
		return float64(curr) < float64(next)*(1+h.safetyParams.SafeChangeDownFraction) || curr <= next+h.safetyParams.SafeChangeDownCount
	} else if h.safetyParams.GrowthIsAlwaysSafe {
		return true
	}
	return float64(curr) > float64(next)*(1-h.safetyParams.SafeChangeUpFraction) || curr >= next-h.safetyParams.SafeChangeUpCount
}

// ==========================================================================================

// GetBytes return ListBytes, DiscoveryBytes, error
//
func (h *Holder) GetBytes(dc, portStr string) ([]byte, []byte, error) {
	var ok bool
	var cv *CacheValue
	var hd *HostsData
	var port int

	h.mutex.RLock()
	defer h.mutex.RUnlock()
	h.requestsTotal++

	if portStr != "" {
		if port, ok = h.ports[portStr]; !ok {
			if portStr, ok = h.portsInv[portStr]; !ok {
				return nil, nil, fmt.Errorf("unknown port")
			} else if port, ok = h.ports[portStr]; !ok {
				// Should never happen
				panic("inconsistent port maps in Holder")
			}
		}
	}
	if dc != "" {
		if hd, ok = h.dcHosts[dc]; !ok {
			return nil, nil, fmt.Errorf("unknown dc")
		}
	} else if len(h.dcHosts) == 0 {
		// No DC was unrolled successfully
		return nil, nil, fmt.Errorf("no hosts found")
	}

	key := CacheKey{
		DC:   dc,
		Port: port,
	}
	if cv, ok = h.bytesCache[key]; !ok {
		cv = &CacheValue{}

		var err error
		var ports map[string]int
		hosts := []*unroller.HostData{}

		// Create HostData list
		if hd != nil {
			hosts = hd.Hosts
		} else {
			// All DCs are selected
			keys := make([]string, 0, len(h.dcHosts))
			for name := range h.dcHosts {
				keys = append(keys, name)
			}
			sort.Strings(keys)
			for _, name := range keys {
				hosts = append(hosts, h.dcHosts[name].Hosts...)
			}
		}

		// XXX Do we need return error if no hosts?
		// Upon agreement, we return empty list

		// Marshal hosts list
		listBytes := []byte{}
		for _, host := range hosts {
			listBytes = append(listBytes, []byte(host.FQDN+"\n")...)
		}

		// Get port map
		if port != 0 {
			ports = map[string]int{portStr: port}
		} else {
			// All ports are selected
			ports = h.ports
		}

		cv.ListBytes = bytes.TrimRight(listBytes, "\n")
		cv.DiscoveryBytes, err = json.MarshalIndent(&DiscoveryFormat{
			EnvType:     h.EnvType,
			ServiceName: h.ServiceName,
			Ports:       ports,
			Hosts:       hosts,
		}, "", "    ")
		if err != nil {
			// Since marshal error could be either UnsupportedTypeError or UnsupportedValueError
			// this will never happen for simple DiscoveryFormat struct
			panic("failed to marshal DiscoveryFormat structure")
		}
		h.bytesCache[key] = cv
	}
	return cv.ListBytes, cv.DiscoveryBytes, nil
}

// ==========================================================================================

func (h *Holder) Save() error {
	h.mutex.RLock()
	defer h.mutex.RUnlock()

	dir := h.dataSaveDir + "/" + h.EnvType
	fileName := dir + "/" + h.ServiceName + ".json"
	h.log(0, nil, "saving data to %s", fileName)

	if err := mkDir(dir, 0); err != nil {
		return err
	}
	f, err := os.OpenFile(fileName, os.O_CREATE|os.O_TRUNC|os.O_WRONLY, 0644)
	if err != nil {
		return err
	}
	defer func() {
		_ = f.Close()
	}()

	_, db, _ := h.GetBytes("", "")
	if _, err := f.Write(db); err != nil {
		return err
	}
	return nil
}
