package kube

import (
	"crypto/tls"
	"crypto/x509"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"net/url"
	"time"

	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/solomon/libs/go/cache"
	"a.yandex-team.ru/solomon/libs/go/workerpool"

	"a.yandex-team.ru/solomon/tools/discovery/internal/config"
)

// ==========================================================================================

type KubeEndpoint struct {
	APIVersion string `json:"apiVersion"`
	Kind       string `json:"kind"`
	Metadata   struct {
		Name              string `json:"name"`
		NameSpace         string `json:"namespace"`
		ResourceVersion   string `json:"resourceVersion"`
		UID               string `json:"uid"`
		CreationTimestamp string `json:"creationTimestamp"`
	} `json:"metadata"`
	Subsets []struct {
		Addresses []struct {
			IP        string `json:"ip"`
			NodeName  string `json:"nodeName"`
			TargetRef struct {
				Kind            string `json:"kind"`
				Name            string `json:"name"`
				NameSpace       string `json:"namespace"`
				ResourceVersion string `json:"resourceVersion"`
				UID             string `json:"uid"`
			} `json:"targetRef"`
		} `json:"addresses"`
	} `json:"subsets"`
}

type KubeConfigFile struct {
	APIVersion string `yaml:"apiVersion"`
	Kind       string `yaml:"kind"`
	Clusters   []struct {
		Name    string `yaml:"name"`
		Cluster struct {
			CertificateAuthorityData string `yaml:"certificate-authority-data"`
			Server                   string `yaml:"server"`
		} `yaml:"cluster"`
	} `yaml:"clusters"`
	Users []struct {
		Name string `yaml:"name"`
		User struct {
			Token                 string `yaml:"token"`
			ClientCertificateData string `yaml:"client-certificate-data"`
			ClientKeyData         string `yaml:"client-key-data"`
		} `yaml:"user"`
	} `yaml:"users"`
}

// ==========================================================================================

type KubeClient struct {
	dnsZone string
	server  string
	token   string
	client  *http.Client
}

func NewKubeClient(config *config.KubeConfig, timeout time.Duration) (*KubeClient, error) {
	var err error
	var certs []tls.Certificate
	var certificateAuthority []byte
	var clientCertificate []byte
	var clientKey []byte
	var kubeData KubeConfigFile

	if config.ConfigFileData == "" {
		if config.ConfigFilePath == "" {
			return nil, fmt.Errorf("cannot create kube http client from empty config")
		}
		data, err := ioutil.ReadFile(config.ConfigFilePath)
		if err != nil {
			return nil, fmt.Errorf("bad kube config file, %v", err)
		}
		config.ConfigFileData = string(data)
	}

	if err = yaml.Unmarshal([]byte(config.ConfigFileData), &kubeData); err != nil {
		return nil, fmt.Errorf("bad yaml in kube config, %v", err)
	}
	if kubeData.Kind != "Config" {
		return nil, fmt.Errorf("bad kube config, kind is '%s' and not 'Config'", kubeData.Kind)
	}

	k := &KubeClient{
		dnsZone: config.DNSZone,
	}
	for idx, user := range kubeData.Users {
		if user.Name == config.User || config.User == "" {
			p := &kubeData.Users[idx].User.ClientCertificateData
			if clientCertificate, err = base64.StdEncoding.DecodeString(*p); err != nil {
				return nil, fmt.Errorf("failed to decode client-certificate-data, %v", err)
			}
			p = &kubeData.Users[idx].User.ClientKeyData
			if clientKey, err = base64.StdEncoding.DecodeString(*p); err != nil {
				return nil, fmt.Errorf("failed to decode client-key-data, %v", err)
			}
			k.token = kubeData.Users[idx].User.Token
			break
		}
	}
	for idx, cluster := range kubeData.Clusters {
		if cluster.Name == config.Cluster || config.Cluster == "" {
			p := &kubeData.Clusters[idx].Cluster.CertificateAuthorityData
			if certificateAuthority, err = base64.StdEncoding.DecodeString(*p); err != nil {
				return nil, fmt.Errorf("failed to decode certificate-authority-data: %v", err)
			}
			k.server = kubeData.Clusters[idx].Cluster.Server
			break
		}
	}
	if len(k.token) == 0 {
		if len(clientCertificate) == 0 {
			return nil, fmt.Errorf("failed to get client-certificate from kube config")
		}
		if len(clientKey) == 0 {
			return nil, fmt.Errorf("failed to get client-key from kube config")
		}
		cert, err := tls.X509KeyPair(clientCertificate, clientKey)
		if err != nil {
			return nil, fmt.Errorf("failed to set TLS certificate, %v", err)
		}
		certs = []tls.Certificate{cert}
	}
	if len(certificateAuthority) == 0 {
		return nil, fmt.Errorf("failed to get certificate-authority from kube config")
	}
	if len(k.server) == 0 {
		return nil, fmt.Errorf("failed to get kuber server from kube config")
	}

	rootCAs := x509.NewCertPool()
	if ok := rootCAs.AppendCertsFromPEM(certificateAuthority); !ok {
		return nil, fmt.Errorf("failed to set root CA certificate, %v", err)
	}
	k.client = &http.Client{
		Transport: &http.Transport{
			TLSClientConfig: &tls.Config{
				Certificates:       certs,
				RootCAs:            rootCAs,
				InsecureSkipVerify: false,
			},
		},
		Timeout: timeout,
	}
	return k, nil
}

func (k *KubeClient) GetEndpoints(nameSpace, service string) (*KubeEndpoint, error) {
	ep := KubeEndpoint{}

	u, err := url.Parse(fmt.Sprintf("%s/api/v1/namespaces/%s/endpoints/%s?limit=8192", k.server, nameSpace, service))
	if err != nil {
		return nil, fmt.Errorf("failed to parse url: %v", err)
	}
	req := http.Request{
		URL:    u,
		Header: http.Header{"Accept": []string{"application/json"}},
		Close:  true,
	}
	if k.token != "" {
		req.Header.Add("Authorization", "Bearer "+k.token)
	}

	resp, err := k.client.Do(&req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	body, err := io.ReadAll(resp.Body)
	if err != nil {
		return nil, fmt.Errorf("failed read response body, %v", err)
	}
	if resp.StatusCode == http.StatusNotFound {
		return nil, fmt.Errorf("not found (404), %s", string(body))
	}
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("bad response (%d), %s", resp.StatusCode, string(body))
	}

	if err = json.Unmarshal(body, &ep); err != nil {
		return nil, fmt.Errorf("bad response, %v", err)
	}
	return &ep, nil
}

// ==========================================================================================

type KubeRequest struct {
	Zone      string
	Namespace string
	Service   string
}

type KubeData struct {
	Hosts map[string]string
	Error error
}

type workerJob struct {
	Request       *KubeRequest
	Data          *KubeData
	MinLastUpdate *time.Time
}

// ==========================================================================================

// Kube cache. Able to serve stale items and prefetch.
//
// type KubeCache interface {
//     Get(reqs []*KubeRequest) map[KubeRequest]*KubeData
//     Purge()
//     Destroy()
//     Dump() ([]byte, error)
//     Restore([]byte) error
// }
//

type KubeCache struct {
	LogPrefix    string
	VerboseLevel int
	workerpool   *workerpool.WorkerPool
	cache        *cache.Cache
	clients      map[string]*KubeClient
}

func NewKubeCache(configs []*config.KubeConfig,
	goodCacheTime, badCacheTime, prefetchTime, cleanUpInterval, requestTimeout time.Duration,
	cacheSize, workers int,
	serveStale bool,
	verboseLevel int) (*KubeCache, error) {

	kubeClients := make(map[string]*KubeClient, len(configs))
	for _, config := range configs {
		k, err := NewKubeClient(config, requestTimeout)
		if err != nil {
			return nil, err
		}
		kubeClients[config.DNSZone] = k
	}

	c := &KubeCache{
		LogPrefix:    "[kube] ",
		VerboseLevel: verboseLevel,
		clients:      kubeClients,
	}
	c.cache = cache.NewCache(
		"kube",
		func(req interface{}) (interface{}, error) {
			return c.cacheKube(req.(KubeRequest))
		},
		goodCacheTime,
		badCacheTime,
		prefetchTime,
		cleanUpInterval,
		serveStale,
		verboseLevel,
		cacheSize,
	)
	c.workerpool = workerpool.NewWorkerPool(
		"kube",
		workers,
		func(req interface{}) {
			c.poolWorker(req.(*workerJob))
		},
		verboseLevel > 1,
	)
	return c, nil
}

func (k *KubeCache) log(lvl int, ts *time.Time, format string, v ...interface{}) {
	if k.VerboseLevel >= lvl {
		tsStr := ""
		if ts != nil {
			tsStr = ", " + time.Since(*ts).String()
		}
		log.Printf(k.LogPrefix+format+tsStr, v...)
	}
}

func (k *KubeCache) cacheKube(req KubeRequest) (map[string]string, error) {
	reqTime := time.Now()
	kc, ok := k.clients[req.Zone]
	if !ok {
		return nil, fmt.Errorf("no kube client for zone=%s", req.Zone)
	}
	ke, err := kc.GetEndpoints(req.Namespace, req.Service)
	if err != nil {
		return nil, err
	}
	result := make(map[string]string)
	for _, subset := range ke.Subsets {
		for _, addrs := range subset.Addresses {
			name := fmt.Sprintf("%s.%s.%s.svc.%s", addrs.TargetRef.Name, req.Service, req.Namespace, req.Zone)
			result[name] = addrs.IP
		}
	}
	k.log(2, &reqTime, "got %d host records for %v", len(result), req)
	return result, nil
}

func (k *KubeCache) poolWorker(job *workerJob) {
	var resp interface{}
	var cacheError error

	if job.MinLastUpdate != nil {
		resp, cacheError = k.cache.GetForceFresh(*job.Request, job.MinLastUpdate)
	} else {
		resp, cacheError = k.cache.Get(*job.Request)
	}
	job.Data.Hosts = resp.(map[string]string)
	job.Data.Error = cacheError
}

func (k *KubeCache) Get(reqs []*KubeRequest, minLastUpdate *time.Time) map[KubeRequest]*KubeData {
	reqTime := time.Now()
	jobs := make([]interface{}, len(reqs))
	result := make(map[KubeRequest]*KubeData, len(reqs))

	for i, req := range reqs {
		d := &KubeData{}
		jobs[i] = &workerJob{
			Request:       req,
			Data:          d,
			MinLastUpdate: minLastUpdate,
		}
		result[*req] = d
	}
	k.workerpool.Do(jobs)
	k.log(2, &reqTime, "unrolled %d records", len(reqs))
	return result
}

func (k *KubeCache) Purge() {
	k.log(1, nil, "purging")
	k.cache.Purge()
}

func (k *KubeCache) Destroy() {
	k.log(1, nil, "destroying")
	k.workerpool.Stop(true)
	k.cache.Destroy()
}

func (k *KubeCache) Dump(onlyFresh bool) ([]byte, error) {
	k.log(1, nil, "dumping %d records (only fresh = %v)", k.cache.Len(), onlyFresh)
	return k.cache.Dump(onlyFresh)
}

func (k *KubeCache) Restore(inData []byte) error {
	k.log(1, nil, "restoring")
	bumpEOL := false
	err := k.cache.Restore(inData, bumpEOL, KubeRequest{}, map[string]string{})
	if err != nil {
		return err
	}
	k.log(1, nil, "restored %d records", k.cache.Len())
	return nil
}
