package metrics

import (
	"bytes"
	"compress/gzip"
	"encoding/json"
)

func MetricsSum(mm ...*Metrics) *Metrics {
	size := 0
	for _, m := range mm {
		size += len(m.MetricsBytes)
	}
	s := &Metrics{
		MetricsBytes: make([]byte, 0, size),
	}
	for _, m := range mm {
		s.MetricsBytes = append(s.MetricsBytes, m.MetricsBytes...)
	}
	return s
}

type Metric struct {
	Labels map[string]string `json:"labels"`
	Kind   string            `json:"kind"`
	Value  interface{}       `json:"value"`
}

type Metrics struct {
	MetricsBytes []byte
}

func NewMetrics() *Metrics {
	return &Metrics{
		MetricsBytes: make([]byte, 0),
	}
}

func (m *Metrics) Bytes() []byte {
	if m == nil {
		return []byte("{}")
	}
	b := []byte(`{"metrics":[`)
	if len(m.MetricsBytes) > 0 {
		b = append(b, m.MetricsBytes[:len(m.MetricsBytes)-1]...)
	}
	b = append(b, []byte(`]}`)...)

	return b
}

func (m *Metrics) ZBytes() ([]byte, error) {
	var buf bytes.Buffer

	cw, _ := gzip.NewWriterLevel(&buf, gzip.BestSpeed)
	if _, err := cw.Write(m.Bytes()); err != nil {
		return nil, err
	}
	if err := cw.Close(); err != nil {
		return nil, err
	}

	return buf.Bytes(), nil
}

func (m *Metrics) addMetric(kind string, inLabels map[string]string, name string, value interface{}) {
	labels := make(map[string]string, len(inLabels)+1)
	for k, v := range inLabels {
		labels[k] = v
	}
	labels["sensor"] = name
	metric := &Metric{
		Labels: labels,
		Kind:   kind,
		Value:  value,
	}

	b, err := json.Marshal(metric)
	if err != nil {
		b = []byte(`{"error": "cannot marshal this metric"}`)
	}
	b = append(b, ',')
	m.MetricsBytes = append(m.MetricsBytes, b...)
}

func (m *Metrics) AddCounter(inLabels map[string]string, name string, value int64) {
	m.addMetric("COUNTER", inLabels, name, value)
}

func (m *Metrics) AddRate(inLabels map[string]string, name string, value int64) {
	m.addMetric("RATE", inLabels, name, value)
}

func (m *Metrics) AddDGauge(inLabels map[string]string, name string, value float64) {
	m.addMetric("DGAUGE", inLabels, name, value)
}

func (m *Metrics) AddIGauge(inLabels map[string]string, name string, value uint64) {
	m.addMetric("IGAUGE", inLabels, name, value)
}
