package metrics

import (
	"bytes"
	"compress/gzip"
	"encoding/json"
)

func OldMetricsSum(mm ...*OldMetrics) *OldMetrics {
	size := 0
	for _, m := range mm {
		size += len(m.MetricsList)
	}
	s := &OldMetrics{
		MetricsList: make([]*OldMetric, 0, size),
	}
	for _, m := range mm {
		s.MetricsList = append(s.MetricsList, m.MetricsList...)
	}
	return s
}

type OldMetric struct {
	Labels map[string]string `json:"labels"`
	Kind   string            `json:"kind"`
	Value  interface{}       `json:"value"`
}

type OldMetrics struct {
	MetricsList []*OldMetric `json:"metrics"`
}

func NewOldMetrics() *OldMetrics {
	return &OldMetrics{
		MetricsList: make([]*OldMetric, 0),
	}
}

func (m *OldMetrics) Bytes() []byte {
	if m == nil {
		return []byte("{}")
	}
	b, err := json.Marshal(m)
	if err != nil {
		b, _ = json.Marshal(map[string]string{"error": err.Error()})
	}
	return b
}

func (m *OldMetrics) ZBytes() ([]byte, error) {
	var buf bytes.Buffer

	cw, _ := gzip.NewWriterLevel(&buf, gzip.BestSpeed)
	if _, err := cw.Write(m.Bytes()); err != nil {
		return nil, err
	}
	if err := cw.Close(); err != nil {
		return nil, err
	}

	return buf.Bytes(), nil
}

func (m *OldMetrics) addSensor(kind string, inLabels map[string]string, name string, value interface{}) {
	labels := make(map[string]string, len(inLabels))
	for k, v := range inLabels {
		labels[k] = v
	}
	labels["sensor"] = name
	metric := &OldMetric{
		Labels: labels,
		Kind:   kind,
		Value:  value,
	}

	m.MetricsList = append(m.MetricsList, metric)
}

func (m *OldMetrics) AddCounter(inLabels map[string]string, name string, value int64) {
	m.addSensor("COUNTER", inLabels, name, value)
}

func (m *OldMetrics) AddRate(inLabels map[string]string, name string, value int64) {
	m.addSensor("RATE", inLabels, name, value)
}

func (m *OldMetrics) AddDGauge(inLabels map[string]string, name string, value float64) {
	m.addSensor("DGAUGE", inLabels, name, value)
}

func (m *OldMetrics) AddIGauge(inLabels map[string]string, name string, value uint64) {
	m.addSensor("IGAUGE", inLabels, name, value)
}
