package units

import (
	"encoding/json"
	"fmt"
	"regexp"
	"strconv"
	"strings"
)

// ==========================================================================================

var binSuffixRe = regexp.MustCompile("(^|[^0-9])(-?[0-9.]+)([EPTGMKBb])?")
var binSuffixes = "EPTGMKBb"
var binSuffixOrd = []float64{1 << 60, 1 << 50, 1 << 40, 1 << 30, 1 << 20, 1 << 10, 1, 1}
var defaultPrecision = 3

// ==========================================================================================

// Do marshalling and unmarshalling of size for JSON.
// Time could be expressed as float64/Size or string/BinSuffix format
//
type Size struct {
	Size int64
}

func (s Size) MarshalJSON() ([]byte, error) {
	return json.Marshal(addBinSuffixPrec(float64(s.Size), defaultPrecision))
}

func (s *Size) UnmarshalJSON(b []byte) error {
	var v interface{}
	if err := json.Unmarshal(b, &v); err != nil {
		return err
	}
	switch value := v.(type) {
	case float64:
		s.Size = int64(value)
		return nil
	case string:
		if size, err := removeBinSuffix(value); err != nil {
			return err
		} else {
			s.Size = int64(size)
		}
		return nil
	default:
		return fmt.Errorf("invalid size")
	}
}

// ==========================================================================================

func removeBinSuffix(s string) (float64, error) {
	submatch := binSuffixRe.FindStringSubmatch(s)
	if submatch != nil {
		ord := float64(1)
		if submatch[3] != "" {
			ord = binSuffixOrd[strings.Index(binSuffixes, submatch[3])]
		}
		if f, err := strconv.ParseFloat(submatch[2], 64); err != nil {
			return 0, err
		} else {
			return f * ord, nil
		}
	}
	return 0, fmt.Errorf("cannot find size in string")
}

func addBinSuffix(n float64) (float64, byte) {
	var idx int
	var ord float64
	for idx, ord = range binSuffixOrd {
		if n >= ord {
			break
		}
	}
	return n / ord, binSuffixes[idx]
}

func addBinSuffixPrec(n float64, prec int) string {
	f, s := addBinSuffix(n)
	return fmt.Sprintf("%.[1]*[2]f%[3]c", prec, f, s)
}
