package unroller

import (
	"bufio"
	"bytes"
	"compress/gzip"
	"fmt"
	"log"
	"net"
	"os"
	"strconv"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/solomon/libs/go/brexp"
	"a.yandex-team.ru/solomon/libs/go/cache"
	"a.yandex-team.ru/solomon/libs/go/ipdc"
	"a.yandex-team.ru/solomon/libs/go/resolver"

	"a.yandex-team.ru/solomon/tools/discovery/internal/conductor"
	"a.yandex-team.ru/solomon/tools/discovery/internal/config"
	"a.yandex-team.ru/solomon/tools/discovery/internal/eds"
	"a.yandex-team.ru/solomon/tools/discovery/internal/kube"
)

// ==========================================================================================

// Check that err is CacheError and it is now wrapping some real error.
// Set EOL to minimum value.
func isDataStaleErr(err error, eol *time.Time) bool {
	cerr, ok := err.(*cache.CacheError)
	if !ok {
		return false
	}
	if cerr.Unwrap() != nil {
		return false
	}
	if eol.After(cerr.EOL) || eol.IsZero() {
		*eol = cerr.EOL
	}
	return true
}

// ==========================================================================================

type oGen struct {
	name     string
	f        func(bool) ([]byte, error)
	data     []byte
	err      error
	disabled bool
}

type iGen struct {
	name     string
	f        func([]byte) error
	data     []byte
	err      error
	disabled bool
}

type HostData struct {
	FQDN    string `json:"fqdn"`
	Address net.IP `json:"address"`
	Cluster string `json:"cluster"`
}

// ==========================================================================================

type Unroller struct {
	LogPrefix          string
	CacheDumpFile      string
	VerboseLevel       int
	useIPDC            bool
	conductor          *conductor.ConductorCache
	kube               *kube.KubeCache
	eds                *eds.EdsCache
	ipdc               *ipdc.IPDC
	resolver           *resolver.Resolver
	stopChan           chan struct{}
	mutex              sync.RWMutex
	cronWg             sync.WaitGroup
	cacheDumpInterval  time.Duration
	cacheDumpFileCount int
}

func NewUnroller(c *config.MainConfig) (*Unroller, error) {
	var err error
	fixDots := true

	u := &Unroller{
		LogPrefix:          "[unroller] ",
		CacheDumpFile:      c.CacheDumpFile,
		VerboseLevel:       c.VerboseLevel,
		useIPDC:            c.UseIPDC,
		stopChan:           make(chan struct{}),
		cacheDumpInterval:  c.CacheDumpInterval.Duration,
		cacheDumpFileCount: c.CacheDumpFileCount,
	}

	u.conductor = conductor.NewConductorCache(
		c.ConductorCacheConfig.CacheGoodTime.Duration,
		c.ConductorCacheConfig.CacheBadTime.Duration,
		c.ConductorCacheConfig.PrefetchTime.Duration,
		c.ConductorCacheConfig.CleanUpInterval.Duration,
		c.ConductorCacheConfig.RequestTimeout.Duration,
		c.ConductorCacheConfig.CacheMaxSize,
		c.ConductorCacheConfig.Workers,
		c.ConductorCacheConfig.ServeStale,
		c.VerboseLevel,
	)
	if u.useIPDC {
		u.ipdc = ipdc.NewIPDC(
			c.IPDCConfig.CacheGoodTime.Duration,
			c.IPDCConfig.CacheBadTime.Duration,
			c.IPDCConfig.PrefetchTime.Duration,
			c.IPDCConfig.CleanUpInterval.Duration,
			c.IPDCConfig.RequestTimeout.Duration,
			c.IPDCConfig.CacheMaxSize,
			c.IPDCConfig.ServeStale,
			c.VerboseLevel,
		)
	}
	u.resolver = resolver.NewResolver(
		c.ResolverConfig.CacheGoodTime.Duration,
		c.ResolverConfig.CacheBadTime.Duration,
		c.ResolverConfig.PrefetchTime.Duration,
		c.ResolverConfig.CleanUpInterval.Duration,
		c.ResolverConfig.CacheMaxSize,
		c.ResolverConfig.Workers,
		c.ResolverConfig.ServeStale,
		fixDots,
		c.VerboseLevel,
	)
	u.kube, err = kube.NewKubeCache(
		c.KubeConfigs,
		c.KubeCacheConfig.CacheGoodTime.Duration,
		c.KubeCacheConfig.CacheBadTime.Duration,
		c.KubeCacheConfig.PrefetchTime.Duration,
		c.KubeCacheConfig.CleanUpInterval.Duration,
		c.KubeCacheConfig.RequestTimeout.Duration,
		c.KubeCacheConfig.CacheMaxSize,
		c.KubeCacheConfig.Workers,
		c.KubeCacheConfig.ServeStale,
		c.VerboseLevel,
	)
	if err != nil {
		return nil, err
	}
	u.eds, err = eds.NewEdsCache(
		c.EdsCacheConfig.CacheGoodTime.Duration,
		c.EdsCacheConfig.CacheBadTime.Duration,
		c.EdsCacheConfig.PrefetchTime.Duration,
		c.EdsCacheConfig.CleanUpInterval.Duration,
		c.EdsCacheConfig.RequestTimeout.Duration,
		c.EdsCacheConfig.CacheMaxSize,
		c.EdsCacheConfig.Workers,
		c.EdsCacheConfig.ServeStale,
		c.VerboseLevel,
	)
	if err != nil {
		return nil, err
	}
	if err := u.prepare(); err != nil {
		return nil, fmt.Errorf("unroller prepare failed, %v", err)
	}
	u.cronWg.Add(1)
	go u.cron()

	u.log(0, nil, "started")
	return u, nil
}

func (u *Unroller) log(lvl int, ts *time.Time, format string, v ...interface{}) {
	if u.VerboseLevel >= lvl {
		tsStr := ""
		if ts != nil {
			tsStr = ", " + time.Since(*ts).String()
		}
		log.Printf(u.LogPrefix+format+tsStr, v...)
	}
}

// ==========================================================================================

func (u *Unroller) cron() {
	defer u.cronWg.Done()

	tickerDump := time.NewTicker(u.cacheDumpInterval)
	defer tickerDump.Stop()

	for {
		select {
		case <-tickerDump.C:
			initTime := time.Now()
			if err := u.Dump(); err != nil {
				u.log(0, &initTime, "failed to dump db, %v", err)
			} else {
				u.log(1, &initTime, "successfully dumped db")
			}
		case <-u.stopChan:
			u.log(1, nil, "exiting cron task")
			return
		}
	}
}

func (u *Unroller) prepare() error {
	if err := u.Restore(); err != nil {
		// Do not fail on restore: must be absent db or corrupted
		u.log(1, nil, "failed to restore db, %v", err)
	} else {
		u.log(1, nil, "db restored")
	}

	return nil
}

// ==========================================================================================

func (u *Unroller) Shutdown() {
	u.log(1, nil, "begin shutdown")
	defer u.log(1, nil, "stopped")

	u.mutex.RLock()
	defer u.mutex.RUnlock()

	close(u.stopChan)

	if err := u.dump(); err != nil {
		u.log(0, nil, "failed to dump data, %v", err)
	}

	u.conductor.Destroy()
	u.kube.Destroy()
	u.eds.Destroy()
	if u.ipdc != nil {
		u.ipdc.Destroy()
	}
	u.resolver.Destroy()

	u.cronWg.Wait()
}

func (u *Unroller) Dump() error {
	u.log(1, nil, "dumpding db to %s", u.CacheDumpFile)

	u.mutex.RLock()
	defer u.mutex.RUnlock()

	return u.dump()
}

func (u *Unroller) Restore() error {
	u.log(1, nil, "restoring db from %s", u.CacheDumpFile)

	u.mutex.Lock()
	defer u.mutex.Unlock()

	return u.restore()
}

// ==========================================================================================

// https://wiki.yandex-team.ru/cloud/devel/ycdns/dns-xds/#paasbaseimagemigracija
//
// Format:
// {"eds": ["fetcher-*.mon.cloud-preprod.yandex.net:xds.dns.cloud-preprod.yandex.net:18000"]}
//

func (u *Unroller) makeEdsHostData(refs []string, cluster string, minEOL, minLastUpdate *time.Time, emptyOk bool) ([]*HostData, error) {
	hosts := make([]*HostData, 0)

	reqs := make([]*eds.EdsRequest, len(refs))
	for idx, ref := range refs {
		pe := strings.SplitN(ref, ":", 2)
		if len(pe) != 2 {
			return nil, fmt.Errorf("bad eds ref=%s", pe)
		}
		reqs[idx] = &eds.EdsRequest{
			Pattern:  pe[0],
			Endpoint: pe[1],
		}
	}
	for req, data := range u.eds.Get(reqs, minLastUpdate) {
		unrollTime := time.Now()
		if data.Error != nil {
			if isDataStaleErr(data.Error, minEOL) {
				u.log(1, nil, "serving stale eds data for req=%v, %v", req, data.Error)
			} else {
				return nil, fmt.Errorf("failed to get eds req=%v, %v", req, data.Error)
			}
		}
		if len(data.Hosts) == 0 {
			if emptyOk {
				continue
			}
			return nil, fmt.Errorf("zero length hosts list for eds req=%v", req)
		}
		hds := make([]*HostData, 0, len(data.Hosts))
		ips := make([]net.IP, 0, len(hds))
		for host, hipStr := range data.Hosts {
			ip := net.ParseIP(hipStr)
			if ip == nil {
				return nil, fmt.Errorf("failed to parse ip=%s for name=%s for eds req=%v", hipStr, host, req)
			}
			hds = append(hds, &HostData{
				FQDN:    host,
				Address: ip,
				Cluster: cluster,
			})
			ips = append(ips, ip)
		}

		if u.ipdc != nil {
			dcs, err := u.ipdc.GetDcMany(ips)
			if err != nil {
				if isDataStaleErr(err, minEOL) {
					u.log(1, nil, "serving stale dc data for eds req=%v, %v", req, err)
				} else {
					return nil, fmt.Errorf("failed getting DCs for eds req=%v, %v", req, err)
				}
			}
			for idx, dc := range dcs {
				hds[idx].Cluster = dc
			}
		}
		hosts = append(hosts, hds...)
		u.log(1, &unrollTime, "unrolled hosts for eds req=%v size=%d", req, len(hds))
	}
	return hosts, nil
}

// Format:
// {"conductor": ["solomon_prod_stockpile_{sas,vla}"]}
//
func (u *Unroller) makeConductorHostData(refs []string,
	cluster string,
	minEOL, minLastUpdate *time.Time,
	emptyOk, bestEffortUnroll bool) ([]*HostData, error) {

	hosts := make([]*HostData, 0)

	for cg, cd := range u.conductor.Get(refs, minLastUpdate) {
		unrollTime := time.Now()
		if cd.Error != nil {
			if isDataStaleErr(cd.Error, minEOL) {
				u.log(1, nil, "serving stale data for conductor group=%s, %v", cg, cd.Error)
			} else if bestEffortUnroll {
				u.log(0, nil, "failed to get hosts for conductor refs=%v, %v, making best effort to unroll", refs, cd.Error)
				continue
			} else {
				return nil, fmt.Errorf("failed to get hosts for conductor refs=%v, %v", refs, cd.Error)
			}
		}
		if len(cd.HostsList) == 0 {
			if emptyOk || bestEffortUnroll {
				u.log(0, nil, "zero length hosts list for conductor group=%s, empty is ok or making best effort to unroll", cg)
				continue
			}
			return nil, fmt.Errorf("zero length hosts list for conductor group=%s", cg)
		}

		hds := make([]*HostData, len(cd.HostsList))
		hdr := make([]*HostData, 0, len(hds))
		ips := make([]net.IP, 0, len(hds))
		reqs := make([]*resolver.Request, len(hds))
		for idx, host := range cd.HostsList {
			reqs[idx] = &resolver.Request{
				Name: host,
				Type: resolver.A,
			}
			hds[idx] = &HostData{
				FQDN:    host,
				Cluster: cluster,
			}
		}
		staleString := ""
		for idx, rec := range u.resolver.Resolv(reqs, minLastUpdate) {
			if rec.Error != nil {
				if isDataStaleErr(rec.Error, minEOL) {
					staleString += fmt.Sprintf("%v ", rec)
				} else if bestEffortUnroll {
					u.log(0, nil, "failed to resolve %s (from conductor group=%s), %v, making best effort to unroll", hds[idx].FQDN, cg, rec.Error)
					continue
				} else {
					return nil, fmt.Errorf("failed to resolve %s (from conductor group=%s), %v", hds[idx].FQDN, cg, rec.Error)
				}
			}
			hips := rec.RDATA.([]net.IP)
			if len(hips) != 1 {
				return nil, fmt.Errorf("bad resolve %s (from conductor group=%s), IPs=%v", hds[idx].FQDN, cg, hips)
			}
			hds[idx].Address = hips[0]
			hdr = append(hdr, hds[idx])
			ips = append(ips, hips[0])
		}
		if staleString != "" {
			u.log(1, nil, "serving stale resolv data for conductor group=%s: %s", cg, staleString)
		}

		if u.ipdc != nil && len(ips) > 0 {
			dcs, err := u.ipdc.GetDcMany(ips)
			if err != nil {
				if isDataStaleErr(err, minEOL) {
					u.log(1, nil, "serving stale dc data for conductor group=%s, %v", cg, err)
				} else {
					return nil, fmt.Errorf("failed getting DCs for conductor group=%s, %v", cg, err)
				}
			}
			for idx, dc := range dcs {
				hdr[idx].Cluster = dc
			}
		}
		hosts = append(hosts, hdr...)
		u.log(1, &unrollTime, "unrolled hosts for conductor group=%s size=%d (skipped=%d)", cg, len(hdr), len(hds)-len(hdr))
	}
	return hosts, nil
}

// Format:
// {"kube": ["sas.local:solomon-testing:alerting-service"]} == "ZONE:NAMESPACE:SERVICE"
//
func (u *Unroller) makeKubeHostData(refs []string, cluster string, minEOL, minLastUpdate *time.Time, emptyOk bool) ([]*HostData, error) {
	hosts := make([]*HostData, 0)

	reqs := make([]*kube.KubeRequest, len(refs))
	for idx, ref := range refs {
		zns := strings.Split(ref, ":")
		if len(zns) != 3 {
			return nil, fmt.Errorf("bad kube ref=%s", zns)
		}
		reqs[idx] = &kube.KubeRequest{
			Zone:      zns[0],
			Namespace: zns[1],
			Service:   zns[2],
		}
	}
	for req, data := range u.kube.Get(reqs, minLastUpdate) {
		unrollTime := time.Now()
		if data.Error != nil {
			if isDataStaleErr(data.Error, minEOL) {
				u.log(1, nil, "serving stale kube data for req=%v, %v", req, data.Error)
			} else {
				return nil, fmt.Errorf("failed to get kube req=%v, %v", req, data.Error)
			}
		}
		if len(data.Hosts) == 0 {
			if emptyOk {
				continue
			}
			return nil, fmt.Errorf("zero length hosts list for kube req=%v", req)
		}
		hds := make([]*HostData, 0, len(data.Hosts))
		ips := make([]net.IP, 0, len(hds))
		for host, hipStr := range data.Hosts {
			ip := net.ParseIP(hipStr)
			if ip == nil {
				return nil, fmt.Errorf("failed to parse ip=%s for name=%s for kube req=%v", hipStr, host, req)
			}
			hds = append(hds, &HostData{
				FQDN:    host,
				Address: ip,
				Cluster: cluster,
			})
			ips = append(ips, ip)
		}

		if u.ipdc != nil {
			dcs, err := u.ipdc.GetDcMany(ips)
			if err != nil {
				if isDataStaleErr(err, minEOL) {
					u.log(1, nil, "serving stale dc data for kube req=%v, %v", req, err)
				} else {
					return nil, fmt.Errorf("failed getting DCs for kube req=%v, %v", req, err)
				}
			}
			for idx, dc := range dcs {
				hds[idx].Cluster = dc
			}
		}
		hosts = append(hosts, hds...)
		u.log(1, &unrollTime, "unrolled hosts for kube req=%v size=%d", req, len(hds))
	}
	return hosts, nil
}

// Format:
// {"raw": ["gateway-{sas,vla}-{00..05}.mon.yandex.net"]}
//
func (u *Unroller) makeRawHostData(refs []string,
	cluster string,
	minEOL, minLastUpdate *time.Time,
	bestEffortUnroll bool) ([]*HostData, error) {

	unrollTime := time.Now()

	hds := make([]*HostData, len(refs))
	hdr := make([]*HostData, 0, len(hds))
	ips := make([]net.IP, 0, len(hds))
	reqs := make([]*resolver.Request, len(hds))
	for idx, host := range refs {
		if len(host) > 0 {
			reqs[idx] = &resolver.Request{
				Name: host,
				Type: resolver.A,
			}
			hds[idx] = &HostData{
				FQDN:    host,
				Cluster: cluster,
			}
		}
	}
	staleString := ""
	for idx, rec := range u.resolver.Resolv(reqs, minLastUpdate) {
		if rec.Error != nil {
			if isDataStaleErr(rec.Error, minEOL) {
				staleString += fmt.Sprintf("%v ", rec)
			} else if bestEffortUnroll {
				u.log(0, nil, "failed to resolve %s (from raw group), %v, making best effort to unroll", hds[idx].FQDN, rec.Error)
				continue
			} else {
				return nil, fmt.Errorf("failed to resolve %s (from raw group), %v", hds[idx].FQDN, rec.Error)
			}
		}
		hips := rec.RDATA.([]net.IP)
		if len(hips) != 1 {
			return nil, fmt.Errorf("bad resolve %s (from raw group), IPs=%v", hds[idx].FQDN, hips)
		}
		hds[idx].Address = hips[0]
		hdr = append(hdr, hds[idx])
		ips = append(ips, hips[0])
	}
	if staleString != "" {
		u.log(1, nil, "serving stale resolv data for raw group: %s", staleString)
	}

	if u.ipdc != nil && len(ips) > 0 {
		dcs, err := u.ipdc.GetDcMany(ips)
		if err != nil {
			if isDataStaleErr(err, minEOL) {
				u.log(1, nil, "serving stale dc data for raw group, %v", err)
			} else {
				return nil, fmt.Errorf("failed getting DCs for raw group, %v", err)
			}
		}
		for idx, dc := range dcs {
			hdr[idx].Cluster = dc
		}
	}
	u.log(1, &unrollTime, "unrolled hosts for raw group size=%d (skipped=%d)", len(hdr), len(hds)-len(hdr))

	return hdr, nil
}

// ==========================================================================================

// Format:
// {
// "conductor": ["solomon_prod_stockpile_{sas,vla}"],
// "kube": ["{sas,vla,myt}.local:solomon-testing:alerting-service"],
// "eds": ["fetcher-*.mon.cloud-preprod.yandex.net:xds.dns.cloud-preprod.yandex.net:18000"]
// "raw": ["gateway-{sas,vla}-{00..05}.mon.yandex.net"]
// }
//
// emptyOk = do not fail if any reference unrolls to zero length hosts list
//
// Return either (nil, nil, err == [failed to unroll]) or (hosts, minEOL == [could be stale], nil)
//
func (u *Unroller) GetHostDataList(srcMap map[string][]string,
	cluster string,
	minLastUpdate *time.Time,
	emptyOk, bestEffortUnroll bool) ([]*HostData, *time.Time, error) {

	var hds []*HostData
	var err error
	minEOL := &time.Time{}
	hosts := make([]*HostData, 0)

	for src, refs := range srcMap {
		xRefs := make([]string, 0)
		for _, ref := range refs {
			xRefs = append(xRefs, brexp.Expand(ref)...)
		}
		if src == "conductor" {
			hds, err = u.makeConductorHostData(xRefs, cluster, minEOL, minLastUpdate, emptyOk, bestEffortUnroll)
		} else if src == "kube" {
			hds, err = u.makeKubeHostData(xRefs, cluster, minEOL, minLastUpdate, emptyOk)
		} else if src == "eds" {
			hds, err = u.makeEdsHostData(xRefs, cluster, minEOL, minLastUpdate, emptyOk)
		} else if src == "raw" {
			hds, err = u.makeRawHostData(xRefs, cluster, minEOL, minLastUpdate, bestEffortUnroll)
		} else {
			return nil, nil, fmt.Errorf("unknown source src=%s", src)
		}
		if err != nil {
			if bestEffortUnroll {
				u.log(0, nil, "failed to unroll %s: %v, %v, making best effort to unroll", src, xRefs, err)
			} else {
				return nil, nil, err
			}
		}
		hosts = append(hosts, hds...)
	}
	// XXX
	// Should empty hosts list be an error?
	// Probably not: we might want to add empty group first, then populate it with hosts
	//
	if !emptyOk && !bestEffortUnroll && len(hosts) == 0 {
		return nil, nil, fmt.Errorf("zero length hosts list")
	}
	return hosts, minEOL, nil
}

// ==========================================================================================

func (u *Unroller) dump() error {
	dumpFileTmp := u.CacheDumpFile + ".tmp"

	if err := u.dumper(dumpFileTmp); err != nil {
		return err
	}
	for idx := u.cacheDumpFileCount - 1; idx > 0; idx-- {
		dumpFileNew := u.CacheDumpFile + "." + strconv.FormatInt(int64(idx), 10)
		dumpFileOld := u.CacheDumpFile + "." + strconv.FormatInt(int64(idx+1), 10)
		if _, err := os.Stat(dumpFileNew); err == nil {
			if err = os.Rename(dumpFileNew, dumpFileOld); err != nil {
				return err
			}
		}

	}
	if _, err := os.Stat(u.CacheDumpFile); err == nil {
		if err = os.Rename(u.CacheDumpFile, u.CacheDumpFile+".1"); err != nil {
			return err
		}
	}
	return os.Rename(dumpFileTmp, u.CacheDumpFile)
}

func (u *Unroller) dumper(dumpFile string) error {
	dumpers := []*oGen{
		{name: "resolver", f: u.resolver.Dump},
		{name: "ipdc", f: u.ipdc.Dump, disabled: u.ipdc == nil},
		{name: "conductor", f: u.conductor.Dump},
		{name: "kube", f: u.kube.Dump},
		{name: "eds", f: u.eds.Dump},
	}
	onlyFresh := false
	sep := []byte{'\n', '\n'}
	wg := sync.WaitGroup{}

	f, err := os.OpenFile(dumpFile, os.O_CREATE|os.O_TRUNC|os.O_WRONLY, 0644)
	if err != nil {
		return err
	}
	defer func() {
		_ = f.Close()
	}()
	gw, err := gzip.NewWriterLevel(f, gzip.BestSpeed)
	if err != nil {
		return err
	}
	defer func() {
		_ = gw.Close()
	}()

	for _, dumper := range dumpers {
		if dumper.disabled {
			dumper.data = []byte("[]")
		} else {
			wg.Add(1)
			go func(p *oGen) {
				p.data, p.err = p.f(onlyFresh)
				wg.Done()
			}(dumper)
		}
	}
	wg.Wait()

	for _, dumper := range dumpers {
		if dumper.err != nil {
			return fmt.Errorf("cannot get %s data to write to cache, %v", dumper.name, err)
		}
		if _, err = gw.Write(dumper.data); err != nil {
			return fmt.Errorf("cannot write %s data to cache file, %v", dumper.name, err)
		}
		if _, err = gw.Write(sep); err != nil {
			return fmt.Errorf("cannot write separator to cache file, %v", err)
		}
	}

	if err = gw.Flush(); err != nil {
		return err
	}
	if err = f.Sync(); err != nil {
		return err
	}
	return nil
}

func (u *Unroller) restore() error {
	bufSize := 16 << 20
	restorers := []*iGen{
		{name: "resolver", f: u.resolver.Restore},
		{name: "ipdc", f: u.ipdc.Restore, disabled: u.ipdc == nil},
		{name: "conductor", f: u.conductor.Restore},
		{name: "kube", f: u.kube.Restore},
		{name: "eds", f: u.eds.Restore},
	}
	wg := sync.WaitGroup{}

	f, err := os.OpenFile(u.CacheDumpFile, os.O_RDONLY, 0644)
	if err != nil {
		return fmt.Errorf("cannot read cache db, %v", err)
	}
	defer func() {
		_ = f.Close()
	}()
	gr, err := gzip.NewReader(f)
	if err != nil {
		return fmt.Errorf("cannot read cache db, %v", err)
	}
	defer func() {
		_ = gr.Close()
	}()

	r := bufio.NewReaderSize(gr, bufSize)
	for _, restorer := range restorers {
		var b []byte
		for {
			if b, err = r.ReadSlice('\n'); err != nil {
				return fmt.Errorf("cannot read data for %s from cache file, %v", restorer.name, err)
			}
			if b = bytes.TrimRight(b, "\n"); len(b) > 0 {
				break
			}
		}
		if !restorer.disabled {
			restorer.data = make([]byte, len(b))
			_ = copy(restorer.data, b)
			wg.Add(1)
			go func(p *iGen) {
				p.err = p.f(p.data)
				wg.Done()
			}(restorer)
		}
	}
	wg.Wait()

	for _, restorer := range restorers {
		if restorer.err != nil {
			return fmt.Errorf("cannot restore %s data from cache, %v", restorer.name, err)
		}
	}
	return nil
}
