package cmd

import (
	"a.yandex-team.ru/solomon/libs/go/abc"
	"bytes"
	"context"
	"fmt"
	"github.com/go-git/go-billy/v5"
	"github.com/go-git/go-billy/v5/memfs"
	"github.com/go-git/go-git/v5"
	"github.com/go-git/go-git/v5/plumbing"
	"github.com/go-git/go-git/v5/storage/memory"
	flag "github.com/spf13/pflag"
	"io"
	"log"
	"os"
	"time"
)

const (
	respRepoSSH      = "ssh://git@bb.yandex-team.ru/cloud/resps.git"
	respRepoURL      = "https://bb.yandex-team.ru/projects/CLOUD/repos/resps"
	scheduleFilename = "cvs/yc_solomon.csv"
	commitMessage    = "update solomon duty schedule"
)

var dryRun = false

func init() {
	updateCmd := &Command{
		Use:     "update",
		Short:   "Update schedule in Cloud repository " + respRepoURL,
		Run:     runUpdateCmd,
		Timeout: time.Minute,
	}
	updateCmd.Init(func(set *flag.FlagSet) {
		set.BoolVarP(&dryRun, "dry-run", "n", false, "don't push updated schedule to cloud repo")
	})
}

func runUpdateCmd(ctx context.Context, args []string, flags *flag.FlagSet) error {
	storage := memory.NewStorage()
	fs := memfs.New()

	log.Printf("cloning repository %s", respRepoSSH)
	repo, err := git.CloneContext(ctx, storage, fs, &git.CloneOptions{URL: respRepoSSH, Progress: os.Stdout})
	if err != nil {
		return fmt.Errorf("cannot clone reps repository: %w", err)
	}
	rev, err := repo.Head()
	if err != nil {
		return fmt.Errorf("cannot get head: %w", err)
	}
	log.Printf("HEAD is %s", rev)

	updated, err := updateSchedule(ctx, fs)
	if err != nil {
		return err
	}

	if dryRun {
		log.Println("pushing to remote repo is disabled, the updated schedule comes below")
		file, err := fs.OpenFile(scheduleFilename, os.O_RDONLY, 0)
		if err != nil {
			return fmt.Errorf("cannot open file %s: %w", scheduleFilename, err)
		}
		defer file.Close()
		schedule, err := io.ReadAll(file)
		if err != nil {
			return fmt.Errorf("cannot read file %s: %w", scheduleFilename, err)
		}
		fmt.Print(string(schedule))
		return nil
	}

	if !updated {
		log.Println("the new schedule is the same as the previous one, so no need to update it")
		return nil
	}

	hash, err := commit(repo)
	if err != nil {
		return err
	}

	log.Println("pushing changes to remote repository")
	if err = repo.Push(&git.PushOptions{RemoteName: "origin"}); err != nil {
		return fmt.Errorf("cannot push changes: %w", err)
	}

	log.Printf("see commit at https://bb.yandex-team.ru/projects/CLOUD/repos/resps/commits/%s", hash.String())
	return nil
}

func generateSchedule(ctx context.Context) ([]byte, error) {
	token := GetAuthToken(ctx)

	now := time.Now()
	from := now.AddDate(0, 0, -40) // -40 days
	to := now.AddDate(0, 0, 40)    // +40 days

	abcClient := abc.NewClient(token)
	shifts, err := abcClient.Shifts(ctx, abc.FilterScheduleRange(abc.ScheduleSRE, from, to))
	if err != nil {
		return nil, err
	}

	buf := bytes.Buffer{}

	backupDuty := ""
	prevDuty := ""
	date := shifts[0].Start
	for _, shift := range shifts {
		for ; date.Before(shift.End); date = date.AddDate(0, 0, 1) {
			person := shift.Duty(date)
			currentDuty := person.Login

			if prevDuty != currentDuty {
				backupDuty = prevDuty
			}

			if backupDuty != "" {
				_, _ = fmt.Fprintf(&buf, "%s,staff:%s,staff:%s\n", date.Format("02/01/2006"), currentDuty, backupDuty)
			}

			prevDuty = currentDuty
		}
	}

	return buf.Bytes(), nil
}

func updateSchedule(ctx context.Context, fs billy.Filesystem) (bool, error) {
	log.Printf("reading previous schedule from %s", scheduleFilename)
	file, err := fs.OpenFile(scheduleFilename, os.O_RDWR|os.O_CREATE, 0)
	if err != nil {
		return false, fmt.Errorf("cannot open file %s: %w", scheduleFilename, err)
	}
	defer file.Close()

	prevSchedule, err := io.ReadAll(file)
	if err != nil {
		return false, fmt.Errorf("cannot read file %s: %w", scheduleFilename, err)
	}

	log.Println("generating new schedule")
	newSchedule, err := generateSchedule(ctx)
	if err != nil {
		return false, fmt.Errorf("cannot write schedule to file %s: %w", scheduleFilename, err)
	}

	if bytes.Equal(newSchedule, prevSchedule) {
		return false, nil
	}

	log.Printf("writing new schedule to %s", scheduleFilename)
	if err = file.Truncate(0); err != nil {
		return false, fmt.Errorf("cannot truncate file %s: %w", scheduleFilename, err)
	}

	if _, err = file.Seek(0, io.SeekStart); err != nil {
		return false, fmt.Errorf("cannot seek to file begin: %w", err)
	}

	if _, err = file.Write(newSchedule); err != nil {
		return false, fmt.Errorf("cannot update file %s: %w", scheduleFilename, err)
	}

	return true, nil
}

func commit(repo *git.Repository) (plumbing.Hash, error) {
	log.Println("committing schedule changes")
	wt, err := repo.Worktree()
	if err != nil {
		return plumbing.ZeroHash, fmt.Errorf("cannot get worktree: %w", err)
	}

	if _, err = wt.Add(scheduleFilename); err != nil {
		return plumbing.ZeroHash, fmt.Errorf("cannot add file %s to worktree: %w", scheduleFilename, err)
	}

	hash, err := wt.Commit(commitMessage, &git.CommitOptions{})
	if err != nil {
		return plumbing.ZeroHash, fmt.Errorf("cannot commit changes: %w", err)
	}

	log.Printf("hash %s", hash.String())

	return hash, nil
}
