package main

import (
	"bufio"
	"crypto/tls"
	"crypto/x509"
	"database/sql"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"regexp"
	"strings"
	"time"

	"github.com/go-sql-driver/mysql"

	"a.yandex-team.ru/solomon/libs/go/cache"
)

// ==========================================================================================

type MainConfig struct {
	HTTPAddr      string        `json:"http_addr"`
	GrafanaIni    string        `json:"grafana_ini"`
	RootCAPath    string        `json:"root_ca_path"`
	MysqlAddr     string        `json:"mysql_addr"`
	MysqlUser     string        `json:"mysql_user"`
	MysqlPassword string        `json:"mysql_password"`
	MysqlDB       string        `json:"mysql_db"`
	ClientTimeout time.Duration `json:"client_timeout"`
}

var reDashboard = regexp.MustCompile(`^/dashboard/db/([a-zA-Z0-9_-]+)$`)
var reRawQuery = regexp.MustCompile(`^[a-zA-Z0-9_=./%:&-]*$`)

// ==========================================================================================

var Config = MainConfig{
	HTTPAddr:      "[::1]:8090",
	GrafanaIni:    "/etc/grafana/grafana.ini",
	RootCAPath:    "/etc/grafana/root.crt",
	MysqlAddr:     "[::1]:3305",
	ClientTimeout: 15 * time.Second,
}

// ==========================================================================================

func readConfig(path string) {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		log.Fatalf("Bad config file path %s: %s", path, err.Error())
	}
	if err = json.Unmarshal(data, &Config); err != nil {
		log.Fatalf("Bad json in config file %s: %s", path, err.Error())
	}
}

func readGrafanaIni(iniFile string) map[string]string {
	var prefix string
	iniContent := make(map[string]string)

	file, err := os.Open(iniFile)
	if err != nil {
		log.Fatalf("Bad grafana password file path %s: %s", iniFile, err.Error())
	}
	defer file.Close()

	scanner := bufio.NewScanner(file)
	for scanner.Scan() {
		line := scanner.Text()
		commentIdx := strings.IndexAny(line, "#;")
		if commentIdx >= 0 {
			line = line[:commentIdx]
		}
		line = strings.TrimSpace(line)
		if strings.HasPrefix(line, "[") {
			prefix = line[1 : len(line)-1]
		} else if len(line) > 0 {
			fields := strings.SplitN(line, "=", 2)
			if len(fields) == 2 {
				iniContent[fmt.Sprintf("%s.%s", prefix, strings.TrimSpace(fields[0]))] = strings.TrimSpace(fields[1])
			}
		}
	}
	return iniContent
}

// ==========================================================================================

func mysqlRegisterCustomTLS(rootCAPath string) error {
	rootCertPool := x509.NewCertPool()
	pem, err := ioutil.ReadFile(rootCAPath)
	if err != nil {
		log.Fatalf("Failed to read root CA file %s: %s", rootCAPath, err.Error())
	}
	if ok := rootCertPool.AppendCertsFromPEM(pem); !ok {
		log.Fatalln("Failed to append PEM to root CA pool.")
	}

	return mysql.RegisterTLSConfig("custom", &tls.Config{
		RootCAs:            rootCertPool,
		InsecureSkipVerify: true,
	})
}

func mysqlDB(host string,
	user string,
	password string,
	dbname string,
	timeout time.Duration) (*sql.DB, error) {

	dataSourceName := fmt.Sprintf("%s:%s@tcp(%s)/%s?readTimeout=3s&writeTimeout=3s&timeout=%ds&tls=custom",
		user,
		password,
		host,
		dbname,
		uint(timeout/time.Second))

	db, err := sql.Open("mysql", dataSourceName)
	if err != nil {
		return nil, err
	}
	db.SetConnMaxLifetime(timeout)

	return db, nil
}

// ==========================================================================================

type mysqlSlugCache struct {
	baseURL string
	db      *sql.DB
	cache   *cache.Cache
}

func NewMysqlSlugCache(baseURL string, db *sql.DB) *mysqlSlugCache {
	goodCacheTime := 2 * time.Minute
	badCacheTime := 10 * time.Second
	prefetchTime := time.Duration(0)
	cleanUpInterval := 5 * time.Second
	serveStale := true
	verboseLevel := 1
	cacheSize := 500

	m := &mysqlSlugCache{
		baseURL: baseURL,
		db:      db,
	}
	m.cache = cache.NewCache(
		"mysql",
		func(req interface{}) (interface{}, error) {
			return m.mysqlReq(req.(string))
		},
		goodCacheTime,
		badCacheTime,
		prefetchTime,
		cleanUpInterval,
		serveStale,
		verboseLevel,
		cacheSize,
	)
	return m
}

func (m *mysqlSlugCache) mysqlReq(slug string) (*string, error) {
	var uid string
	var err error

	if err = m.db.Ping(); err != nil {
		return nil, fmt.Errorf("failed to ping MySQL server: %v", err)
	}

	var rows *sql.Rows
	if rows, err = m.db.Query("SELECT uid FROM dashboard WHERE slug = ?", slug); err != nil {
		return nil, fmt.Errorf("failed to get rows from MySQL server for slug='%s': %v", slug, err)
	}
	if !rows.Next() {
		err = fmt.Errorf("uid not found for slug %s", slug)
	} else {
		if err = rows.Scan(&uid); err != nil {
			err = fmt.Errorf("failed to scan rows for slug %s, %v", slug, err)
		}
	}
	_ = rows.Close()
	return &uid, err
}

func (m *mysqlSlugCache) httpHandle(w http.ResponseWriter, r *http.Request) {
	reqStart := time.Now()
	log.Printf("connection from %s: %s", r.RemoteAddr, r.URL.Path)

	if len(r.URL.Path) > 128 {
		w.Header().Set("Connection", "Close")
		w.WriteHeader(http.StatusRequestURITooLong)
		log.Printf("request URI too long 414 (%s): path=%s, %v", r.RemoteAddr, r.URL.Path, time.Since(reqStart))
	} else if idxs := reDashboard.FindStringSubmatchIndex(r.URL.Path); idxs != nil && reRawQuery.MatchString(r.URL.RawQuery) {
		var data []byte
		var code int

		slug := r.URL.Path[idxs[2]:idxs[3]]

		uid, err := m.cache.Get(slug)
		if err != nil || uid == nil {
			code = http.StatusNotFound
			log.Printf("not found %d (%s): slug=%s err=%v, %v", code, r.RemoteAddr, slug, err, time.Since(reqStart))
			data = []byte("slug not found")
		} else {
			uidString := *(uid.(*string))
			code = http.StatusFound
			dashboardURL := m.baseURL + "/d/" + uidString + "/" + slug + "?" + r.URL.RawQuery
			w.Header().Set("Location", dashboardURL)
			log.Printf("found %d (%s): redir=%s, %v", code, r.RemoteAddr, dashboardURL, time.Since(reqStart))
			data = []byte("<a href=\"" + dashboardURL + "\">URL</a>.\n")
		}

		w.Header().Set("Content-Type", "text/html; charset=utf-8")
		w.Header().Set("Connection", "Close")
		w.WriteHeader(code)
		if _, err := w.Write(data); err != nil {
			log.Printf("failed to send %d response (%s): %v", code, r.RemoteAddr, err)
		}
	} else {
		w.Header().Set("Connection", "Close")
		w.WriteHeader(http.StatusNotFound)
		log.Printf("request not found 404 (%s): url=%#v, %v", r.RemoteAddr, r.URL, time.Since(reqStart))
	}
}

func (m *mysqlSlugCache) Shutdown() {
	m.cache.Destroy()
}

// ==========================================================================================

func serveHTTP(httpHandle func(w http.ResponseWriter, r *http.Request)) {
	server := &http.Server{
		Addr:           Config.HTTPAddr,
		Handler:        http.HandlerFunc(httpHandle),
		ReadTimeout:    Config.ClientTimeout,
		WriteTimeout:   Config.ClientTimeout,
		MaxHeaderBytes: 1 << 20,
	}
	log.Println(server.ListenAndServe())
}

// ==========================================================================================

func main() {
	log.SetFlags(log.Ldate | log.Ltime | log.Lmicroseconds)
	if len(os.Args) >= 2 {
		readConfig(os.Args[1])
	}

	if len(Config.GrafanaIni) > 0 {
		grafanaIni := readGrafanaIni(Config.GrafanaIni)
		Config.MysqlUser = grafanaIni["database.user"]
		Config.MysqlPassword = grafanaIni["database.password"]
		Config.MysqlDB = grafanaIni["database.name"]
	}
	if Config.MysqlUser == "" || Config.MysqlDB == "" || Config.MysqlPassword == "" {
		log.Fatalln("Cannot get mysql endpoint parameters")
	}
	var baseURL string
	if name, err := os.Hostname(); err == nil && strings.HasPrefix(name, "grafana-dev") {
		baseURL = "https://grafana-dev.yandex-team.ru"
	} else {
		baseURL = "https://grafana.yandex-team.ru"
	}

	if err := mysqlRegisterCustomTLS(Config.RootCAPath); err != nil {
		log.Fatalf("Register root CA failed: %v", err)
	}
	db, err := mysqlDB(
		Config.MysqlAddr,
		Config.MysqlUser,
		Config.MysqlPassword,
		Config.MysqlDB,
		Config.ClientTimeout/2,
	)
	if err != nil {
		log.Fatalf("Failed to open DB %s@%s/%s: %v", Config.MysqlUser, Config.MysqlAddr, Config.MysqlDB, err)
	}
	defer db.Close()

	m := NewMysqlSlugCache(baseURL, db)
	serveHTTP(m.httpHandle)
	m.Shutdown()
}
