#include "storage.h"
#include "table_reader.h"

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_scheme/scheme.h>

#include <library/cpp/getopt/last_getopt.h>

#include <util/generic/vector.h>
#include <util/generic/utility.h>
#include <util/string/builder.h>
#include <util/stream/format.h>
#include <util/thread/pool.h>

#include <cmath>
#include <atomic>

namespace NSolomon::NIndexer {
namespace {

TString ToEndpointAddr(const TString& host, ui16 port) {
    TString addr = host;
    if (TString::npos == addr.find('.')) {
        addr.append(".search.yandex.net");
    }
    addr.append(':');
    addr.append(ToString(port));
    return addr;
}

void EnsureSuccess(const NYdb::TStatus& s, TStringBuf message) {
    Y_ENSURE(s.IsSuccess(),
             message << " [" << s.GetStatus() << "] "
             << s.GetIssues().ToString());
}

struct TTable {
    TString Path;
    TString Alias;
};

class TReader {
public:
    TReader(const TString& host, ui16 port, ui32 maxInFligh) {
        TString endpointAddr = ToEndpointAddr(host, port);
        Cerr << "connecting to " << endpointAddr << Endl;

        Driver_ = MakeHolder<NYdb::TDriver>(NYdb::TDriverConfig()
                .SetEndpoint(endpointAddr)
                .SetNetworkThreadsNum(2)
                .SetDiscoveryMode(NYdb::EDiscoveryMode::Sync));

        SchemeClient_ = MakeHolder<NYdb::NScheme::TSchemeClient>(*Driver_);
        TableReader_ = MakeHolder<TStreamTableReader>(endpointAddr, maxInFligh);
    }

    TVector<TTable> ResolvePath(const TString& path, const TString& filter) {
        auto pathDesc = SchemeClient_->DescribePath(path).ExtractValueSync();
        EnsureSuccess(pathDesc, "cannot describe path");

        TVector<TTable> tables;
        switch (pathDesc.GetEntry().Type) {
        case NYdb::NScheme::ESchemeEntryType::Table: {
            TString alias(TStringBuf(path).RAfter('/'));
            tables.push_back(TTable{path, alias});
            break;
        }
        case NYdb::NScheme::ESchemeEntryType::Directory: {
            auto res = SchemeClient_->ListDirectory(path).ExtractValueSync();
            EnsureSuccess(res, "cannot list directory");
            for (const NYdb::NScheme::TSchemeEntry& e: res.GetChildren()) {
                if (e.Type != NYdb::NScheme::ESchemeEntryType::Table) {
                    continue;
                }
                if (!filter.Empty() && !e.Name.Contains(filter)) {
                    continue;
                }
                tables.push_back(TTable{path + '/' + e.Name, e.Name});
            }
            break;
        }
        default:
            ythrow yexception() << "given path [" << path << "] is a "
                                << pathDesc.GetEntry().Type
                                << ", but expected Table or Directory";
        }
        return tables;
    }

    ui64 ReadTable(const TString& tablePath, const TRowConsumer& consumer) {
        return static_cast<ui64>(TableReader_->Read(tablePath, consumer));
    }

private:
    THolder<NYdb::TDriver> Driver_;
    THolder<NYdb::NScheme::TSchemeClient> SchemeClient_;
    THolder<TStreamTableReader> TableReader_;
};

struct TReaderState {
    std::atomic<size_t> TableIdx;
    std::atomic<size_t> RowsRead;
};

} // namespace

int CmdLoad(int argc, const char* argv[]) {
    TString ydbHost;
    ui16 ydbPort;
    TString ydbPath;
    TString ydbFilter;
    TVector<TString> ydbTables;
    TString dbPath;
    NLastGetopt::TOpts opts;
    size_t threadNum;

    opts.AddLongOption("host", "YDB hostname (fqdn or short name)")
            .RequiredArgument("STR")
            .StoreResult(&ydbHost)
            .Required();
    opts.AddLongOption("port", "YDB port")
            .RequiredArgument("NUM")
            .StoreResult(&ydbPort)
            .DefaultValue(2135)
            .Optional();
    opts.AddLongOption("db", "database path to write to")
            .RequiredArgument("STR")
            .StoreResult(&dbPath)
            .Required();
    opts.AddLongOption("path", "path in YDB to a single table or to a directory")
            .RequiredArgument("STR")
            .StoreResult(&ydbPath)
            .Optional();
    opts.AddLongOption("table", "path in YDB to a single table in form 'name:path'")
            .RequiredArgument("STR")
            .AppendTo(&ydbTables)
            .Optional();
    opts.AddLongOption("filter", "filter table name by given substring")
            .RequiredArgument("STR")
            .StoreResult(&ydbFilter)
            .Optional();
    opts.AddLongOption("threads", "number of threads to load data")
            .RequiredArgument("NUM")
            .StoreResult(&threadNum)
            .DefaultValue(4)
            .Optional();

    try {
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);

        if (ydbPath.empty() && ydbTables.empty()) {
            Cerr << "either --path or --table must be provided" << Endl;
            return 1;
        }

        TVector<TTable> tables;
        if (!ydbPath.empty()) {
            TReader reader(ydbHost, ydbPort, 1);
            tables = reader.ResolvePath(ydbPath, ydbFilter);
        }

        for (const TString& table: ydbTables) {
            TStringBuf alias, path;
            if (!TStringBuf(table).TrySplit(':', alias, path)) {
                path = table;
                alias = path.RAfter('/');
            }
            tables.push_back(TTable{ TString{path}, TString{alias} });
        }

        size_t tableCount = tables.size();
        size_t width = static_cast<size_t>(std::llround(std::log10(tableCount))) + 1;

        auto threadPool = CreateThreadPool(threadNum);
        TVector<TReaderState> state(threadNum);
        std::atomic<size_t> tableIdx{0};
        std::atomic<size_t> tableDone{0};
        std::atomic<size_t> totalRows{0};

        auto storage = CreateMmsStorage(dbPath, EUsageMode::WRITE_ONLY);
        for (size_t thread = 0; thread < threadNum; ++thread) {
            threadPool->SafeAddFunc([&, thread]() {
                TReader reader(ydbHost, ydbPort, 4);
                TStringBuilder buf;
                while (true) {
                    size_t idx = tableIdx++;
                    if (idx >= tableCount) {
                        break;
                    }

                    state[thread].TableIdx = idx;
                    state[thread].RowsRead = 0;

                    const TTable& table = tables[idx];
                    reader.ReadTable(table.Path, [&](TMetricId id, const TString& name) mutable {
                        buf.clear();
                        buf << table.Alias << '/' << name;
                        storage->Write(id, buf);
                        ++state[thread].RowsRead;
                    });

                    totalRows += state[thread].RowsRead;
                    ++tableDone;
                }

                state[thread].TableIdx = Max<size_t>();
            });
        }

        while (tableDone.load() < tableCount) {
            Cerr << "\33[2K\r["
                 << LeftPad(tableDone.load(), width) << '/' << tableCount
                 << "] tables done\n";

            for (size_t thread = 0; thread < threadNum; ++thread) {
                size_t idx = state[thread].TableIdx;
                size_t rows = state[thread].RowsRead;
                if (idx != Max<size_t>()) {
                    const TTable& table = tables[idx];
                    Cerr << "\33[2K\r  from " << table.Path << " read " << rows << " rows\n";
                } else {
                    Cerr << "\33[2K\r  done\n";
                }
            }

            Sleep(TDuration::Seconds(1));
            Cerr << "\33[1000D\33[" << threadNum + 1 << 'A';
        }

        Cerr << "\ntotal number of rows: " << totalRows.load();
        Cerr << "\ncompacting... ";
        storage->Compact();
        Cerr << "done." << Endl;

        return 0;
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return 1;
    }
}

} // namespace NSolomon::NIndexer
