#include "storage.h"

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/service/monservice.h>
#include <library/cpp/monlib/service/pages/html_mon_page.h>
#include <library/cpp/monlib/service/pages/templates.h>
#include <library/cpp/monlib/service/pages/version_mon_page.h>

#include <util/string/builder.h>
#include <util/string/split.h>

namespace NSolomon {
namespace NIndexer {
namespace {

constexpr ui32 ServerThreadCount = 4;

TString CreateUrl(const TString& address, ui16 port) {
    TStringBuilder url;
    url << "http://";
    if (address.Contains(':')) {
        url << '[' << address << ']';
    } else {
        url << address;
    }
    url << ':' << port << '/';
    return std::move(url);
}

class TSearchPage: public NMonitoring::THtmlMonPage {
public:
    explicit TSearchPage(IStoragePtr storage)
        : THtmlMonPage("search", "Search metric by Stockpile ID", false)
        , Storage_(std::move(storage))
    {
    }

    void OutputContent(NMonitoring::IMonHttpRequest& request) override {
        const auto& postParams = request.GetParams();
        const auto& idsStr = postParams.Get(TStringBuf("ids"));

        OutputForm(request.Output(), idsStr);
        if (!idsStr.empty()) {
            OutputResults(request.Output(), idsStr);
        }
    }

    void OutputForm(IOutputStream& out, const TString& idsStr) {
        HTML(out) {
            out << "<form method=\"GET\">";
            DIV_CLASS("form-group") {
                LABEL() {
                    out << "Stockpile ID(s)";
                }
                out << "<p class=\"help-block\">ID must be written in this format: "
                       "{shardId}/{localid}. Several IDs can be divided by any of "
                       "this characters [ ,;\\t\\r\\n]. Empty or invalid ID will "
                       "be skipped.</p>";
                out << "<textarea class=\"form-control\" name=\"ids\" rows=\"10\">";
                out << idsStr;
                out << "</textarea>";
            }
            DIV_CLASS("form-group") {
                out << "<button type=\"submit\" class=\"btn btn-default\">Search</button>";
            }
            out << "</form>";
        }
    }

    void OutputResults(IOutputStream& out, const TString& idsStr) {
        HTML(out) {
            TVector<TString> warns;

            TABLE_CLASS("table table-condensed") {
                CAPTION() { out << "Found metrics"; }
                TABLEHEAD() {
                    TABLER() {
                        TABLEH() { out << "Metric Id"; }
                        TABLEH() { out << "Table Name"; }
                        TABLEH() { out << "Labels/Name"; }
                    }
                }
                TABLEBODY() {
                    for (const auto& it: StringSplitter(idsStr).SplitBySet(",; \t\n\r").SkipEmpty()) {
                        TStringBuf idStr = it.Token();
                        if (idStr.find('/') == TStringBuf::npos) {
                            warns.push_back(TStringBuilder() << "invalid metric id \'" << idStr << '\'');
                            continue;
                        }

                        TMetricId id = FromString<TMetricId>(idStr);
                        TMaybe<TString> value = Storage_->Read(id);
                        if (!value) {
                            warns.push_back(TStringBuilder() << "metric \'" << idStr << "\' was not found");
                            continue;
                        }

                        TStringBuf tableName, labels;
                        if (!TStringBuf(*value).TrySplit('/', tableName, labels)) {
                            tableName = TStringBuf("");
                            labels = *value;
                        }

                        TABLER() {
                            TABLED() { out << id; }
                            TABLED() { out << tableName; }
                            TABLED() { out << labels; }
                        }
                    }
                }
            }

            if (!warns.empty()) {
                DIV_CLASS("alert alert-warning") {
                    STRONG() { out << "Warning!"; }
                    UL() {
                        for (const TString& warn: warns) {
                            LI() { out << warn; }
                        }
                    }
                }
            }
        }
    }

private:
    IStoragePtr Storage_;
};

} // namespace

int CmdServer(int argc, const char* argv[]) {
    TString address;
    ui16 port;
    TString dbPath;
    NLastGetopt::TOpts opts;

    opts.AddLongOption("address", "HTTP server bind address")
            .RequiredArgument("STR")
            .StoreResult(&address)
            .DefaultValue("localhost")
            .Optional();
    opts.AddLongOption("port", "HTTP server port")
            .RequiredArgument("NUM")
            .StoreResult(&port)
            .DefaultValue(8601)
            .Optional();
    opts.AddLongOption("db", "path to a database directory")
            .RequiredArgument("STR")
            .StoreResult(&dbPath)
            .Required();

    try {
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);

        Cerr << "reading index from " << dbPath << Endl;
        auto storage = CreateMmsStorage(dbPath, EUsageMode::READ_ONLY);

        Cerr << "listening on " << CreateUrl(address, port) << Endl;
        NMonitoring::TMonService2 server(port, address, ServerThreadCount);
        server.Register(new TSearchPage(std::move(storage)));
        server.Register(new NMonitoring::TVersionMonPage);
        server.StartOrThrow();

        for (;;) {
            Sleep(TDuration::Hours(1));
        }

        return 0;
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return 1;
    }
}

} // namespace NIndexer
} // namespace NSolomon
