#include "storage.h"

#include <library/cpp/blockcodecs/core/codecs.h>
#include <library/cpp/on_disk/mms/mapping.h>
#include <library/cpp/on_disk/mms/unordered_map.h>
#include <library/cpp/on_disk/mms/writer.h>

#include <util/generic/maybe.h>
#include <util/generic/typetraits.h>
#include <util/string/cast.h>
#include <util/string/subst.h>
#include <util/stream/file.h>
#include <util/system/spinlock.h>
#include <util/generic/vector.h>
#include <util/system/fs.h>
#include <util/generic/buffer.h>
#include <util/system/unaligned_mem.h>
#include <util/thread/pool.h>
#include <util/system/event.h>
#include <util/string/builder.h>

namespace NSolomon::NIndexer {
struct Y_PACKED TOffsetValue {
    ui64 FileOffset:48;
    ui64 BlockOffset:16;

    bool operator==(TOffsetValue rhs) const noexcept {
        return FileOffset == rhs.FileOffset && BlockOffset == rhs.BlockOffset;
    }
};
} // namespace NSolomon::NIndexer

static_assert (sizeof(NSolomon::NIndexer::TOffsetValue) == 8, "unexpected struct size");
Y_DECLARE_PODTYPE(NSolomon::NIndexer::TOffsetValue);

namespace NSolomon::NIndexer {
namespace {

static constexpr ui32 FIRST_SHARD = 1;
static constexpr ui32 LAST_SHARD = 4096;

template <typename P>
using TOffsetsMap = NMms::TUnorderedMap<P, ui64, TOffsetValue>;

const NBlockCodecs::ICodec* Zstd() {
    static const auto* CODEC = NBlockCodecs::Codec("zstd_11"); // optimal level
    return CODEC;
}

////////////////////////////////////////////////////////////////////////////////
// TNamesFile
////////////////////////////////////////////////////////////////////////////////
class TNamesFile {
public:
    enum EMode {
        READ,
        WRITE,
    };
    static constexpr TFlags<EOpenModeFlag> READ_MODE{OpenExisting | RdOnly};
    static constexpr TFlags<EOpenModeFlag> WRITE_MODE{OpenAlways | RdWr | ForAppend};

public:
    TNamesFile(const TString& path, EMode mode)
        : File_(path, mode == READ ? READ_MODE : WRITE_MODE)
        , Buffer_(mode == READ ? 0 : 64 << 10)
        , Offset_(mode == READ ? 0 : static_cast<ui64>(File_.GetLength()))
    {
    }

    TOffsetValue Write(TStringBuf name) {
        ui16 length = static_cast<ui16>(name.size());
        if (name.size() + sizeof(length) > Buffer_.Avail()) {
            FlushBuffer();
            Buffer_.Clear();
        }

        ui32 blockOffset = static_cast<ui32>(Buffer_.Size());
        WriteUnaligned<ui16>(Buffer_.Pos(), length);
        Buffer_.Advance(sizeof(length));
        Buffer_.Append(name.data(), name.size());
        return { Offset_, ui64(blockOffset) };
    }

    TString Read(TOffsetValue offset) const {
        File_.Seek(static_cast<i64>(offset.FileOffset), SeekDir::sSet);

        ui32 compressLen = 0;
        File_.Load(&compressLen, sizeof(compressLen));

        TTempBuf tmpBuf(compressLen);
        File_.Load(tmpBuf.Data(), compressLen);
        TStringBuf compressBuf{tmpBuf.Data(), size_t(compressLen)};

        auto* codec = Zstd();
        size_t decompressLen = codec->DecompressedLength(compressBuf);
        TTempBuf decompressBuf(decompressLen);
        codec->Decompress(compressBuf, decompressBuf.Data());

        decompressBuf.SetPos(offset.BlockOffset);
        ui16 nameLen = ReadUnaligned<ui16>(decompressBuf.Proceed(sizeof(nameLen)));

        return TString{decompressBuf.Current(), size_t(nameLen)};
    }

    void Flush() {
        if (!Buffer_.Empty()) {
            FlushBuffer();
            Buffer_.Reset();
            File_.Flush();
        }
    }

private:
    void FlushBuffer() {
        auto* codec = Zstd();
        ui32 len = 0;
        TTempBuf tmpBuf(codec->MaxCompressedLength(Buffer_) + sizeof(len));
        len = static_cast<ui32>(codec->Compress(Buffer_, tmpBuf.Data() + sizeof(len)));
        WriteUnaligned<decltype(len)>(tmpBuf.Data(), len);
        File_.Write(tmpBuf.Data(), len + sizeof(len));
        Offset_ += len + sizeof(len);
    }

private:
    mutable TFile File_;
    TBuffer Buffer_;
    ui64 Offset_;
};

////////////////////////////////////////////////////////////////////////////////
// TMmsWriter
////////////////////////////////////////////////////////////////////////////////
class TMmsWriter: public IStorage {
    struct Y_PACKED TOffsetRecord {
        ui64 LocalId;
        TOffsetValue Offset;
    };

    struct TShard: private TMoveOnly {
        const ui16 ShardId;
        TNamesFile Names;
        TString OffsetsFilename;
        THolder<TFileOutput> Offsets;
        mutable TAdaptiveLock Lock;
        bool Dirty{false};

        TShard(const TString& path, ui16 shardId)
            : ShardId(shardId)
            , Names(path + '/' + ToString(shardId) + ".names", TNamesFile::WRITE)
            , OffsetsFilename(path + '/' + ToString(shardId) + ".offsets")
            , Offsets(MakeHolder<TFileOutput>(TFile::ForAppend(OffsetsFilename)))
        {
        }

        void Finish() {
            Names.Flush();
            Offsets->Flush();
        }

        void Append(ui64 localId, TStringBuf value) {
            auto g = Guard(Lock);
            Dirty = true;
            TOffsetValue offset = Names.Write(value);
            TOffsetRecord offsetRecord{localId, offset};
            Offsets->Write(&offsetRecord, sizeof(offsetRecord));
        }

        void BuildIndex() {
            auto indexFilename = SubstGlobalCopy(OffsetsFilename, ".offsets", ".idx");
            if (!Dirty && NFs::Exists(indexFilename)) {
                // no writes to shard and index is already exists
                return;
            }

            Cerr << (TStringBuilder() << "build index for " << ShardId <<" shard\n");

            TFileInput offsetsFile(OffsetsFilename);
            TOffsetsMap<NMms::TStandalone> offsetsMap;
            while (true)  {
                TOffsetRecord o;
                size_t n = offsetsFile.Load(&o, sizeof(o));
                if (n == 0) {
                    break;
                }

                auto [it, inserted] = offsetsMap.emplace(o.LocalId, o.Offset);
                if (inserted) {
                    // new local id
                    continue;
                }

                TOffsetValue prevOffset = it->second;
                if (prevOffset == o.Offset) {
                    // same offsets
                    continue;
                }

                auto prevName = Names.Read(prevOffset);
                auto nextName = Names.Read(o.Offset);
                if (prevName == nextName) {
                    // same names
                    continue;
                }

                // same local id used for really different names
                Cout << (TStringBuilder()
                     << "collision: " << TMetricId{ShardId, o.LocalId}
                     << ", prevName: " << prevName
                     << ", nextName: " << nextName
                     << '\n');
            }

            NFs::Remove(indexFilename);
            TFileOutput indexFile(indexFilename);
            NMms::SafeWrite(indexFile, offsetsMap);
        }
    };

public:
    explicit TMmsWriter(const TString& dbPath) {
        NFs::MakeDirectory(dbPath);
        Shards_.reserve(LAST_SHARD);
        for (ui16 shardId = FIRST_SHARD; shardId <= LAST_SHARD; shardId++) {
            Shards_.emplace_back(dbPath, shardId);
        }
    }

private:
    bool Write(TMetricId id, TStringBuf value) override {
        Y_ENSURE(id.ShardId >= FIRST_SHARD && id.ShardId <= LAST_SHARD,
                 "invalid shardId: " << id.ShardId);
        Shards_[id.ShardId - 1].Append(id.LocalId, value);
        return true;
    }

    void Compact(IThreadPool* tp) override {
        if (tp == nullptr) {
            for (ui16 shardId = FIRST_SHARD; shardId <= LAST_SHARD; shardId++) {
                auto& shard = Shards_[shardId - 1];
                shard.Finish();
                shard.BuildIndex();
            }
        } else {
            TManualEvent event;
            std::atomic<ui16> shardDone{0};
            for (ui16 shardId = FIRST_SHARD; shardId <= LAST_SHARD; shardId++) {
                tp->SafeAddFunc([&, shardId]() {
                    auto& shard = Shards_[shardId - 1];
                    shard.Finish();
                    shard.BuildIndex();

                    if (++shardDone == LAST_SHARD) {
                        event.Signal();
                    }
                });
            }
            event.WaitI();
        }
    }

    TMaybe<TString> Read(TMetricId id) const override {
        Y_UNUSED(id);
        ythrow yexception() << "Read() is unsupported";
    }

private:
    TVector<TShard> Shards_;
};

////////////////////////////////////////////////////////////////////////////////
// TMmsReader
////////////////////////////////////////////////////////////////////////////////
class TMmsReader: public IStorage {
    struct TShard: private TMoveOnly {
        TNamesFile Names;
        NMms::TMapping<TOffsetsMap<NMms::TMmapped>> OffsetsMap;

        TShard(const TString& path, ui16 shardId)
            : Names(path + '/' + ToString(shardId) + ".names", TNamesFile::READ)
            , OffsetsMap(path + '/' + ToString(shardId) + ".idx")
        {
        }

        TMaybe<TString> Find(ui64 localId) const {
            auto it = OffsetsMap->find(localId);
            if (it == OffsetsMap->end()) {
                return {};
            }

            TOffsetValue offset = it->second;
            return Names.Read(offset);
        }
    };

public:
    explicit TMmsReader(const TString& dbPath) {
        Shards_.reserve(LAST_SHARD);
        for (ui16 shardId = FIRST_SHARD; shardId <= LAST_SHARD; shardId++) {
            Shards_.emplace_back(dbPath, shardId);
        }
    }

private:
    bool Write(TMetricId id, TStringBuf value) override {
        Y_UNUSED(id);
        Y_UNUSED(value);
        ythrow yexception() << "Write() is unsupported";
    }

    TMaybe<TString> Read(TMetricId id) const override {
        Y_ENSURE(id.ShardId >= FIRST_SHARD && id.ShardId <= LAST_SHARD,
                 "invalid shardId: " << id.ShardId);
        return Shards_[id.ShardId - 1].Find(id.LocalId);
    }

private:
    TVector<TShard> Shards_;
};

} // namespace

IStoragePtr CreateMmsStorage(TStringBuf dbPath, EUsageMode usagMode) {
    switch (usagMode) {
    case EUsageMode::WRITE_ONLY:
        return MakeHolder<TMmsWriter>(TString(dbPath));
    case EUsageMode::READ_ONLY:
        return MakeHolder<TMmsReader>(TString(dbPath));
    case EUsageMode::READ_WRITE:
        ythrow yexception() << "READ_WRITE mode is unsupported";
    }
}

} // namespace NSolomon::NIndexer
