#include "storage.h"

#include <util/generic/ptr.h>
#include <util/generic/yexception.h>
#include <util/generic/maybe.h>
#include <util/system/fs.h>

#include <contrib/libs/rocksdb/include/rocksdb/db.h>

namespace NSolomon {
namespace NIndexer {
namespace {

class TRocksDbStorage: public IStorage {
public:
    TRocksDbStorage(TStringBuf dbPath, EUsageMode usageMode) {
        rocksdb::Options opts;
        if (usageMode == EUsageMode::WRITE_ONLY) {
            opts.OptimizeForPointLookup(64);
            opts.PrepareForBulkLoad();
            opts.create_if_missing = true;
        }

#ifdef ZSTD
        opts.compression = rocksdb::CompressionType::kZSTD;
        opts.bottommost_compression = rocksdb::CompressionType::kZSTD;
#endif

        rocksdb::DB* db;
        TString path(dbPath);
        rocksdb::Status status;

        if (usageMode == EUsageMode::READ_ONLY) {
            NFs::EnsureExists(path);
            status = rocksdb::DB::OpenForReadOnly(opts, path, &db);
        } else {
            status = rocksdb::DB::Open(opts, path, &db);
        }

        Y_ENSURE(status.ok(), "cannot open RocksDB storage " << dbPath
                 << ": " << status.ToString());
        Db_.Reset(db);
    }

    void Compact(IThreadPool*) override {
        rocksdb::CompactRangeOptions opts;
        opts.bottommost_level_compaction = rocksdb::BottommostLevelCompaction::kForce;
        auto status = Db_->CompactRange(opts, nullptr, nullptr);
        Y_ENSURE(status.ok(), "cannot compact RocksDB: " << status.ToString());
    }

    bool Write(TMetricId id, TStringBuf value) override {
        rocksdb::Slice k = { reinterpret_cast<const char*>(&id), sizeof(id) };
        rocksdb::Slice v = { value.data(), value.size() };
        auto status = Db_->Put(rocksdb::WriteOptions(), k, v);
        Y_ENSURE(status.ok(), "cannot write (" << id << ", " << value << "): "
                 << status.ToString());
        return true;
    }

    TMaybe<TString> Read(TMetricId id) const override {
        rocksdb::Slice k = { reinterpret_cast<const char*>(&id), sizeof(id) };
        std::string v;
        auto status = Db_->Get(rocksdb::ReadOptions(), k, &v);
        if (status.IsNotFound()) {
            return Nothing();
        }
        Y_ENSURE(status.ok(), "cannot read (" << id << "): " << status.ToString());
        return TString(v);
    }

private:
    THolder<rocksdb::DB> Db_;
};

} // namespace


IStoragePtr CreateRocksDbStorage(TStringBuf dbPath, EUsageMode usageMode) {
    return MakeHolder<TRocksDbStorage>(dbPath, usageMode);
}

} // namespace NIndexer
} // namespace NSolomon

