#include "cmd_base.h"
#include "stockpile_consts.h"

#include <solomon/libs/cpp/kv/kv_client.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

namespace {
    void S3Options(NLastGetopt::TOpts *opts) {
        opts->AddLongOption('e', "s3-enpoint", "s3 endpoint to connect to")
                .RequiredArgument("HOST")
                .DefaultValue("s3.mdst.yandex.net")
                .Optional();
        opts->AddLongOption('k', "key-file", "file with json map, with 'AccessKeyId' and 'AccessSecretKey' keys")
                .RequiredArgument("KEYFILE")
                .Required();
        opts->AddLongOption('b', "s3-bucket", "s3 bucket to use")
                .RequiredArgument("BUCKET")
                .DefaultValue("solomon-backup")
                .Optional();
        opts->AddLongOption("balancer-bypass", "bypass s3 balancer")
                .NoArgument()
                .Optional();
        opts->AddLongOption('q', "quiet", "be quiet")
                .NoArgument()
                .Optional();
    }
}

int TCliCommandBase::operator()(const int argc, const char** argv) {
    TString host;
    ui16 port;

    NLastGetopt::TOpts opts;
    opts.AddLongOption('h', "host", "kikimr host to connect to")
            .RequiredArgument("HOST")
            .StoreResult(&host)
            .DefaultValue("localhost")
            .Optional();
    opts.AddLongOption('p', "port", "kikimr port to connect to")
            .RequiredArgument("PORT")
            .StoreResult(&port)
            .DefaultValue(2135)
            .Optional();
    opts.AddHelpOption();

    Options(&opts);

    NLastGetopt::TOptsParseResult optsRes(&opts, argc, argv);

    if (host != "localhost" && !host.Contains('.')) {
        host += ".search.yandex.net";
    }

    yandex::solomon::config::rpc::TGrpcClientConfig conf;
    conf.AddAddresses(host + ':' + ToString(port));
    conf.MutableMaxInboundMessageSize()->SetValue(64); // default value for grpc - DEFAULT_GRPC_MESSAGE_SIZE_LIMIT
    conf.MutableMaxInboundMessageSize()->SetUnit(yandex::solomon::config::DataUnit::MEGABYTES);
    conf.MutableMaxOutboundMessageSize()->SetValue(64); // default value for grpc - DEFAULT_GRPC_MESSAGE_SIZE_LIMIT
    conf.MutableMaxOutboundMessageSize()->SetUnit(yandex::solomon::config::DataUnit::MEGABYTES);

    Rpc_ = NSolomon::NKikimr::CreateNodeGrpc(conf, *NMonitoring::TMetricRegistry::Instance(), "kv-cli");
    KvClient_ = std::make_unique<NSolomon::TKikimrKvClient>(Rpc_.get());

    return Run(optsRes);
}

void TCliCommandTablet::Options(NLastGetopt::TOpts *opts) {
    opts->AddLongOption("path")
            .Help("path of solomon volume")
            .RequiredArgument("PATH")
            .Required();
    opts->AddLongOption("tablet")
            .Help("use specified KV-tablet identificator")
            .RequiredArgument("ID")
            .Optional();
    opts->AddLongOption("shard")
            .Help("use stockpile shard number to find KV-tablet")
            .RequiredArgument("ID")
            .Optional();
}

int TCliCommandTablet::Run(const NLastGetopt::TOptsParseResult& opts) {
    TString path = opts.Get("path");
    TVector<ui64> tabletIds = WaitAndCheck(KvClient_->ResolveTablets(path));

    Y_ENSURE(!tabletIds.empty(), "cannot resolve KV-tablets in path: " << path);

    ui64 tabletId;
    ui64 shardId;
    if (opts.Has("tablet")) {
        tabletId = FromString<ui64>(opts.Get("tablet"));
        auto it = Find(tabletIds, tabletId);
        Y_ENSURE(it != tabletIds.end(),
                 "invalid tablet id: " << tabletId);
        shardId = it - tabletIds.begin() + 1;
    } else if (opts.Has("shard")) {
        shardId = FromString<ui64>(opts.Get("shard"));
        Y_ENSURE(shardId >= 1 && shardId <= tabletIds.size(),
                 "invalid shard id: " << shardId);
        tabletId = tabletIds[shardId - 1];
    } else {
        ythrow yexception() << "either '--tablet' or '--shard' option "
                               "must be provided";
    }

    return RunOnTablet(tabletId, shardId, opts);
}

int TCliS3CommandBase::operator()(const int argc, const char** argv) {
    int r;
    NLastGetopt::TOpts opts;

    S3Options(&opts);
    opts.AddHelpOption();
    Options(&opts);

    NLastGetopt::TOptsParseResult optsRes(&opts, argc, argv);

    TString keyFile     = optsRes.Get("key-file");
    TString endpoint    = optsRes.Get("s3-enpoint");
    TString bucket      = optsRes.Get("s3-bucket");
    bool balancerBypass = optsRes.Has("balancer-bypass");
    bool quiet          = optsRes.Has("quiet");

    S3Client_ = CreateS3Client(keyFile, endpoint, bucket, balancerBypass, !quiet);

    r = Run(optsRes);
    S3Client_ = nullptr;

    return r;
}

void TCliS3CommandTablet::Options(NLastGetopt::TOpts *opts) {
    TCliCommandTablet::Options(opts);
    S3Options(opts);
}

int TCliS3CommandTablet::RunOnTablet(ui64 tabletId, ui64 shardId, const NLastGetopt::TOptsParseResult& opts) {
    int r;
    TString keyFile     = opts.Get("key-file");
    TString endpoint    = opts.Get("s3-enpoint");
    TString bucket      = opts.Get("s3-bucket");
    bool balancerBypass = opts.Has("balancer-bypass");
    bool quiet          = opts.Has("quiet");

    S3Client_ = CreateS3Client(keyFile, endpoint, bucket, balancerBypass, !quiet);

    r = RunS3OnTablet(tabletId, shardId, opts);
    S3Client_ = nullptr;

    return r;
}
