#include "cmd_base.h"
#include "stockpile_consts.h"

#include <util/generic/algorithm.h>
#include <util/generic/singleton.h>
#include <util/generic/utility.h>
#include <util/stream/format.h>

#include <cmath>

using NLastGetopt::TOptsParseResult;

namespace {

constexpr TStringBuf COLUMN_SHARD = "SHARD";
constexpr TStringBuf COLUMN_TABLET = "TABLET";
constexpr TStringBuf COLUMN_HOST = "HOST";
constexpr TStringBuf COLUMN_CHANGED_AT = "CHANGED_AT";

constexpr TStringBuf COLUMN_NAME = "NAME";
constexpr TStringBuf COLUMN_SIZE = "SIZE";
constexpr TStringBuf COLUMN_TIME = "TIME";

template <typename T>
size_t WidthOf(T v) {
    if (v == 0) {
        return 1;
    }
    size_t add = (v > 0) ? 1 : 2; // minus sign
    return add + static_cast<size_t>(std::log10(static_cast<double>(v)));
}

/**
 * List KV-tablets in cluster or files located on specific KV-tablet.
 */
class TCmdLs: public TCliCommandBase {
private:
    void Options(NLastGetopt::TOpts* opts) override {
        opts->AddLongOption("path")
                .Help("path of solomon volume")
                .RequiredArgument("PATH")
                .Required();
        opts->AddLongOption("tablet")
                .Help("use specified KV-tablet identificator")
                .RequiredArgument("ID")
                .Optional();
        opts->AddLongOption("shard")
                .Help("use stockpile shard number to find KV-tablet")
                .RequiredArgument("ID")
                .Optional();
        opts->AddLongOption("sort")
                .Help("sort files by specified field {NAME,SIZE,TIME}, add '-' "
                      "in front to sort in reverse order")
                .RequiredArgument("FIELD")
                .Optional();
        opts->AddLongOption("prefix")
                .Help("list tablet files with specified prefix")
                .RequiredArgument("PREFIX")
                .Optional();
    }

    int Run(const TOptsParseResult& opts) override {
        TString path = opts.Get("path");
        TVector<ui64> tabletIds = WaitAndCheck(KvClient_->ResolveTablets(path));

        Y_ENSURE(!tabletIds.empty(), "cannot resolve KV-tablets in path: " << path);
        if (opts.Has("tablet") || opts.Has("shard")) {
            ListFiles(tabletIds, opts);
        } else {
            ListTablets(tabletIds);
        }

        return 0;
    }

    void ListTablets(const TVector<ui64>& tabletIds) {
        TVector<NSolomon::TTabletInfo> tabletsInfo = WaitAndCheck(KvClient_->TabletsInfo(tabletIds));

        THashMap<ui64, const NSolomon::TTabletInfo*> tabletsInfoById;
        size_t hostWidth = 0;
        for (const auto& info: tabletsInfo) {
            tabletsInfoById.emplace(info.TabletId, &info);
            hostWidth = Max(hostWidth, info.Host.size(), COLUMN_HOST.size());
        }

        ui64 maxTabletId = 0;
        for (size_t i = 0; i < tabletIds.size(); i++) {
            maxTabletId = Max(maxTabletId, tabletIds[i]);
        }

        size_t shardWidth = Max(WidthOf(tabletIds.size()), COLUMN_SHARD.size()) + 2;
        size_t tabletWidth = Max(WidthOf(maxTabletId), COLUMN_TABLET.size()) + 2;
        hostWidth += 2;

        Cout << RightPad(COLUMN_SHARD, shardWidth)
             << RightPad(COLUMN_TABLET, tabletWidth)
             << RightPad(COLUMN_HOST, hostWidth)
             << COLUMN_CHANGED_AT
             << '\n';

        for (size_t i = 0; i < tabletIds.size(); i++) {
            ui64 tabletId = tabletIds[i];
            Cout << RightPad(i + 1, shardWidth)
                 << RightPad(tabletId, tabletWidth);

            auto it = tabletsInfoById.find(tabletId);
            if (it != tabletsInfoById.end()) {
                Cout << RightPad(it->second->Host, hostWidth);
                Cout << it->second->ChangedAt;
            } else {
                Cout << RightPad("<unknown>", hostWidth);
                Cout << "<unknown>";
            }

            Cout << '\n';
        }

        Cout << Flush;
    }

    void ListFiles(const TVector<ui64>& tabletIds, const TOptsParseResult& opts) {
        ui64 tabletId;
        if (opts.Has("tablet")) {
            tabletId = FromString<ui64>(opts.Get("tablet"));
            Y_ENSURE(Find(tabletIds, tabletId) != tabletIds.end(),
                     "invalid tablet id: " << tabletId);
        } else if (opts.Has("shard")) {
            ui32 shardId = FromString<ui32>(opts.Get("shard"));
            Y_ENSURE(shardId >= 1 && shardId <= tabletIds.size(),
                     "invalid shard id: " << shardId);
            tabletId = tabletIds[shardId - 1];
        } else {
            ythrow yexception() << "either '--tablet' or '--shard' option "
                                   "must be provided";
        }

        TVector<NSolomon::TFileInfo> files;
        auto* prefixParseResult = opts.FindLongOptParseResult("prefix");
        if (prefixParseResult) {
            if (prefixParseResult->Count() == 1) {
                auto prefix = prefixParseResult->Values().front();
                files = WaitAndCheck(KvClient_->ListPrefix(tabletId, prefix));
            } else {
                NSolomon::TKikimrKvBatchRequest batch(tabletId);
                for (const auto& prefix: prefixParseResult->Values()) {
                    batch.ListPrefix(prefix);
                }
                auto batchResp = WaitAndCheck(KvClient_->BatchRequest(batch));
                if (!batchResp.ListFileQueryResults.empty()) {
                    Y_ASSERT(batchResp.ListFileQueryResults.size() == prefixParseResult->Count());
                    for(auto& res: batchResp.ListFileQueryResults) {
                        TVector<NSolomon::TFileInfo> chunk{std::move(res)};
                        std::move(chunk.begin(), chunk.end(), std::back_inserter(files));
                    }
                }
            }
        } else {
            files = WaitAndCheck(KvClient_->ListFiles(tabletId));
        }

        TStringBuf sort = opts.GetOrElse("sort", "");
        bool reverseOrder = false;
        if (sort && sort[0] == '-') {
            sort = sort.Skip(1);
            reverseOrder = true;
        }

        if (sort == TStringBuf("NAME")) {
            SortBy(files, [](const auto& f) { return f.Name; });
        } else if (sort == TStringBuf("SIZE")) {
            SortBy(files, [](const auto& f) { return f.SizeBytes; });
        } else if (sort == TStringBuf("TIME")) {
            SortBy(files, [](const auto& f) { return f.CreatedAt; });
        }

        size_t nameWidth = COLUMN_NAME.size();
        ui32 maxSize = 0;
        for (const auto& file: files) {
            nameWidth = Max(nameWidth, file.Name.size());
            maxSize = Max(maxSize, file.SizeBytes);
        }
        nameWidth += 2;
        size_t sizeWidth = Max(WidthOf(maxSize), COLUMN_SIZE.size()) + 2;

        Cout << RightPad(COLUMN_NAME, nameWidth)
             << RightPad(COLUMN_SIZE, sizeWidth)
             << COLUMN_TIME
             << '\n';

        if (reverseOrder) {
            for (auto it = files.rbegin(), end = files.rend(); it != end; ++it) {
                PrintFile(*it, nameWidth, sizeWidth);
            }
        } else {
            for (const auto& file: files) {
                PrintFile(file, nameWidth, sizeWidth);
            }
        }

        Cout << Flush;
    }

    void PrintFile(const NSolomon::TFileInfo& file, size_t nameWidth, size_t sizeWidth) {
        Cout << RightPad(file.Name, nameWidth)
             << RightPad(file.SizeBytes, sizeWidth)
             << file.CreatedAt.ToStringUpToSeconds()
             << '\n';
    }
};

} // namespace

TMainClass* CmdLs() {
    return Singleton<TCmdLs>();
}
